/*================================================================
 * AWE MIDI player ncurses interface
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "config.h"
#ifndef NCURSES_HEADER
#include <ncurses/curses.h>
#else
#include NCURSES_HEADER
#endif

#ifdef NCURSES_VERSION
#define CASEKEY(x)	case x:
#else
/* BSD standard curses */
#include <sys/ioctl.h>
#define wattron(w,a) /*wstandout(w)*/
#define wattroff(w,a) /*wstandend(w)*/
#define nodelay(w,b) /**/
#define keypad(w,b) /**/
#define start_color() /**/
#define init_pair(n,f,b) /**/
#define CASEKEY(x)	/**/
#endif

#include "util.h"
#include "midievent.h"
#include "channel.h"
#include "controls.h"
#include "midievent.h"

#ifdef linux
#include <linux/awe_voice.h>
#elif defined(__FreeBSD__)
#include <awe_voice.h>
#endif

static void ctl_update(MidiInfo *mp);
static void ctl_help_mode(void);
static void ctl_total_time(int tt);
static void ctl_main_vol(int mv);
static void ctl_file_name(char *name);
static void ctl_title(char *name);
static void ctl_current_time(int ct);
static void ctl_note(int ch, int note, int vel);
static void ctl_program(int ch, int val);
static void ctl_bank(int ch, int val);
static void ctl_control_change(int ch, int type, int val);
static void ctl_wheel(int channel, int val);
static void ctl_pitch_sense(int channel, int val);
static void ctl_system_change(int type, int mode);
static void ctl_effects_change(int ch, int type, int value);
static void ctl_reset(MidiInfo *mp);
static int ctl_open(int using_stdin, int using_stdout);
static void ctl_close(void);
static int ctl_read(MidiInfo *mp, int *valp);
static int ctl_blocking_read(MidiInfo *mp, int *valp);
static int cmsg(int type, int verbosity_level, char *fmt, ...);
void dumb_pass_playing_list(int number_of_files, char *list_of_files[]);

char *inst_name(int val);
char *drum_name(int val);

/* export the interface functions */

#define ctl ncurses_control_mode

ControlMode ctl= 
{
  "ncurses interface", 'n',
  TRUE, /* need_args */
  TRUE, /* need_sync */
  TRUE, /* fancy_prev */
  0,0,0,0, /* verbose, trace, open, repeat */
  ctl_open, dumb_pass_playing_list, ctl_close, ctl_read, ctl_blocking_read, cmsg,
  ctl_update, ctl_reset, ctl_file_name, ctl_title, ctl_total_time, ctl_current_time, 
  ctl_note, ctl_program, ctl_bank, ctl_control_change,
  ctl_wheel, ctl_pitch_sense,
  ctl_system_change, ctl_effects_change, ctl_main_vol,
};


/*----------------------------------------------------------------*/

static int msglen, msgpos;
static WINDOW *dftwin = NULL, *msgwin = NULL, *trcwin = NULL, *helpwin = NULL;
static int bar_len;
static int multi_part = 1;
static int cur_chorus, cur_reverb;

#define FLAG_NOTE_OFF	1
#define FLAG_NOTE_ON	2

#define FLAG_BANK	1
#define FLAG_PROG	2
#define FLAG_PAN	4
#define FLAG_SUST	8
#define FLAG_WHEEL	16

static unsigned char cvel[MAX_MIDI_CHANNELS];
static unsigned char cnote[MAX_MIDI_CHANNELS];
static int ctotal[MAX_MIDI_CHANNELS];
static int clen[MAX_MIDI_CHANNELS];
static char v_flags[MAX_MIDI_CHANNELS];

static int cbank[MAX_MIDI_CHANNELS];
static int cprog[MAX_MIDI_CHANNELS];
static int cpan[MAX_MIDI_CHANNELS];
static int cbend[MAX_MIDI_CHANNELS];
static int csust[MAX_MIDI_CHANNELS];
static int c_flags[MAX_MIDI_CHANNELS];


/*----------------------------------------------------------------*/

static void ctl_help_mode(void)
{
	if (msglen <= 0) return;
	if (helpwin) {
		delwin(helpwin);
		touchwin(msgwin);
		wmove(dftwin, 0, COLS-1);
		wrefresh(msgwin);
		helpwin = NULL;
	} else {
		helpwin = newwin(3, COLS, 2, 0);
		if (helpwin == NULL)
			return;
		werase(helpwin); 
		waddstr(helpwin, 
			"V/Up=Louder    b/Left=Skip back      "
			"n/Next=Next      r/Home=Restart");
		wmove(helpwin, 1, 0);
		waddstr(helpwin, 
			"v/Down=Softer  f/Right=Skip forward  "
			"p/Prev=Previous  q/End=Quit");
		wmove(helpwin, 2, 0);
		waddstr(helpwin,
			"c/C=chorus mode  e/E=reverb mode     Space=Stop");
		wmove(dftwin, 0, COLS-1);
		wrefresh(helpwin);
	}
}


static void ctl_total_time(int secs)
{
	secs /= 100;
	wmove(dftwin, 1, 40 + 8);
	wattron(dftwin, A_BOLD);
	wprintw(dftwin, "%02d:%02d", secs / 60, secs % 60);
	wattroff(dftwin, A_BOLD);
	wmove(dftwin, 0, COLS-1);
	wrefresh(dftwin);
}

static void ctl_main_vol(int mv)
{
	wmove(dftwin, 1, 56 + 8);
	wattron(dftwin, A_BOLD);
	wprintw(dftwin, "%03d %%", mv);
	wattroff(dftwin, A_BOLD);
	wmove(dftwin, 0, COLS-1);
	wrefresh(dftwin);
}

static void ctl_file_name(char *name)
{
	char tmp[20+1], *p;
	sprintf(tmp, "%20.20s", name);
	if ((p = strchr(tmp, '\n')) != NULL) *p = 0;
	wmove(dftwin, 0, COLS-21);
	wattron(dftwin, A_BOLD);
	waddstr(dftwin, tmp);
	wattroff(dftwin, A_BOLD);
	wmove(dftwin, 0, COLS-1);
	wrefresh(dftwin);
	ctl_title(name);
}

static void ctl_title(char *name)
{
	char tmp[32+1], *p;
	sprintf(tmp, "%-30.30s", name);
	if ((p = strchr(tmp, '\n')) != NULL) *p = 0;
	wmove(dftwin, 1, 9);
	wattron(dftwin, A_BOLD);
	waddstr(dftwin, tmp);
	wattroff(dftwin, A_BOLD);
	wmove(dftwin, 0, COLS-1);
	wrefresh(dftwin);
}

static void ctl_system_change(int type, int mode)
{
	switch (type) {
	case SY_CHORUS_MODE:
		cur_chorus = mode;
		wmove(dftwin, 1, 73);
		wattron(dftwin, A_BOLD);
		wprintw(dftwin, "C%d/", mode);
		wattroff(dftwin, A_BOLD);
		wmove(dftwin, 0, COLS-1);
		wrefresh(dftwin);
		break;
	case SY_REVERB_MODE:
		cur_reverb = mode;
		wmove(dftwin, 1, 76);
		wattron(dftwin, A_BOLD);
		wprintw(dftwin, "R%d", mode);
		wattroff(dftwin, A_BOLD);
		wmove(dftwin, 0, COLS-1);
		wrefresh(dftwin);
		break;
	}
}

/*----------------------------------------------------------------*/

static void ctl_current_time(int secs)
{
	static int lastsec = 0;
	secs /= 100;
	if (secs != lastsec) {
		wmove(dftwin, 4,6);
		wmove(dftwin, 1, 40);
		wattron(dftwin, A_BOLD);
		wprintw(dftwin, "%02d:%02d", secs / 60, secs % 60);
		wattroff(dftwin, A_BOLD);
		wmove(dftwin, 0, COLS-1);
		wrefresh(dftwin);
	}
}

/*----------------------------------------------------------------*/

#define DELTA_VEL	32

static void update_note(int ch)
{
	int i, xl, len, color, pat;
	xl = (ch >= 16) ? COLS/2 : 0;
	len = bar_len * ctotal[ch] / 127;
	if (clen[ch] < len) {
		wmove(trcwin, ch % 16 + 1, xl + 3+8+4 + clen[ch]);
		if (cbank[ch] == 128)
			color = 1;
		else
			color = 2 + ch%2;
		wattron(trcwin, A_REVERSE|COLOR_PAIR(color));
		pat = '#';
#ifdef NCURSES_VERSION
		if (cbank[ch] == 128)
			pat = ACS_CKBOARD;
		else
			pat = ACS_S9;
#endif
		for (i = clen[ch]; i < len; i++)
			waddch(trcwin, pat);
		wattroff(trcwin, A_REVERSE|A_COLOR);
	} else if (clen[ch] > len) {
		wmove(trcwin, ch % 16 + 1, xl + 3+8+4 + len);
		for (i = len; i < clen[ch]; i++)
			waddch(trcwin, ' ');
	}
	clen[ch] = len;
}

static void update_program(int ch)
{
	int xl;
	xl = (ch >= 16) ? COLS/2 : 0;
	wmove(trcwin, ch % 16 + 1, xl + 3);
	if (cbank[ch] == 128) {
		wattron(trcwin, A_BOLD);
		wprintw(trcwin, "%-7s", drum_name(cprog[ch]));
		wattroff(trcwin, A_BOLD);
	}
	else
		wprintw(trcwin, "%-7s", inst_name(cprog[ch]));
	wmove(dftwin, 0, COLS-1);
}

static void update_pan(int ch)
{
	int xl;
	xl = (ch >= 16) ? COLS/2 : 0;
	wmove(trcwin, ch % 16 + 1, xl + 3 + 8);
	waddch(trcwin, cpan[ch]);
}

static void update_sustain(int ch)
{
	int xl;
	xl = (ch >= 16) ? COLS/2 : 0;
	wmove(trcwin, ch % 16 + 1, xl + 3 + 8 + 1);
	waddch(trcwin, csust[ch]);
}


static void update_wheel(int ch)
{
	int xl;
	xl = (ch >= 16) ? COLS/2 : 0;
	wmove(trcwin, ch % 16 + 1, xl + 3 + 8 + 2);
	waddch(trcwin, cbend[ch]);
}

static void ctl_update(MidiInfo *mp)
{
	int i, do_refresh = 0;
	if (!(ctl.playing_mode & PLAY_TRACE))
		return;
	for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
		if (v_flags[i]) {
			if (v_flags[i] == FLAG_NOTE_OFF) {
				ctotal[i] -= DELTA_VEL;
				if (ctotal[i] <= 0) {
					ctotal[i] = 0;
					v_flags[i] = 0;
				}
			} else
				v_flags[i] = 0;
			update_note(i);
			do_refresh = 1;
		}
		if (c_flags[i]) {
			if (c_flags[i] & FLAG_PAN)
				update_pan(i);
			if (c_flags[i] & FLAG_PROG)
				update_program(i);
			if (c_flags[i] & FLAG_SUST)
				update_sustain(i);
			if (c_flags[i] & FLAG_WHEEL)
				update_wheel(i);
			c_flags[i] = 0;
			do_refresh = 1;
		}
	}
	if (do_refresh) {
		wmove(dftwin, 0, COLS-1);
		wrefresh(trcwin);
		wrefresh(dftwin);
	}
}

/*----------------------------------------------------------------*/

static void ctl_note(int ch, int note, int vel)
{
	if (!(ctl.playing_mode & PLAY_TRACE))
		return;

	if (vel == 0) {
		if (note == cnote[ch]) {
			v_flags[ch] = FLAG_NOTE_OFF;
			cvel[ch] = 0;
		}
	} else if (vel >= cvel[ch]) {
		cvel[ch] = vel;
		cnote[ch] = note;
		ctotal[ch] = vel * channels[ch].controls[CTL_MAIN_VOLUME] *
			channels[ch].controls[CTL_EXPRESSION] / (127*127);
		v_flags[ch] = FLAG_NOTE_ON;
	}
}

static void ctl_control_change(int ch, int type, int val)
{
	int pat;
	if (!(ctl.playing_mode & PLAY_TRACE)) 
		return;
	switch (type) {
	case CTL_MAIN_VOLUME:
	case CTL_EXPRESSION:
		ctl_note(ch, cnote[ch], cvel[ch]);
		break;
	case CTL_PAN:
		if (val < 0)
			pat = ' ';
		else if (val < 5)
			pat = 'L';
		else if (val <= 60)
			pat = 'l';
		else if (val <= 67)
			pat = ' ';
		else if (val <= 122)
			pat = 'r';
		else
			pat = 'R';
		if (pat == cpan[ch]) return;

		cpan[ch] = pat;
		c_flags[ch] |= FLAG_PAN;
		break;

	case CTL_SUSTAIN:
		if (val == 127)
			pat = 'S';
		else
			pat = ' ';
		if (pat == csust[ch])
			return;
		csust[ch] = pat;
		c_flags[ch] |= FLAG_SUST;
		break;
	}
}

static void ctl_program(int ch, int val)
{
	if (!(ctl.playing_mode & PLAY_TRACE))
		return;
	if (cprog[ch] == val)
		return;
	cprog[ch] = val;
	c_flags[ch] |= FLAG_PROG;
}


static void ctl_wheel(int ch, int val)
{
	int pat;
	if (!(ctl.playing_mode & PLAY_TRACE)) 
		return;
	if (val > 0x2000) pat = '+';
	else if (val < 0x2000) pat = '-';
	else pat = ' ';

	if (pat == cbend[ch])
		return;

	cbend[ch] = pat;
	c_flags[ch] |= FLAG_WHEEL;
}

static void ctl_bank(int ch, int val)
{
	if (cbank[ch] != val) {
		cbank[ch] = val;
		c_flags[ch] |= FLAG_PROG;
	}
}

static void ctl_pitch_sense(int ch, int val) {}
static void ctl_effects_change(int ch, int type, int value) {}

static void ctl_reset(MidiInfo *mp)
{
	int i, imax;
	if (!(ctl.playing_mode & PLAY_TRACE)) 
		return;

	/*
	if (mp->multi_part != multi_part) {
		multi_part = mp->multi_part;
	}
	*/
	wmove(trcwin, 0, 0);
	for (i = 0; i < COLS-1; i++)
		waddch(trcwin, ' ');
	wmove(trcwin, 0, 0);
	waddstr(trcwin, "Ch Program psb");
	imax = 16;
	if (multi_part) {
		wmove(trcwin, 0, COLS/2);
		waddstr(trcwin, "Ch Program psb");
		bar_len = COLS/2 - (3+8+5);
		imax = 32;
	} else
		bar_len = COLS - (3+8+5);
			
	for (i = 0; i < imax; i++) {
		wmove(trcwin, i % 16 + 1, (i / 16) * (COLS/2));
		wprintw(trcwin, "%02d", i);
		cvel[i] = 0;
		ctotal[i] = 0;
		clen[i] = bar_len + 1;
		update_note(i);
		v_flags[i] = 0;
		ctl_program(i, channels[i].preset);
		ctl_bank(i, channels[i].bank);
		ctl_control_change(i, CTL_PAN, channels[i].controls[CTL_PAN]);
		ctl_control_change(i, CTL_SUSTAIN, channels[i].controls[CTL_SUSTAIN]);
		ctl_wheel(i, channels[i].pitchbend);
	}
	wmove(dftwin, 0, COLS-1);
	wrefresh(trcwin);
}

/***********************************************************************/

static int ctl_open(int using_stdin, int using_stdout)
{
	int i;
#ifdef NCURSES_VERSION
	if (using_stdin) {
		SCREEN *dftscr, *oldscr;
		fflush(stdout);
		setvbuf(stdout, 0, _IOFBF, BUFSIZ);
		dftscr = newterm(0, stdout, stdout);
		if (! dftscr)
			return -1;
		oldscr = set_term(dftscr);
	} else
#endif
		initscr();

	start_color();
	raw();
	cbreak();
	noecho();
	nonl();
	ctl.opened = 1;

	dftwin = newwin(2, COLS, 0, 0);
	if (dftwin == NULL || LINES < 4) {
		fprintf(stderr, "Too small window size!\n");
		exit(1);
	}
	werase(dftwin);
	idlok(dftwin, TRUE);
	nodelay(dftwin, TRUE);
	scrollok(dftwin, FALSE);
	keypad(dftwin, TRUE);

	init_pair(1, COLOR_RED, COLOR_BLACK);
	init_pair(2, COLOR_GREEN, COLOR_BLACK);
	init_pair(3, COLOR_BLUE, COLOR_BLACK);

	wmove(dftwin, 0, 0);
	wprintw(dftwin, "AWE MIDI Player  ver.%s  [h] for help", VERSION_STR);
	wmove(dftwin, 1, 0);
	waddstr(dftwin, "Playing:");
	wmove(dftwin, 1, 40);
	waddstr(dftwin, "00:00 / ");
	wmove(dftwin, 1, 48);
	wattron(dftwin, A_BOLD);
	waddstr(dftwin, "00:00");
	wattroff(dftwin, A_BOLD);
	wmove(dftwin, 1, 56);
	waddstr(dftwin, "Volume:");
	wrefresh(dftwin);

	msgpos = 0;
	msglen = 0;
	if ((ctl.playing_mode & PLAY_TRACE) && LINES >= 17 + 3) {
		msglen = LINES - 17 - 2;
		msgwin = newwin(msglen, COLS, 2, 0);
		trcwin = newwin(17, COLS, 2 + msglen, 0);
		scrollok(msgwin, 1);
		scrollok(trcwin, 0);
		werase(msgwin);
		wrefresh(msgwin);
		wrefresh(trcwin);
		for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
			cprog[i] = -1;
			cbank[i] = -1;
		}
	} else {
		ctl.playing_mode &= ~PLAY_TRACE;
		msglen = LINES - 2;
		msgwin = newwin(msglen, COLS, 2, 0);
		werase(msgwin);
		scrollok(msgwin, 1);
	}

	return 0;
}

static void ctl_close(void)
{
	if (ctl.opened) {
		endwin();
		ctl.opened=0;
	}
}

static int ctl_blocking_read(MidiInfo *mp, int *valp)
{
	int rc;
	nodelay(dftwin, FALSE);
	rc = ctl_read(mp, valp);
	nodelay(dftwin, TRUE);
	return rc;
}

static int ctl_read(MidiInfo *mp, int *valp)
{
#ifndef NCURSES_VERSION
	int num;
	ioctl(fileno(stdin),FIONREAD,&num);
	if (num == 0) return RC_NONE;
#endif

	switch (wgetch(dftwin)) {
	case 'h':
	case '?':
	CASEKEY(KEY_F(1))
		ctl_help_mode();
		return RC_NONE;
	  
	case 'V':
	CASEKEY(KEY_UP)
		*valp = mp->volume_base + 10;
		if (*valp <= 200)
			return RC_CHANGE_VOLUME;
		else
			return RC_NONE;
	case 'v':
	CASEKEY(KEY_DOWN)
		*valp = mp->volume_base - 10;
		if (*valp >= 0)
			return RC_CHANGE_VOLUME;
		else
			return RC_NONE;
	case 'q':
	CASEKEY(KEY_END)
		return RC_QUIT;
	case 'n':
	CASEKEY(KEY_NPAGE)
		return RC_NEXT;
	case 'p':
	CASEKEY(KEY_PPAGE)
		return RC_REALLY_PREVIOUS;
	case 'r':
	CASEKEY(KEY_HOME)
		return RC_RESTART;

	case 'f':
	CASEKEY(KEY_RIGHT)
		*valp = 200;
		return RC_SEARCH;
	case 'b':
	CASEKEY(KEY_LEFT)
		*valp = -200;
		return RC_SEARCH;
	case 'c':
		*valp = cur_chorus + 1;
		if (*valp >= AWE_CHORUS_NUMBERS) *valp = 0;
		ctl_system_change(SY_CHORUS_MODE, *valp);
		return RC_CHANGE_CHORUS;
	case 'C':
		*valp = cur_chorus - 1;
		if (*valp < 0) *valp = AWE_CHORUS_NUMBERS-1;
		ctl_system_change(SY_CHORUS_MODE, *valp);
		return RC_CHANGE_CHORUS;
	case 'e':
		*valp = cur_reverb + 1;
		if (*valp >= AWE_REVERB_NUMBERS) *valp = 0;
		ctl_system_change(SY_REVERB_MODE, *valp);
		return RC_CHANGE_REVERB;
	case 'E':
		*valp = cur_reverb - 1;
		if (*valp < 0) *valp = AWE_REVERB_NUMBERS-1;
		ctl_system_change(SY_REVERB_MODE, *valp);
		return RC_CHANGE_REVERB;
	case ' ':
		return *valp ? RC_CONTINUE : RC_PAUSE;
	case 'u':
		*valp = 1; return RC_BASE_CHANGE;
	case 'd':
		*valp = -1; return RC_BASE_CHANGE;
	}
	return RC_NONE;
}

static int cmsg(int type, int verbosity_level, char *fmt, ...)
{
	va_list ap;
	if ((type==CMSG_TEXT || type==CMSG_INFO || type==CMSG_WARNING) &&
	    ctl.verbosity<verbosity_level)
		return 0;
	va_start(ap, fmt);
	if (!ctl.opened || msgwin == NULL) {
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
	} else {
		if (helpwin)
			ctl_help_mode();
		if (type == CMSG_TEXT)
			waddstr(msgwin, fmt);
		else
			vwprintw(msgwin, fmt, ap);
		wprintw(msgwin, "\n");
		wrefresh(msgwin);
		wmove(dftwin, 0, 0);
		if (type == CMSG_ERROR)
			sleep(2);
	}

	va_end(ap);
	return 0;
}
