/*
   AVR LIBC
   strlcat()

   Copyright (c) 2003, 2007 Eric B. Weddington, R. Patommel
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
   * Neither the name of the copyright holders nor the names of
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

*/

/* $Id: strlcat.S,v 1.5 2007/01/23 15:32:48 joerg_wunsch Exp $ */

/** \file */

/** \ingroup avr_string
    \fn size_t strlcat (char *dst, const char *src, size_t siz)
    \brief Concatenate two strings.

    Appends src to string dst of size siz (unlike strncat(), siz is the
    full size of dst, not space left).  At most siz-1 characters
    will be copied.  Always NULL terminates (unless siz <= strlen(dst)).

    \returns The strlcat() function returns strlen(src) + MIN(siz,
    strlen(initial dst)).  If retval >= siz, truncation occurred.  */

#if !defined(__DOXYGEN__)

#include "gasava.inc"
#include "macros.inc"

#define dst_hi		r25
#define dst_lo		r24
#define src_hi		r23
#define src_lo		r22
#define siz_hi		r21
#define siz_lo		r20
#define dlen_hi		r19
#define dlen_lo		r18
#define rWord_hi	r25
#define rWord_lo	r24

	.text
	.global _U(strlcat)
	.type	_U(strlcat),@function

_U(strlcat):
	X_movw	XL, dst_lo		; X = dst
	X_movw	ZL, src_lo		; Z = src

.L_strlcat_dlen:			; Find end of dst string
	ld	__tmp_reg__, X+		; get next char from dst
	cp	siz_lo, __zero_reg__	;  and calc dlen = len of dst
	cpc	siz_hi, __zero_reg__	; size == 0 ?
	breq	 1f			; --> done
	tst	__tmp_reg__		; end of dst ?
	breq	 1f			; --> done
	subi	siz_lo, lo8(-(-1))
	sbci	siz_hi, hi8(-(-1))	; siz--
	rjmp	.L_strlcat_dlen		; --> next char
1:	sbiw	XL, 1			; undo post increment
	X_movw	dlen_lo, XL
	sub	dlen_lo, dst_lo
	sbc	dlen_hi, dst_hi		; dlen = X - dst
	cp	siz_lo, __zero_reg__
	cpc	siz_hi, __zero_reg__	; size == 0 ?
	breq	.L_strlcat_slen		; --> done
	subi	siz_lo, lo8(-(-1))
	sbci	siz_hi, hi8(-(-1))	; siz--

.L_strlcat_concat:			; Concatenate
	ld	__tmp_reg__, Z+		; get next char from src
	cp	siz_lo, __zero_reg__
	cpc	siz_hi, __zero_reg__	; size == 0 ?
	breq	1f			; --> done
	tst	__tmp_reg__		; end of src ?
	breq	1f			; --> done
	st	X+, __tmp_reg__		; store in dest
	subi	siz_lo, lo8(-(-1))
	sbci	siz_hi, hi8(-(-1))	; siz--
	rjmp	.L_strlcat_concat		; --> next char
1:	st	X, __zero_reg__		; *X = '\0'
	sbiw	ZL, 1			; undo post increment

.L_strlcat_slen:
	ld	__tmp_reg__, Z+		; get next char from src
	tst	__tmp_reg__		; end of src ?
	brne	.L_strlcat_slen		; --> next char
	sbiw	ZL, 1			; undo post increment
	X_movw	rWord_lo, dlen_lo
	add	rWord_lo, ZL
	adc	rWord_hi, ZH
	sub	rWord_lo, src_lo
	sbc	rWord_hi, src_hi	; return(dlen + (Z - src))
	ret

.L_strlcat_end:
.size	_U(strlcat), .L_strlcat_end - _U(strlcat)

#endif /* not __DOXYGEN__ */
