/*
 * rbtree.h - red-black tree library
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 *
 * Properties satisfied by a red-black tree:
 * - Root Property: the root is black
 * - External Property: every leaf is black
 * - Internal Property: the children of a red node are black
 * - Depth Property: all the leaves have the same black depth
 */

#ifndef RBTREE_H
#define RBTREE_H

#include <config.h>

/* Node colors */
typedef enum {RB_BLACK, RB_RED} rbcolor_t;

/* Red black tree node structure */
typedef struct rbnode rbnode_t;
struct rbnode {
	rbnode_t*	parent;	/* parent */
	rbnode_t*	left;	/* left child node */
	rbnode_t*	right;	/* right child node */
	const void*	key; 	/* node attributes */
	rbcolor_t	color; 	/* color black or red */
};

/* Red black tree structure */
typedef struct rbtree rbtree_t;
struct rbtree {
	rbnode_t*	root; 	/* root of the tree */
	size_t		size;	/* number of nodes */
	/* define order */
	int 			(*cmp) (const void*, const void*);
};

/* Creates an empty red-black tree.
 * @param cmpf: a function that can compare two nodes.
 * @return: on success, a pointer to the created empty red-black tree,
 *	    on failure (malloc failed), NULL
 */
rbtree_t* rbt_create(int (*cmpf)(const void*, const void*));

/* Get the color of a node.
 * @param node: node in the tree.
 * @return: RB_RED if the node is red-colored,
 *	    else: RB_BLACK.
 */
rbcolor_t rbt_color(rbnode_t* node);

/* Retrieve the first node in the total order.
 * @param tree: the tree to search in.
 * @return: if tree is empty: NULL,
 *	    else: pointer to first node.
 */
rbnode_t* rbt_first(rbtree_t* tree);

/* Retrieve the last node in the total order.
 * @param tree: the tree to search in.
 * @return: if tree is empty: NULL,
 *	    else: pointer to last node.
 */
rbnode_t* rbt_last(rbtree_t* tree);

/* Retrieve the predecessor of a node.
 * @param node: node in the tree.
 * @return: if predecessor exists: pointer to predecessor,
 *	    else: NULL.
 */
rbnode_t* rbt_predecessor(rbnode_t* node);

/* Retrieve the successor of a node.
 * @param node: node in the tree.
 * @return: if successor exists: pointer to successor,
 *	    else: NULL.
 */
rbnode_t* rbt_successor(rbnode_t* node);

/* Insert a node into a red-black tree.
 * If new node cannot be created, error and exit.
 * @param tree: the red-black tree to insert into.
 * @param key: the key to be inserted.
 * @return: if key exists: 0,
 *	    if malloc failed: -1
 * 	    on success: 1.
 */
int rbt_insert(rbtree_t* tree, const void* key);

/* Remove a node from a red-black tree.
 * @param tree: the red-black tree to remove from.
 * @param key: the key to be removed from the tree.
 * @return: if key exists: pointer to the removed node,
 *	    else: NULL.
 */
rbnode_t* rbt_remove(rbtree_t* tree, const void* key);

/* Search for a node in a red-black tree.
 * @param tree: the red-black tree to search in.
 * @param key: the key that identifies the node to be searched.
 * @return: if node exists: pointer to the node,
 *	    else: NULL.
 */
rbnode_t* rbt_search(rbtree_t* tree, const void* key);

/* Get the sibling of a node.
 * @param tree: the red-black tree.
 * @param node: a node in the red-black tree, usually not a leaf.
 * @return: a pointer to the sibling if the sibling exists,
 *	    NULL otherwise.
 */
rbnode_t* rbt_sibling(rbtree_t* tree, rbnode_t* node);

/* Remove a node and all his child nodes from the memory.
 * @param node: node to be freed.
 * @param func: function to be called (usually free_object oid).
 */
void rbt_freenode(rbnode_t* node, void (*func)(const void*));

/* Remove a rbtree from the memory.
 * @param tree: tree to be freed.
 * @param func: function to be called (usually free_object oid).
 */
void rbt_free(rbtree_t* tree, void (*func)(const void*));

#endif /* RBTREE_H */
