/* atari-fdisk.c -- Atari partition table manipulator for Linux (and GEMDOS).
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * Some parts of this Atari version of fdisk have been taken from the sources
 * for the original PC Linux fdisk by A. V. Le Blanc (LeBlanc@mcc.ac.uk) to
 * get a somewhat similar user interface.
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: atari-fdisk.c,v 1.16 1998/09/19 01:29:27 rnhodek Exp $
 *
 * $Log: atari-fdisk.c,v $
 * Revision 1.16  1998/09/19 01:29:27  rnhodek
 * New clear_partitions() function.
 *
 * Revision 1.15  1998/02/07 21:19:58  rnhodek
 * Description for UNX type was too long...
 * If user presses just return in the menu, just reprint the prompt instead of
 * listing the menu.
 *
 * Revision 1.14  1997/09/30 07:58:30  rnhodek
 * w command should also exit
 *
 * Revision 1.13  1997/08/22 15:36:35  rnhodek
 * Implemented basic support for ICD format. Should work, but conversion
 * AHDI<->ICD isn't very clever yet.
 *
 * Revision 1.12  1997/07/10 20:48:41  rnhodek
 * Removed "not very well tested" in disclaimer...
 *
 * Revision 1.11  1997/07/10 20:36:34  rnhodek
 * New menu function move_partition_ars().
 * Fixed some minor bugs.
 *
 * Revision 1.10  1997/07/08 18:59:00  rnhodek
 * Ouch, too many changes without commits in between...
 * Implemented moving of partitions
 * Implemented checking of "extended start condition" (great changes in
 *   new_partition())
 * Some more tests in verify()
 *
 * Revision 1.9  1997/06/22 10:30:53  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.8  1997/06/21 20:47:40  rnhodek
 * Added RCS keywords
 *
 * Revision 1.7  1997/06/13 12:42:38  rnhodek
 * Be more verbose on expert mode switches
 * 
 * Revision 1.6  1997/06/13 10:18:45  rnhodek
 * Implemented settings bootflags on XGM part.
 * allow write command if opened read-only and ALPHA_VERSION; actual writing
 *   skipped in swrite
 * nicer output of list_table
 * 
 * Revision 1.5  1997/06/12 14:11:19  rnhodek
 * Fix minor things again; correct wrong calculations in new_partition
 * 
 * Revision 1.4  1997/06/12 13:43:38  rnhodek
 * Fix many small bugs here and there. The ones you find when first running a
 * program...
 * 
 * Revision 1.3  1997/06/11 19:49:14d  rnhodek
 * Implemented bad sector list handling
 * 
 * Revision 1.2  1997/06/11 14:41:58  rnhodek
 * Added 'b' and 'f' menu items
 * 
 * Revision 1.1  1997/06/11 14:36:35  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:35  rnhodek
 * Started using CVS for atafdisk
 * 
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: atari-fdisk.c,v 1.16 1998/09/19 01:29:27 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <errno.h>

#include "fdisk.h"
#include "version.h"
#include "readpart.h"
#include "menufuncs.h"
#include "savetab.h"
#include "input.h"
#include "util.h"


/* normally O_RDWR, -l option gives O_RDONLY */
int type_open = O_RDWR;

char *disk_device = DEFAULT_DEVICE;	/* sda, unless specified */
char *line_ptr;				/* interactive input */
char line_buffer[LINE_LENGTH];

int fd;					/* the disk */
int ext_index;				/* the prime extended partition */
int listing = 0;			/* no aborts for fdisk -l */
int size_flag = 0;
int force = 0;				/* force working on disk where
					 * partitions are mounted */
int expert_mode = 0;			/* allow certain operations */
int partitions;				/* number of existing partitions */

char *save_sector_file = NULL;
char *restore_sector_file = NULL;

unsigned long hd_size;			/* total size of disk */
unsigned long rs_hd_size;		/* size of disk noted in root sector */
unsigned long bsl_start;		/* start of bad sector list */
unsigned long bsl_size;			/* size of bad sector list */
unsigned long saved_bsl_start;		/* saved versions of above, to */
unsigned long saved_bsl_size;		/* detect changes */
int bsl_HDX_compat;			/* make HDX compatible bad sec. list */
int XGM_flag;				/* boot flags for XGM part. */

PARTITION part_table[MAXIMUM_PARTS];

struct bootflags bootflags[] = {
    { 0x80, "GEMDOS", "TOS" },
    { 0x40, "Atari System V", "ASV" },
    { 0x20, "NetBSD (?)", "BSD" },
    { 0x10, "Linux", "LNX" },
    { 0x08, "(not assigned)", "$08" }
};
int n_bootflags = fieldsize(bootflags);

struct partition_ID partition_IDs[] = {
    { "BGM", "GEMDOS (>32 MB)" },
    { "GEM", "GEMDOS (<32 MB)" },
    { "LNX", "Linux native" },
    { "MAC", "Mac HFS" },
    { "MIX", "Minix" },
    { "MNX", "Minix" },
    { "RAW", "unspecified" },
    { "SWP", "Linux swap" },	/* Debian installation wants to see that
				 * string, but that doesn't mean that other
				 * OSes may not use that swap partition... */
    { "UNX", "Atari SysV Unix" },
};
int n_partition_IDs = fieldsize(partition_IDs);

#if 0
/* global partition format options */
int opt_unordered = 0;
int opt_conversions = 1;
int opt_overlapping = 0;
int opt_separate_aux = 0;
int opt_multi_xgm = 0;

struct option global_options[] = {
    { "unordered", &opt_unordered,
      "allow partition numbering to be different from physical locations" },
    { "conversions", &opt_conversions,
      "make primary <-> extended conversions if possible" },
    { "overlap", &opt_overlapping,
      "allow XGM containers to overlap primary partitions" },
    { "separate", &opt_separate_aux,
      "allow to separate auxiliary root sector from its partition" },
    { "multi-xgm", &opt_multi_xgm,
      "allow more than one extended partition containers (XGMs)" },
};
int n_global_options = fieldsize(global_options);
#endif

jmp_buf listingbuf;



/***************************** Prototypes *****************************/

static void try( char *device );
static void menu( void );

/************************* End of Prototypes **************************/



static void try( char *device )
{
    disk_device = device;
    if (!setjmp(listingbuf)) {
	if ((fd = open( disk_device, type_open )) >= 0) {
	    close(fd);
	    get_boot();
	    list_table();
	} else {
	    /* Ignore other errors, since we try IDE
	       and SCSI hard disks which may not be
	       installed on the system. */
	    if (errno == EACCES) {
		fprintf(stderr, "Cannot open %s\n", device);
		exit(1);
	    }
	}
    }
}

int main( int argc, char *argv[] )

{
    char **p;
#if 0
    char *opts = "";
#endif

    while( argc > 1 && argv[1][0] == '-' ) {
	switch( argv[1][1] ) {

	  case 'f':
	    /* force operation on disks from which some partitions seems
	     * mounted */
	    force = 1;
	    ++argv; --argc;
	    break;
	    
	  case 'x':
	    /* enter expert mode */
	    expert_mode = 1;
	    ++argv; --argc;
	    break;

	  case 'r':
	    /* open disk device read-only */
	    type_open = O_RDONLY;
	    ++argv; --argc;
	    break;
	    
	  case 'v':
	    /* print version */
	    printf("This is Atari fdisk version " VERSION "\n");
	    exit(0);

	  case 'O':
	    /* save (output) partitioning information */
	    p = &save_sector_file;
	    goto sector_file_opt;
	  case 'I':
	    /* restore (input) partitioning information */
	    p = &restore_sector_file;
	  sector_file_opt:
	    if (argv[1][2]) {
		*p = argv[1]+2;
		++argv; --argc;
	    }
	    else if (argc > 2) {
		++argv; --argc;
		*p = argv[1];
		++argv; --argc;
	    }
	    else
		fatal(usage);
	    break;

#if 0
	  case 'o':
	    /* set arbitrary options */
	    if (argv[1][2]) {
		opts = argv[1]+2;
		++argv; --argc;
	    }
	    else if (argc > 2) {
		++argv; --argc;
		opts = argv[1];
		++argv; --argc;
	    }
	    else
		fatal(usage);
	    parse_options( opts );
	    break;
#endif
	    
	  case 'l':
	    /* just listing for all devices */
	    listing = 1;
	    type_open = O_RDONLY;
	    try("/dev/hda");
	    try("/dev/hdb");
	    try("/dev/sda");
	    try("/dev/sdb");
	    try("/dev/sdc");
	    try("/dev/sdd");
	    try("/dev/sde");
	    try("/dev/sdf");
	    try("/dev/sdg");
	    try("/dev/sdh");
	    try("/dev/ada");
	    try("/dev/adb");
	    try("/dev/adc");
	    try("/dev/add");
	    try("/dev/ade");
	    try("/dev/adf");
	    try("/dev/adg");
	    try("/dev/adh");
	    exit( 0 );

	  case 's':
	    /* print partition size */
	  {
	      int i;

	      if (argc < 3)
		  fatal( usage );
	      if (strncmp( argv[2], "/dev/", 5 ) || !(i = atoi(argv[2] + 8)))
		  fatal( usage );
	      disk_device = (char *)malloc( 20 );
	      strncpy( disk_device, argv[2], 8 );
	      disk_device[8] = 0;
	      if ((fd = open(disk_device, O_RDONLY)) >= 0) {
		  close(fd);
		  type_open = O_RDONLY;
		  get_boot();
		  if (i > partitions) exit( 1 );
		  printf( "%ld\n", part_table[i].size / 2);
		  exit( 0 );
	      }
	      exit( 1 );
	  }
	      
	  default:
	    fatal( usage );
	}
    }
	
    if (argc > 2)
	fatal(usage);
    else if (argc == 2)
	disk_device = argv[argc - 1];
    else if ((fd = open( disk_device = DEFAULT_DEVICE, type_open )) >= 0) {
	printf( "Using " DEFAULT_DEVICE " as default device.\n" );
	close( fd );
    }
    else if ((fd = open( disk_device = ALTERNATE_DEVICE, type_open )) >= 0) {
	printf( "Using " ALTERNATE_DEVICE " as default device.\n" );
	close( fd );
    }
    else {
	printf( "Using " ALTERNATE2_DEVICE " as default device.\n" );
	disk_device = ALTERNATE2_DEVICE;
    }
    
    get_boot();
    if (argc == 1)
	printf( "Using %s as default device\n", disk_device );
    if (save_sector_file) {
	if (!save_sectors())
	    fatal(couldnt_save_sectors);
    }

#ifdef ALPHA_VERSION
    printf("\nWarning: This software is absolutly alpha!\n");
    printf("         Use on your own risk!!\n");
    printf("         Read source code before use\n");
#endif

    while (1) {
	putchar( '\n' );
	switch( read_char("Command (m for help): ") ) {

	  case 'a':
	    set_bootable();
	    break;

	  case 'b':
	    bad_sector_list(); 
	    break;

	  case 'c':
	    if (partitions)
		change_partition_size( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'C':
	    clear_partitions();
	    break;

	  case 'd':
	    if (partitions)
		delete_partition( get_partition( partitions )); 
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'e':
	    if (partitions)
		convert_partition( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;
	    
	  case 'f':
	    select_xfmt(); 
	    break;

	  case 'i':
	    if (partitions)
		change_contents_flag( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'l':
	    list_atari_types(); 
	    break;

	  case 'm':
	    menu();
	    break;
	    
	  case 'n':
	    new_partition(); 
	    break;

	  case 'o':
	    if (partitions)
		move_partition( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'O':
	    if (!expert_mode)
		goto default_case;
	    if (partitions)
		move_partition_ars( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'p':
	    list_table(); 
	    break;

	  case 'q':
	    close(fd);
	    exit( 0 );

	  case 't':
	    if (partitions)
		change_partid( get_partition( partitions ));
	    else
		printf( "No partitions!\n" );
	    break;

	  case 'v':
	    verify(); 
	    break;

	  case 'w':
	    write_table(); 
	    close(fd);
	    exit( 0 );
	    break;

	  case 'x':
	    expert_mode = !expert_mode;
	    printf( "Expert mode %s\n", expert_mode ? "on" : "off" );
	    break;

	  default:
	  default_case:
	    break;
	}
    }
}

static void menu( void )
{
    puts( "Command action\n"
	  "   a   set bootable flags\n"
	  "   b   bad sector list management\n"
	  "   c   change partition size\n"
	  "   C   clear all partitions\n"
	  "   d   delete a partition\n"
	  "   e   change primary/extended\n"
	  "   f   select partition format (AHDI/ICD)\n"
	  "   i   toggle contents valid/invalid\n"
	  "   l   list known partition types\n"
	  "   m   print this menu\n"
	  "   n   add a new partition\n"
	  "   o   move a partition" );
    if (expert_mode)
    puts( "   O   move a partition's aux. root sector" );
    puts( "   p   print the partition table\n"
	  "   q   quit without saving changes\n"
	  "   t   change a partition's ID\n"
	  "   v   verify the partition table\n"
	  "   w   write table to disk and exit" );
    printf( "   x   %s expert mode\n", expert_mode ? "leave" : "enter" );
}


/* Local Variables: */
/* tab-width: 8     */
/* End:             */
