// Three-dimensional graphing routines

private import math;
import graph;
import three;
import light;

triple Scale(picture pic, triple v)
{
  return (pic.scale.x.T(v.x),pic.scale.y.T(v.y),pic.scale.z.T(v.z));
}

typedef pair direction(real);

pair dir(triple v, triple dir, projection P=currentprojection)
{
  return unit(project(v+dir,P)-project(v,P));
}

direction dir(path3 G, triple dir, projection P=currentprojection)
{
  return new pair(real t) {
    return dir(point(G,t),dir,P);
  };
}

direction perpendicular(path3 G, triple normal,
                        projection P=currentprojection)
{
  return new pair(real t) {
    return dir(point(G,t),cross(dir(G,t),normal),P);
  };
}

real projecttime(path3 G, real T, path g, projection P=currentprojection)
{
  triple v=point(G,T);
  pair z=project(v,P);
  pair dir=dir(v,dir(G,T),P);
  return intersect(g,z)[0];
}

real projecttime(path3 G, real T, projection P=currentprojection)
{
  return projecttime(G,T,project(G,P),P);
}

valuetime linear(picture pic=currentpicture, path3 G, scalefcn S,
                 real Min, real Max, projection P=currentprojection)
{
  real factor=Max == Min ? 0.0 : 1.0/(Max-Min);
  path g=project(G,P);
  return new real(real v) {
    return projecttime(G,(S(v)-Min)*factor,g,P);
  };
}

// Draw a general three-dimensional axis.
void axis(picture pic=currentpicture, Label L="", path3 G, pen p=currentpen,
          ticks ticks, ticklocate locate, arrowbar arrow=None,
          int[] divisor=new int[], bool put=Above,
          projection P=currentprojection,  bool opposite=false) 
{
  divisor=copy(divisor);
  locate=locate.copy();
  Label L=L.copy();
  if(L.defaultposition) L.position(0.5*length(G));
  
  path g=project(G,P);
  pic.add(new void (frame f, transform t, transform T, pair lb, pair rt) {
      frame d;
      ticks(d,t,L,0,g,g,p,arrow,locate,divisor,opposite);
      (put ? add : prepend)(f,t*T*inverse(t)*d);
    });
  
  pic.addPath(g,p);
  
  if(L.s != "") {
    frame f;
    Label L0=L.copy();
    L0.position(0);
    add(f,L0);
    pair pos=point(g,L.relative()*length(g));
    pic.addBox(pos,pos,min(f),max(f));
  }
}

private triple defaultdir(triple X, triple Y, triple Z, projection P) {
  triple u=cross(P.camera-P.target,Z);
  return abs(dot(u,X)) > abs(dot(u,Y)) ? X : Y;
}

// Draw an x axis in three dimensions.
void xaxis(picture pic=currentpicture, Label L="", triple min, triple max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=Y,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection, bool opposite=false) 
{
  if(dir == O) dir=defaultdir(Y,Z,X,P);
  bounds m=autoscale(min.x,max.x,pic.scale.x.scale);
  path3 G=min--max;
  valuetime t=linear(pic,G,pic.scale.x.T(),min.x,max.x,P);
  axis(pic,opposite ? "" : L,G,p,ticks,
       ticklocate(min.x,max.x,pic.scale.x,m.min,m.max,t,dir(G,dir,P)),
       arrow,m.divisor,put,P,opposite);
}

void xaxis(picture pic=currentpicture, Label L="", triple min, real max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=Y,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection, bool opposite=false) 
{
  xaxis(pic,L,min,(max,min.y,min.z),p,ticks,dir,arrow,put,P,opposite);
}

// Draw a y axis in three dimensions.
void yaxis(picture pic=currentpicture, Label L="", triple min, triple max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=X,
           arrowbar arrow=None, bool put=Above, 
           projection P=currentprojection, bool opposite=false) 
{
  if(dir == O) dir=defaultdir(Z,X,Y,P);
  bounds m=autoscale(min.y,max.y,pic.scale.y.scale);
  path3 G=min--max;
  valuetime t=linear(pic,G,pic.scale.y.T(),min.y,max.y,P);
  axis(pic,L,G,p,ticks,
       ticklocate(min.y,max.y,pic.scale.y,m.min,m.max,t,dir(G,dir,P)),
       arrow,m.divisor,put,P,opposite);
}

void yaxis(picture pic=currentpicture, Label L="", triple min, real max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=X,
           arrowbar arrow=None, bool put=Above, 
           projection P=currentprojection, bool opposite=false) 
{
  yaxis(pic,L,min,(min.x,max,min.z),p,ticks,dir,arrow,put,P,opposite);
}

// Draw a z axis in three dimensions.
void zaxis(picture pic=currentpicture, Label L="", triple min, triple max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=O,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection, bool opposite=false) 
{
  if(dir == O) dir=defaultdir(X,Y,Z,P);
  bounds m=autoscale(min.z,max.z,pic.scale.z.scale);
  path3 G=min--max;
  valuetime t=linear(pic,G,pic.scale.z.T(),min.z,max.z,P);
  axis(pic,L,G,p,ticks,
       ticklocate(min.z,max.z,pic.scale.z,m.min,m.max,t,dir(G,dir,P)),
       arrow,m.divisor,put,P,opposite);
}

void zaxis(picture pic=currentpicture, Label L="", triple min, real max,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=O,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection, bool opposite=false) 
{
  zaxis(pic,L,min,(min.x,min.y,max),p,ticks,dir,arrow,put,P,opposite);
}

// Draw an x axis.
// If all=true, also draw opposing edges of the three-dimensional bounding box.
void xaxis(picture pic=currentpicture, Label L="", bool all=false, bbox3 b,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=Y,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection) 
{
  if(dir == O) dir=defaultdir(Y,Z,X,P);

  if(all) {
    bool swap=(abs(dot(dir,Z)) > abs(dot(dir,Y)));
    bounds m=autoscale(b.min.x,b.max.x,pic.scale.x.scale);
  
    void axis(Label L, triple min, triple max, bool opposite=false,
              int sign=1) {
      xaxis(pic,L,min,max,p,ticks,sign*dir,arrow,put,P,opposite);
    }
    bool back=dot(b.Y()-b.O(),P.camera)*sgn(P.camera.z) > 0;
    int sign=back ? -1 : 1;
    axis(L,b.min,b.X(),back,sign);
    axis(L,(b.min.x,b.max.y,b.min.z),(b.max.x,b.max.y,b.min.z),!back,
         swap ? 1 : sign);
    axis(L,(b.min.x,b.min.y,b.max.z),(b.max.x,b.min.y,b.max.z),true,
         swap ? 1 : -1);
    axis(L,(b.min.x,b.max.y,b.max.z),b.max,true);
  } else xaxis(pic,L,b.O(),b.X(),p,ticks,dir,arrow,put,P);
}

// Draw a y axis.
// If all=true, also draw opposing edges of the three-dimensional bounding box.
void yaxis(picture pic=currentpicture, Label L="", bool all=false, bbox3 b,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=X,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection) 
{
  if(dir == O) dir=defaultdir(Z,X,Y,P);

  if(all) {
    bool swap=(abs(dot(dir,Z)) > abs(dot(dir,X)));
    bounds m=autoscale(b.min.y,b.max.y,pic.scale.y.scale);
  
    void axis(Label L, triple min, triple max, bool opposite=false,
              int sign=1) {
      yaxis(pic,L,min,max,p,ticks,sign*dir,arrow,put,P,opposite);
    }
    bool back=dot(b.X()-b.O(),P.camera)*sgn(P.camera.z) > 0;
    int sign=back ? -1 : 1;
    axis(L,b.min,b.Y(),back,sign);
    axis(L,(b.max.x,b.min.y,b.min.z),(b.max.x,b.max.y,b.min.z),!back,
         swap ? 1 : sign);
    axis(L,(b.min.x,b.min.y,b.max.z),(b.min.x,b.max.y,b.max.z),true,
         swap ? 1 : -1);
    axis(L,(b.max.x,b.min.y,b.max.z),b.max,true);
  } else yaxis(pic,L,b.O(),b.Y(),p,ticks,dir,arrow,put,P);
}

// Draw a z axis.
// If all=true, also draw opposing edges of the three-dimensional bounding box.
void zaxis(picture pic=currentpicture, Label L="", bool all=false, bbox3 b,
           pen p=currentpen, ticks ticks=NoTicks, triple dir=O,
           arrowbar arrow=None, bool put=Above,
           projection P=currentprojection) 
{
  if(dir == O) dir=defaultdir(X,Y,Z,P);

  if(all) {
    bool swap=(abs(dot(dir,Y)) > abs(dot(dir,X)));
    bounds m=autoscale(b.min.z,b.max.z,pic.scale.z.scale);
  
    void axis(Label L, triple min, triple max, bool opposite=false,
              int sign=1) {
      zaxis(pic,L,min,max,p,ticks,sign*dir,arrow,put,P,opposite);
    }
    bool back=dot(b.X()-b.O(),P.camera)*sgn(P.camera.y) > 0;
    int sign=back ? -1 : 1;
    axis(L,b.min,b.Z(),back,sign);
    axis(L,(b.max.x,b.min.y,b.min.z),(b.max.x,b.min.y,b.max.z),!back,
         swap ? 1 : sign);
    axis(L,(b.min.x,b.max.y,b.min.z),(b.min.x,b.max.y,b.max.z),true,
         swap ? 1 : -1);
    axis(L,(b.max.x,b.max.y,b.min.z),b.max,true);
  } else zaxis(pic,L,b.O(),b.Z(),p,ticks,dir,arrow,put,P);
}

bounds autolimits(real min, real max, autoscaleT A)
{
  bounds m;
  min=A.scale.T(min);
  max=A.scale.T(max);
  if(A.automin() || A.automax())
    m=autoscale(min,max,A.scale);
  if(!A.automin()) m.min=min;
  if(!A.automax()) m.max=max;
  return m;
}

bbox3 autolimits(picture pic=currentpicture, triple min, triple max) 
{
  bbox3 b;
  bounds mx=autolimits(min.x,max.x,pic.scale.x);
  bounds my=autolimits(min.y,max.y,pic.scale.y);
  bounds mz=autolimits(min.z,max.z,pic.scale.z);
  b.min=(mx.min,my.min,mz.min);
  b.max=(mx.max,my.max,mz.max);
  return b;
}

bbox3 limits(picture pic=currentpicture, triple min, triple max)
{
  bbox3 b;
  b.min=(pic.scale.x.T(min.x),pic.scale.y.T(min.y),pic.scale.z.T(min.z));
  b.max=(pic.scale.x.T(max.x),pic.scale.y.T(max.y),pic.scale.z.T(max.z));
  return b;
};
  
real crop(real x, real min, real max) {return min(max(x,min),max);}

triple xcrop(triple v, real min, real max) 
{
  return (crop(v.x,min,max),v.y,v.z);
}

triple ycrop(triple v, real min, real max) 
{
  return (v.x,crop(v.y,min,max),v.z);
}

triple zcrop(triple v, real min, real max) 
{
  return (v.x,v.y,crop(v.z,min,max));
}

void xlimits(bbox3 b, real min, real max)
{
  b.min=xcrop(b.min,min,max);
  b.max=xcrop(b.max,min,max);
}

void ylimits(bbox3 b, real min, real max)
{
  b.min=ycrop(b.min,min,max);
  b.max=ycrop(b.max,min,max);
}

void zlimits(bbox3 b, real min, real max)
{
  b.min=zcrop(b.min,min,max);
  b.max=zcrop(b.max,min,max);
}

// Restrict the x, y, and z limits to box(min,max).
void limits(bbox3 b, triple min, triple max)
{
  xlimits(b,min.x,max.x);
  ylimits(b,min.y,max.y);
  zlimits(b,min.z,max.z);
}
  
void axes(picture pic=currentpicture,
          Label xlabel="", Label ylabel="", Label zlabel="", 
          bbox3 b, pen p=currentpen, arrowbar arrow=None,
          bool put=Below, projection P=currentprojection)
{
  xaxis(pic,xlabel,b,p,arrow,put,P);
  yaxis(pic,ylabel,b,p,arrow,put,P);
  zaxis(pic,zlabel,b,p,arrow,put,P);
}

void axes(picture pic=currentpicture,
          Label xlabel="", Label ylabel="", Label zlabel="", 
          triple min, triple max, pen p=currentpen, arrowbar arrow=None,
          bool put=Below, projection P=currentprojection)
{
  axes(pic,xlabel,ylabel,zlabel,limits(min,max),p,arrow,put,P);
}

void xtick(picture pic=currentpicture, Label L="", triple v, triple dir=Y,
           string format="", real size=Ticksize, pen p=currentpen,
           projection P=currentprojection)
{
  if(L.s == "") L.s=format(format == "" ? defaultformat : format,v.x);
  xtick(pic,L,project(v,P),project(dir,P),format,size,p);
}

void ytick(picture pic=currentpicture, Label L="", triple v, triple dir=X,
           string format="", real size=Ticksize, pen p=currentpen,
           projection P=currentprojection)
{
  if(L.s == "") L.s=format(format == "" ? defaultformat : format,v.y);
  xtick(pic,L,project(v,P),project(dir,P),format,size,p);
}

void ztick(picture pic=currentpicture, Label L="", triple v, triple dir=Y,
           string format="", real size=Ticksize, pen p=currentpen,
           projection P=currentprojection)
{
  if(L.s == "") L.s=format(format == "" ? defaultformat : format,v.z);
  xtick(pic,L,project(v,P),project(dir,P),format,size,p);
}

typedef guide3 graph(triple F(real), real, real, int);

typedef guide3 interpolate3(... guide3[]);

graph graph(interpolate3 join)
{
  return new guide3(triple f(real), real a, real b, int n) {
    real width=b-a;
    return n == 0 ? join(f(a)) :
      join(...sequence(new guide3(int i) {
            return f(a+(i/n)*width);
          },n+1));
  };
}

guide3 Straight(... guide3[])=operator --;
guide3 Spline(... guide3[])=operator ..;
                       
guide3 graph(picture pic=currentpicture, real x(real), real y(real),
             real z(real), real a, real b, int n=ngraph,
             interpolate3 join=operator --)
{
  return graph(join)(new triple(real t) {return Scale(pic,(x(t),y(t),z(t)));},
                     a,b,n);
}

guide3 graph(picture pic=currentpicture, triple v(real), real a, real b,
             int n=ngraph, interpolate3 join=operator --)
{
  return graph(join)(new triple(real t) {return Scale(pic,v(t));},a,b,n);
}

int[] conditional(triple[] v, bool[] cond)
{
  if(cond.length > 0) {
    checklengths(cond.length,v.length,conditionlength);
    return cond ? sequence(cond.length) : null;
  } else return sequence(v.length);
}

guide3 graph(picture pic=currentpicture, triple[] v, bool[] cond={},
             interpolate3 join=operator --)
{
  int[] I=conditional(v,cond);
  int k=0;
  return graph(join)(new triple(real) {
      int i=I[k]; ++k;
      return Scale(pic,v[i]);}
    ,0,0,I.length-1);
}

guide3 graph(picture pic=currentpicture, real[] x, real[] y, real[] z,
             bool[] cond={}, interpolate3 join=operator --)
{
  checklengths(x.length,y.length);
  checklengths(x.length,z.length);
  int[] I=conditional(x,cond);
  int k=0;
  return graph(join)(new triple(real) {
      int i=I[k]; ++k;
      return Scale(pic,(x[i],y[i],z[i]));
    },0,0,I.length-1);
}

// The graph of a function along a path.
guide3 graph(triple F(path, real), path p, int n=1,
             interpolate3 join=operator --)
{
  guide3 g=join(...sequence(new guide3(int i) {
        return F(p,i/n);
      },n*length(p)));
  return cyclic(p) ? join(g,cycle) : join(g,F(p,length(p)));
}

guide3 graph(triple F(pair), path p, int n=1, interpolate3 join=operator --)
{
  return graph(new triple(path p, real position) 
               {return F(point(p,position));},p,n,join);
}

guide3 graph(picture pic=currentpicture, real f(pair), path p, int n=1,
             interpolate3 join=operator --) 
{
  return graph(new triple(pair z) {return Scale(pic,(z.x,z.y,f(z)));},p,n,
               join);
}

guide3 graph(real f(pair), path p, int n=1, real T(pair),
             interpolate3 join=operator --)
{
  return graph(new triple(pair z) {pair w=T(z); return (w.x,w.y,f(w));},p,n,
               join);
}

// Connect points in v into segments corresponding to consecutive true elements
// of b using interpolation operator join. 
path3[] segment(triple[] v, bool[] b, interpolate3 join=operator --)
{
  checklengths(v.length,b.length,conditionlength);
  int[][] segment=segment(b);
  return sequence(new path3(int i) {return join(... v[segment[i]]);},
                  segment.length);
}

// draw the surface described by a matrix f, respecting lighting
picture surface(triple[][] f, bool[][] cond={}, bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  picture pic;
  
  if(!rectangular(f)) abort("matrix is not rectangular");
  
  // Draw a mesh in the absence of lighting (override with meshpen=invisible).
  if(light.source == O && meshpen == nullpen) meshpen=currentpen;

  int nx=f.length-1;
  int ny=nx > 0 ? f[0].length-1 : 0;
  
  // calculate colors at each point
  pen color(int i, int j) {
    triple dfx,dfy;
    if(i == 0) dfx=f[1][j]-f[0][j];
    else if(i == nx) dfx=f[nx][j]-f[nx-1][j];
    else dfx=0.5(f[i+1][j]-f[i-1][j]);
    if(j == 0) dfy=f[i][1]-f[i][0];
    else if(j == ny) dfy=f[i][ny]-f[i][ny-1];
    else dfy=0.5(f[i][j+1]-f[i][j-1]);
    triple v=cross(dfx,dfy);
    if(!outward)
      v *= sgn(dot(v,P.camera-P.target));
    return light.intensity(v)*surfacepen;
  }

  int[] edges={0,0,0,2};
  
  void drawcell(int i, int j) {
    path g=project(f[i][j],P)--project(f[i][j+1],P)--project(f[i+1][j+1],P)--
      project(f[i+1][j],P)--cycle;
    if(surfacepen != nullpen) {
      if(light.source == O)
        fill(pic,g,surfacepen);
      else {
        pen[] pcell={color(i,j),color(i,j+1),color(i+1,j+1),color(i+1,j)};
        gouraudshade(pic,g,pcell,edges);
      }
    }
    if(meshpen != nullpen) draw(pic,g,meshpen);
  }
  
  bool all=cond.length == 0;

  if(surfacepen == nullpen) {
    for(int i=0; i < nx; ++i)
      for(int j=0; j < ny; ++j)
	if(all || cond[i][j])
	  drawcell(i,j);
  } else {
    // Sort cells by distance from camera
    triple camera=P.camera;
    if(P.infinity)
      camera *= max(abs(minbound(f)),abs(maxbound(f)));

    real[][] depth;
    for(int i=0; i < nx; ++i) {
      for(int j=0; j < ny; ++j) {
	if(all || cond[i][j]) {
	  real d=abs(camera-0.25(f[i][j]+f[i][j+1]+f[i+1][j]+f[i+1][j+1]));
	  depth.push(new real[] {d,i,j});
	}
      }
    }

    depth=sort(depth);
  
    // Draw from farthest to nearest
    while(depth.length > 0) {
      real[] a=depth.pop();
      drawcell(round(a[1]),round(a[2]));
    }
  }

  return pic;
}

// draw the surface described by a real matrix f, respecting lighting
picture surface(real[][] f, pair a, pair b, bool[][] cond={},
		bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  if(!rectangular(f)) abort("matrix is not rectangular");

  int nx=f.length-1;
  int ny=nx > 0 ? f[0].length-1 : 0;

  if(nx == 0 || ny == 0) return new picture;

  triple[][] v=new triple[nx+1][ny+1];

  for(int i=0; i <= nx; ++i) {
    real x=interp(a.x,b.x,i/nx);
    for(int j=0; j <= ny; ++j) {
      v[i][j]=(x,interp(a.y,b.y,j/ny),f[i][j]);
    }
  }
  return surface(v,cond,outward,surfacepen,meshpen,light,P);
}

// draw the surface described by a parametric function f over box(a,b),
// respecting lighting.
picture surface(triple f(pair z), pair a, pair b, int nu=nmesh, int nv=nu,
                bool cond(pair z)=null, bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  if(nu <= 0 || nv <= 0) return new picture;
  triple[][] v=new triple[nu+1][nv+1];
  bool[][] active;

  bool all=cond == null;
  if(!all) active=new bool[nu+1][nv+1];

  real du=1/nu;
  real dv=1/nv;
  pair Idv=(0,dv);
  pair dz=(du,dv);
  for(int i=0; i <= nu; ++i) {
    real x=interp(a.x,b.x,i*du);
    for(int j=0; j <= nv; ++j) {
      pair z=(x,interp(a.y,b.y,j*dv));
      v[i][j]=f(z);
      if(!all) active[i][j]=cond(z) || cond(z+du) || cond(z+Idv) || cond(z+dz);
    }
  }
  return surface(v,active,outward,surfacepen,meshpen,light,P);
}

// draw the surface described by a real function f over box(a,b),
// respecting lighting.
picture surface(real f(pair z), pair a, pair b, int nx=nmesh, int ny=nx,
                bool cond(pair z)=null, bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  return surface(new triple(pair z) {return (z.x,z.y,f(z));},a,b,nx,ny,
                 cond,outward,surfacepen,meshpen,light,P);
}

// draw the surface described by a parametric function f over box(a,b),
// subsampled nsub times, respecting lighting.
picture surface(triple f(pair z), int nsub, pair a, pair b,
		int nu=nmesh, int nv=nu,
		bool cond(pair z)=null, bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  picture pic;

  // Draw a mesh in the absence of lighting (override with meshpen=invisible).
  if(light.source == O && meshpen == nullpen) meshpen=currentpen;

  pair sample(int i, int j) {
    return (interp(a.x,b.x,i/nu),interp(a.y,b.y,j/nv));
  }
  
  pen color(int i, int j) {
    triple dfx,dfy;
    if(i == 0) dfx=f(sample(1,j))-f(sample(0,j));
    else if(i == nu) dfx=f(sample(nu,j))-f(sample(nu-1,j));
    else dfx=0.5(f(sample(i+1,j))-f(sample(i-1,j)));
    if(j == 0) dfy=f(sample(i,1))-f(sample(i,0));
    else if(j == nv) dfy=f(sample(i,nv))-f(sample(i,nv-1));
    else dfy=0.5(f(sample(i,j+1))-f(sample(i,j-1)));
    triple v=cross(dfx,dfy);
    if(!outward)
      v *= sgn(dot(v,P.camera-P.target));
    return light.intensity(v)*surfacepen;
  }

  path3 cell(int i, int j) { 
    return graph(f,box(sample(i,j),sample(i+1,j+1)),nsub);
  }

  void drawcell(int i, int j) {
    path g=project(cell(i,j),P);
    fill(pic,g,color(i,j));
    if(meshpen != nullpen) draw(pic,g,meshpen);
  }
  
  bool all=cond == null;

  bool active(int i, int j) {
    return all ||
      cond(sample(i,j)) || cond(sample(i,j+1)) ||
      cond(sample(i+1,j)) || cond(sample(i+1,j+1));
  }
  
  if(surfacepen == nullpen) {
    for(int i=0; i < nu; ++i)
      for(int j=0; j < nv; ++j)
	if(active(i,j))
	  draw(pic,project(cell(i,j),P),meshpen);
  } else {
    // Sort cells by distance from camera
    triple camera=P.camera;
    if(P.infinity) {
      real r=0;
      for(int i=0; i <= nu; ++i)
        for(int j=0; j <= nv; ++j)
	  if(active(i,j))
	    r=max(r,abs(f(sample(i,j))));
      camera *= r;
    }

    real[][] depth;
    for(int i=0; i < nu; ++i) {
      for(int j=0; j < nv; ++j) {
	if(active(i,j)) {
	  real d=abs(camera-0.25*(f(sample(i,j))+f(sample(i,j+1))+
				  f(sample(i+1,j))+f(sample(i+1,j+1))));
	  depth.push(new real[] {d,i,j});
	}
      }
    }

    depth=sort(depth);
  
    // Draw from farthest to nearest
    while(depth.length > 0) {
      real[] a=depth.pop();
      drawcell(round(a[1]),round(a[2]));
    }
  }
  return pic;
}

// draw the surface described by a real function f over box(a,b),
// subsampled nsub times, respecting lighting.
picture surface(real f(pair z), int nsub, pair a, pair b, 
                int nx=nmesh, int ny=nx, bool cond(pair z)=null,
		bool outward=false,
                pen surfacepen=lightgray, pen meshpen=nullpen,
                light light=currentlight, projection P=currentprojection)
{
  return surface(new triple(pair z) {return (z.x,z.y,f(z));},nsub,a,b,nx,ny,
                 cond,outward,surfacepen,meshpen,light,P);
}

guide3[][] lift(real f(real x, real y), guide[][] g,
                interpolate3 join=operator --)
{
  guide3[][] G=new guide3[g.length][];
  for(int cnt=0; cnt < g.length; ++cnt) {
    guide[] gcnt=g[cnt];
    guide3[] Gcnt=new guide3[gcnt.length];
    for(int i=0; i < gcnt.length; ++i) {
      guide gcnti=gcnt[i];
      guide3 Gcnti=join(...sequence(new guide3(int j) {
            pair z=point(gcnti,j);
            return (z.x,z.y,f(z.x,z.y));
          },size(gcnti)));
      if(cyclic(gcnti)) Gcnti=Gcnti..cycle;
      Gcnt[i]=Gcnti;
    }
    G[cnt]=Gcnt;
  }
  return G;
}

guide3[][] lift(real f(pair z), guide[][] g, interpolate3 join=operator --)
{
  return lift(new real(real x, real y) {return f((x,y));},g,join);
}

void draw(picture pic=currentpicture, Label[] L=new Label[],
          guide3[][] g, pen[] p, Label[] legend=new Label[],
          projection P=currentprojection)
{
  begingroup(pic);
  for(int cnt=0; cnt < g.length; ++cnt) {
    guide3[] gcnt=g[cnt];
    pen pcnt=p[cnt];
    for(int i=0; i < gcnt.length; ++i) {
      guide G=project(gcnt[i],P);
      if (i == 0 && legend.length > 0) draw(pic,G,pcnt,legend[cnt]);
      else draw(pic,G,pcnt);
      if(L.length > 0) {
        Label Lcnt=L[cnt];
        if(Lcnt.s != "" && size(gcnt[i]) > 1)
          label(pic,Lcnt,G,pcnt);
      }
    }
  }
  endgroup(pic);
}

void draw(picture pic=currentpicture, Label[] L=new Label[],
          guide3[][] g, pen p=currentpen, Label[] legend=new Label[],
          projection P=currentprojection)
{
  draw(pic,L,g,sequence(new pen(int) {return p;},g.length),legend,P);
}

triple polar(real r, real theta, real phi)
{
  return r*expi(theta,phi);
}

guide3 polargraph(real r(real,real), real theta(real), real phi(real),
                  int n=ngraph, interpolate3 join=operator --)
{
  return graph(join)(new triple(real t) {
      return polar(r(theta(t),phi(t)),theta(t),phi(t));
    },0,1,n);
}

picture vectorfield(path3 vector(pair z), triple f(pair z),
		    pair a, pair b, int nx=nmesh, int ny=nx,
		    bool autoscale=true,
		    pen p=currentpen, arrowbar arrow=Arrow,
		    projection P=currentprojection)
{
  picture pic;
  real dx=1/nx;
  real dy=1/ny;
  real scale;
  if(autoscale) {
    real size(pair z) {
      path3 g=vector(z);
      return abs(point(g,size(g)-1)-point(g,0));
    }
    real max=size((0,0));
    for(int i=0; i <= nx; ++i) {
      real x=interp(a.x,b.x,i*dx);
      for(int j=0; j <= ny; ++j)
	max=max(max,size((x,interp(a.y,b.y,j*dy))));
    }
    pair lambda=(abs(f((b.x,a.y))-f(a)),abs(f((a.x,b.y))-f(a)));
    scale=min(lambda.x/nx,lambda.y/ny)/max;
  } else scale=1;
  for(int i=0; i <= nx; ++i) {
    real x=interp(a.x,b.x,i*dx);
    for(int j=0; j <= ny; ++j) {
      real y=interp(a.y,b.y,j*dy);
      pair z=(x,y);
      draw(pic,project(shift(f(z))*scale3(scale)*vector(z),P),p,arrow);
    }
  }
  return pic;
}

// True arc
path3 Arc(triple c, real r, real theta1, real phi1, real theta2, real phi2,
          triple normal=Z, int n=400)
{
  path3 p=polargraph(new real(real theta, real phi) {return r;},
                     new real(real t) {
                       return radians(interp(theta1,theta2,t));},
                     new real(real t) {return radians(interp(phi1,phi2,t));},
                     n,operator ..);
  if(normal != Z)
    p=rotate(longitude(normal,warn=false),Z)*rotate(colatitude(normal),Y)*p;
  return shift(c)*p;
}

// True circle
path3 Circle(triple c, real r, triple normal=Z, int n=400)
{
  return Arc(c,r,90,0,90,360,normal,n)..cycle;
}
