// Copyright 2000 by Kevin Atkinson under the terms of the LGPL

#include <vector>
#include "hash-t.hh"
#include "hash_string_s.hh"

#include "writable_base.hh"
#include "language.hh"
#include "emulation.hh"
#include "data_util.hh"
#include "file_exceps.hh"

#include "simple_string.hh"
#include "hash_simple_string.hh"

namespace aspell_default_writable_repl {

  using namespace aspell;
  using namespace autil;
  using namespace aspell::writable_base;
  using namespace aspell::data_util;


  /////////////////////////////////////////////////////////////////////
  // 
  //  WritableReplList
  //

  // LookupTable looks like this
  // HashMap:
  //    key:   soundslike (for misspelled word)
  //    value: vector: RealReplacementList:
  //                     misspelled_word()
  //                     begin() | 
  //                     end()   | Correct spelling(s) for misspelled word.
  //                     size()  |


  class WritableReplS : public WritableBase<WritableReplacementSet>
  {
  public: // but don't use
      
    class RealReplacementList {
      vector<string> info;
    public:
      typedef vector<string>::const_iterator const_iterator;
      typedef const_iterator                 iterator;
      typedef vector<string>::size_type      size_type;
      RealReplacementList() : info(1) {}
      RealReplacementList(const string &mis, size_type num) {
	info.reserve(num+1); info.push_back(mis);
      }
	
      RealReplacementList(const string &mis, const string &cor) 
	: info(2) 
      {
	info[0] = mis; info[1] = cor;
      }
	
      const string& misspelled_word() const {return info[0];}
      const_iterator begin() const {return info.begin()+1;}
      const_iterator end()   const {return info.end();}
      size_type      size()  const {return info.size()-1;}
      bool add(const string &);
      void add_nocheck(const string &r) {info.push_back(r);}
      bool erase(const string &);
      bool exists(const string &);
    };
      
    class RealReplList : public vector<RealReplacementList> {};
    // ^ needed to reduce symbol length for some non-gnu assemblers
    typedef hash_map<SimpleString, RealReplList>  LookupTable;
      
  private:
    //   RealRepl.. is a custom struct where begin() and end() are
    //     iterators for the repl. list for a misspelled word
    //   The _elements_ of the hash_map are a vector of RealRepl.
    //      where the mispelled_word all have a soundslike eqilvent
    //      to _key_.
      
    LookupTable * lookup_table;
    void save(SimpleFstream &, const string &);
    void merge(SimpleFstream &, const string &, Config * config = 0);
    
  private:
    WritableReplS(const WritableReplS&);
    WritableReplS& operator=(const WritableReplS&);
  public:
    WritableReplS() 
      : WritableBase<WritableReplacementSet>(".prepl",".rpl")
    {lookup_table = new LookupTable();}
    ~WritableReplS() {delete lookup_table;}

    struct ElementsVirEmulImpl;
    VirEmul * elements() const;
    Size      size()     const;
    bool      empty()    const;
      
    void clear();
      
    void add(const string &mis, const string &cor);
    void add(const string &mis, const string &cor, const string &s);
      
    struct ReplsWSoundslikeParms;
    VirEmul * repls_w_soundslike(const char * soundslike) const;
    VirEmul * repls_w_soundslike(SoundslikeWord soundslike) const;
      
    struct SoundslikeElementsParms;
    VirSoundslikeEmul * soundslike_elements() const;
  };
    
  //
  // FIXME:  Fix file nameing issue
  // so that aspell create repl < tmp will WORK!!!!
  // change class file_name to cur_file_name
  // and same with cur_file_date
  //
    
  bool WritableReplS::RealReplacementList::exists(const string &word) {
    iterator i = begin();
    iterator e = end();
    while (i != e) {
      if (*i == word) return true;
      ++i;
    }
    return false;
  }
    
  bool WritableReplS::RealReplacementList::add(const string &word) {
    if (exists(word)) return false;
    info.push_back(word);
    return true;
  }
    
  bool WritableReplS::RealReplacementList::erase(const string &word) {
    vector<string>::iterator i = info.begin() + 1;
    vector<string>::iterator e = info.end();
    while (i != e) {
      if (*i == word) {
	info.erase(i);
	return true;
      }
      ++i;
    }
    return false;
  }
    
  class WritableReplS::ElementsVirEmulImpl : public VirEmulation<ReplacementList> {
  private:
    typedef LookupTable::const_iterator  OuterItr;
    typedef RealReplList::const_iterator InnerItr;
    OuterItr outer_;
    OuterItr end_;
    InnerItr inner_;
  public:
    // this assums LookupTable is non empty
    ElementsVirEmulImpl (const LookupTable & c)
      : outer_(c.begin()), end_(c.end()) 
    {if (outer_ != end_) inner_ = outer_->second.begin();}
	
    ElementsVirEmulImpl * clone() const {
      return new ElementsVirEmulImpl(*this);
    }
      
    void assign(const VirEmulation<Value> * other) {
      *this = *static_cast<const ElementsVirEmulImpl *>(other);
    }
      
    Value next() {
      if (outer_ == end_) return ReplacementList();
      if (inner_ == outer_->second.end()) {
	++outer_;
	if (outer_ == end_) return ReplacementList();
	inner_ = outer_->second.begin();
      }
      ReplacementList temp
	(inner_->misspelled_word().c_str(), 
	 new MakeVirEmulation<StrParms<RealReplacementList::const_iterator> >
	 (inner_->begin(), inner_->end()));
      ++inner_;
      return temp;
    }
    
    bool at_end() const {return outer_ == end_;}
      
  };
    
  WritableReplS::VirEmul * WritableReplS::elements() const {
    return new ElementsVirEmulImpl(*lookup_table);
  }
    
  //FIXME: Don't always return a size of 0!!!!
  WritableReplS::Size WritableReplS::size() const {return 0;}
      
  bool WritableReplS::empty() const {
    return lookup_table->empty();
  }

  void WritableReplS::add(const string &mis, const string &cor) {
    add(mis, cor, lang()->to_soundslike(mis));
  }

  void WritableReplS::add(const string &mis, const string &cor, const string &s) {

    LookupTable::iterator i = lookup_table->find(SimpleString(s.c_str(),1));
    if (i == lookup_table->end())
      i = lookup_table->insert
	(LookupTable::value_type(s.c_str(), RealReplList())).first;
  
    RealReplList::iterator    j = i->second.begin();
    RealReplList::iterator    e = i->second.end();
    for (; j != e; ++j) {
      if ((*j).misspelled_word() == mis) {
	(*j).add(cor);
	return;
      }
    }
    i->second.push_back(RealReplacementList(mis,cor));
  }

  struct WritableReplS::ReplsWSoundslikeParms {
    typedef ReplacementList                             Value;
    typedef vector<RealReplacementList>::const_iterator Iterator;
    Iterator end_;
    ReplsWSoundslikeParms(Iterator e) : end_(e) {}
    bool endf(Iterator i) const {return i == end_;}
    Value end_state() const {return ReplacementList();}
    Value deref(Iterator i) const {
      return ReplacementList
	(i->misspelled_word().c_str(), 
	 new MakeVirEmulation<StrParms<RealReplacementList::const_iterator> >
	 (i->begin(), i->end()));
    }
  };
    
  WritableReplS::VirEmul *
  WritableReplS::repls_w_soundslike(const char * soundslike) const {
      
    LookupTable::const_iterator i = 
      lookup_table->find(SimpleString(soundslike,1));
      
    if (i == lookup_table->end()) {
      return new MakeAlwaysEndEmulation<ReplacementList>();
    } else {
      return new MakeVirEmulation<ReplsWSoundslikeParms>
	(i->second.begin(), ReplsWSoundslikeParms(i->second.end()));
    }
  }
  

  WritableReplS::VirEmul *
  WritableReplS::repls_w_soundslike(SoundslikeWord soundslike) const {
    
    const RealReplList * p = 
      reinterpret_cast<const RealReplList *>(soundslike.word_list_pointer);

    return new MakeVirEmulation<ReplsWSoundslikeParms>(p->begin(), p->end());

  }

  struct WritableReplS::SoundslikeElementsParms {
    typedef SoundslikeWord                   Value;
    typedef LookupTable::const_iterator      Iterator;
    Iterator end_;
    SoundslikeElementsParms(Iterator e) : end_(e) {}
    bool endf(Iterator i) const {return i==end_;}
    static Value deref(Iterator i) {
      return Value(i->first.c_str(),
		   reinterpret_cast<const void *>(&i->second));
    }
    static Value end_state() {return Value(0,0);}
  };

  WritableReplS::VirSoundslikeEmul * 
  WritableReplS::soundslike_elements() const {
    return new MakeVirEmulation<SoundslikeElementsParms>
      (lookup_table->begin(),SoundslikeElementsParms(lookup_table->end()));
  }

  void WritableReplS::save (SimpleFstream & out, const string & file_name) 
  {
    out << "personal_repl-1.1" << ' ' << lang_name() <<  " 0 \n";
  
    LookupTable::iterator i = lookup_table->begin();
    LookupTable::iterator e = lookup_table->end();
  
    for (;i != e; ++i) {
      for (RealReplList::iterator j = i->second.begin(); 
	   j != i->second.end(); 
	   ++j) 
	{
	  for (RealReplacementList::iterator k = j->begin(); 
	       k != j->end(); 
	       ++k) 
	    {
	      out << (*j).misspelled_word() << ' ' << *k << '\n';
	    }
	}
    }
  }

  void WritableReplS::merge(SimpleFstream & in,
			    const string & file_name, 
			    Config * config)
  {
    unsigned int c;
    unsigned int version;
    string word, mis, sound, repl;
    unsigned int num_words, num_repls;

    in >> word;
    if (word == "personal_repl")
      version = 10;
    else if (word == "personal_repl-1.1") 
      version = 11;
    else
      throw BadFileFormat(file_name);

    in >> word;

    try {
      set_check_lang(word, config);
    } catch (RethrowWFile & e) {
      e.rethrow_w_file(file_name);
    }

    unsigned int num_soundslikes;
    if (version == 10) {
      in >> num_soundslikes;
    }
    in >> c;  // not used at the moment
    in.skipws();

    if (version == 11) {

      do {
	getline(in, mis, ' ');
	if (!in) break;
	getline(in, repl, '\n');
	if (!in) throw BadFileFormat(file_name);
	add(mis, repl);
      } while (true);

    } else {

      int h,i,j;
      for (h=0; h != num_soundslikes; ++h) {
	in >> sound >> num_words;
	for (i = 0; i != num_words; ++i) {
	  in >> mis >> num_repls;
	  in.ignore(); // ignore space
	  for (j = 0; j != num_repls; ++j) {
	    getline(in, repl, ',');
	    add(mis, repl);
	  }
	}
      }

    }

  }

  void WritableReplS::clear() {
    delete lookup_table;
    lookup_table = new LookupTable();
  }
}

namespace aspell {
  WritableReplacementSet * new_default_writable_replacement_set() {
    return new aspell_default_writable_repl::WritableReplS();
  }
}

