// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "ThisJoinPoint.h"
#include "Naming.h"
#include "Binding.h"
#include "ACModel/Utils.h"
#include "ResultBuffer.h"
#include "AdviceInfo.h"

#include <sstream>
using std::stringstream;
using std::endl;

// special setup function for bindings
void ThisJoinPoint::setup (const Binding &binding) {
  if (binding._used) {
    _used |= (binding._this != 0) ? THAT : 0;
    _used |= (binding._target != 0) ? TARGET : 0;
    _used |= (binding._result != 0) ? RESULT : 0;
    _used |= (binding._args.length () > 0) ? ARG : 0;
    _used |= (PTR_NEEDED|TYPE_NEEDED);
  }
}

void ThisJoinPoint::check_field (const char* field, bool dyn_only) {
  if (strcmp (field, "signature") == 0)
    _used |= SIGNATURE;
  else if (strcmp (field, "filename") == 0)
    _used |= FILENAME;
  else if (strcmp (field, "line") == 0)
    _used |= LINE;
  else if (strcmp (field, "args") == 0)
    _used |= ARGS;
  else if (strcmp (field, "arg") == 0)
    _used |= ARG;
  else if (strcmp (field, "argtype") == 0)
    _used |= ARG_TYPE;
  else if (strcmp (field, "type") == 0)
    _used |= TYPE;
  else if (strcmp (field, "id") == 0 || (!dyn_only && strcmp (field, "JPID") == 0))
    _used |= ID;
  else if (strcmp (field, "resulttype") == 0)
    _used |= RESULT_TYPE;
  else if (strcmp (field, "that") == 0)
    _used |= THAT;
  else if (strcmp (field, "target") == 0)
    _used |= TARGET;
  else if (strcmp (field, "result") == 0)
    _used |= RESULT;
  else if (strcmp (field, "jptype") == 0)
    _used |= JP_TYPE;
  else if (strcmp (field, "action") == 0)
    _used |= ACTION;
  else if (strcmp (field, "wrapper") == 0)
    _used |= WRAPPER;
  else if (strcmp (field, "proceed") == 0)
    _used |= PROCEED;
}

bool ThisJoinPoint::check_type (const string &name) {
  if (name == "JoinPoint") {
    _used |= TYPE_NEEDED;
    return true;
  }
  return false;
}

bool ThisJoinPoint::check_obj (const string &name) {
  if (name == "tjp") {
    _used |= (PTR_NEEDED | TYPE_NEEDED);
    return true;
  }
  else if (name == "thisJoinPoint") {
    _used |= (PTR_NEEDED | PTR_ALIAS_NEEDED | TYPE_NEEDED);
    return true;
  }
  return false;
}

void ThisJoinPoint::gen_tjp_struct (ostream &code, ACM_Code *loc,
                                    BackEndProblems &bep, int depth) const {
  
  // the level 0 struct -> contains all the data and types
  code << "template <typename TResult, typename TThat, typename TTarget, ";
  code << "typename TArgs> struct ";
  Naming::tjp_struct (code, loc, depth);
  if (depth > 0) {
    // the level N > 0 structs -> are derived from level N - 1
    code << " : ";
    Naming::tjp_struct (code, loc, depth - 1);
    code << "<TResult, TThat, TTarget, TArgs>";
  }
  else {
    // the level N == 0 structs are derived from AC::Action, if action() used
    if (useAction())
      code << " : AC::Action";
  }
  code << " {" << endl;
  code << "  typedef ";
  Naming::tjp_struct (code, loc, depth);
  code << " __TJP;" << endl; // internal type definition
  
  stringstream types, data, fct;

  // start: type definitions --------------------------------------------------
  bool res_is_undef = !has_result_type(*loc);
  bool res_is_ref   = false;
  if (!res_is_undef) {
    ACM_Type *rtype = get_result_type (*loc);
    MatchSignature &match_sig = rtype->get_match_sig();
    if (match_sig.is_new())
      match_sig.parse(format_type (*rtype));
    if (match_sig.type ().is_reference())
      res_is_ref = true;
  }
  types << "  typedef TResult " << (res_is_ref ? "*" : "") << "Result;" << endl;
  types << "  typedef TThat   That;" << endl;
  types << "  typedef TTarget Target;" << endl;
  
  // argument count and types
  types << "  enum { ARGS = TArgs::ARGS };" << endl;
  types << "  template <int I> struct Arg : AC::Arg<TArgs, I> {};" << endl;
    
  // join point id and type;
  // TODO: handle join-point ID in TJP
  if (id()) {
    int jpid = loc->get_jpid(); //(loc->assigned_id () == -1 ? loc->id () : loc->assigned_id ());
    types << "  static const int JPID = " << jpid << ";" << endl;
  }
  types << "  static const AC::JPType JPTYPE = (AC::JPType)" << loc->type_val ()
        << ";" << endl;
  
  // result type
  types << "  struct Res {" << endl;
  types << "    typedef " << (res_is_undef ? "void" : "TResult") << " "
        << (res_is_ref ? "&" : "") << "Type;" << endl;
  types << "    typedef " << (res_is_undef ? "void" : "TResult") << " ReferredType;" << endl;
  types << "  };" << endl;
  
  // argument types
  unsigned arg_count = get_arg_count (*loc);
  
  // begin of static data -----------------------------------------------------
  if (signature ()) {
    fct << "  inline static const char *signature () { return \"";
    fct << ::signature (*loc) << "\";}" << endl;
  }
  if (filename ()) {
    fct << "  inline static const char *filename () { return \"";
    fct << ::filename (*loc) << "\";}" << endl;
  }
  if (line ()) {
    fct << "  inline static int line () { return ";
    fct << ::line(*loc) << ";}" << endl;
  }
  if (args ()) {
    fct << "  inline static const int args () { return ";
    fct << arg_count << ";";
    fct << " }" << endl;
  }
  if (type()) {
    fct << "  inline static  AC::Type type() { return ";
    fct << "\"";
    if (has_result_type(*loc))
      Naming::mangle (fct, get_result_type(*loc));
    list<ACM_Type*> tlist;
    get_arg_types (*loc, tlist);
    for (list<ACM_Type*>::iterator iter = tlist.begin(); iter != tlist.end();
        ++iter)
      Naming::mangle (fct, *iter);
    fct << "\"; }" << endl;
  }
  if (id()) {
    fct << "  inline static unsigned int id() { return JPID; }" << endl;
  }
  
  if (resulttype()) {
    fct << "  inline static AC::Type resulttype() {return ";
    if (has_result_type(*loc)) {
      fct << "\"";
      Naming::mangle (fct, get_result_type(*loc));
      fct << "\";";
    } else fct << "\"<unknown signature>\";";
    fct << "}" << endl;    
  }
  
  if (jptype()) {
    fct << "  inline static AC::JPType jptype() { return JPTYPE; }" << endl;
  }
  
  if (argtype()) {
    fct << "  inline static AC::Type const argtype(unsigned i) {" << endl;
    if (arg_count > 0) {
      fct << "    static AC::Type const type[" << arg_count << "] = {";
      list<ACM_Type*> tlist;
      get_arg_types (*loc, tlist);
      unsigned i = 0;
      for (list<ACM_Type*>::iterator iter = tlist.begin(); iter != tlist.end();
           ++iter, ++i) {
        if (i > 0) fct << ", ";
        fct << "\"";
        Naming::mangle (fct, *iter);
        fct << "\"";
      }
      fct << "};" << endl;
      fct << "    return type[i];" << endl;
      fct << "  }" << endl;
    } else {
      fct << "    return \"\";" << endl;
      fct << "  }" << endl;
    }
  }
   
  // begin of dynamic data ----------------------------------------------------
  if (arg () || useAction () || (proceed () && proceed_needs_args (*loc))) {
    if (!useAction () && arg_count > 0)
      data << "  void *_args[ARGS];" << endl;
    fct <<  "  inline void *arg (int n) {return "
        << (arg_count > 0 ? "_args[n]" : "0") << ";}" << endl;
    fct <<  "  template <int I> typename Arg<I>::ReferredType *arg () {"
        << endl;
    fct <<  "    return (typename Arg<I>::ReferredType*)arg (I);" << endl;
    fct <<  "  }" << endl;
  }

  if (result() || useAction () || (proceed () && proceed_needs_result (*loc))) {
    if (!useAction ())
      data << "  Result *_result;" << endl;
    fct <<  "  inline Result *result() {return (Result*)_result;}" << endl;
  }

  if (target() || useAction () || (proceed () && proceed_needs_target (*loc))) {
    if (!useAction ())
      data << "  Target *_target;" << endl;
    fct <<  "  inline Target *target() {return (Target*)_target;}" << endl;
  }

  if (that() || useAction () || (proceed () && proceed_needs_that (*loc))) {
    if (!useAction ())
      data << "  That *_that;" << endl;
    fct <<  "  inline That *that() {return (That*)_that;}" << endl;
  }

  if (wrapper() || useAction ()) {
    if (!useAction ())
      data << "  void (*_wrapper)(AC::Action &);" << endl;
    fct <<  "  inline void (*wrapper ())(AC::Action &) {return _wrapper;}" << endl;
  }

  // terminate all the strings
  types << endl;
  data << endl;
  fct << endl;
  
  // add all types, attributes, and functions (on level 0)
  if (depth == 0) {
    code << types.str ().data ();
    code << data.str ().data ();
    code << fct.str ().data ();
  }
  
  // the closing bracket is *NOT* generated here -> for external extensions
}

ACM_Function *ThisJoinPoint::target_func (ACM_Code *loc) {
  ACM_Function *result = 0;
  if (loc->type_val() == JPT_Call)
    result = ((ACM_Call*)loc)->get_target();
  else if (loc->type_val() & (JPT_Execution|JPT_Construction|JPT_Destruction))
    result = (ACM_Function*)loc->get_parent();
  return result;
}

ACM_Function *ThisJoinPoint::that_func (ACM_Code *loc) {
  return (((ACM_Any*)loc->get_parent ())->type_val() == JPT_Function) ?
      (ACM_Function*)loc->get_parent () : (ACM_Function*)0;
}

void ThisJoinPoint::gen_tjp_init (ostream &code, ACM_Code *loc,
                                  BackEndProblems &bep, int depth, bool is_dep, vector<string> *arg_names) const {
  const ThisJoinPoint &tjp = *this; // TODO: not necessary!
  JoinPointType jptype = loc->type_val ();
  
  if (tjp.pointer_needed ()) {    

    code << "  __TJP ";
    Naming::tjp_instance(code, loc);
    code << ";" << endl;

    int args = get_arg_count(*loc);
    if (arg_needed (loc)) {
      if (args) {
      	if (tjp.useAction ()) {
	        code << "  void *";
  	      Naming::tjp_args_array(code, loc);
    	    code << "[] = { ";
      	  for (int i = 0; i < args; i++) {
        	  if (i > 0) code << ", ";
          	code << "(void*)&";
          	if (arg_names)
          	  code << (*arg_names)[i];
          	else
          	  code << "arg" << i;
        	}
        	code << " };" << endl;
      	}
      }

	    if (tjp.useAction ()) {
	      code << "  ";
  	    Naming::tjp_instance(code, loc);
    	  code << "._args = ";
      	if (args)
        	Naming::tjp_args_array(code, loc);
      	else
        	code << "0";
      	code << ";" << endl;
			}
			else {
    	  for (int i = 0; i < args; i++) {
		      code << "  ";
  		    Naming::tjp_instance(code, loc);
    		  code << "._args[" << i << "] = (void*)&";
          if (arg_names)
            code << (*arg_names)[i];
          else
            code << "arg" << i;
          code << ";" << endl;
    	  }
			}
    }
    
    if (tjp.result() || tjp.useAction() ||
      (tjp.proceed () && proceed_needs_result (*loc))) {
        code << "  ";
      Naming::tjp_instance(code, loc);
      code << "._result = ";
      if (!has_result_type(*loc) || get_result_type(*loc)->get_signature() == "void") {
        code << "0";
      } else {
        if (tjp.useAction ())
          code << "(void*)";
        code << "&(" << (is_dep ? "typename " : "") << "__TJP::Result&)" << ResultBuffer::result_name();
      }
      code << ";" << endl;
    }
    
    if (tjp.target() || tjp.useAction() ||
        (tjp.proceed () && proceed_needs_target (*loc))) {
      code << "  ";
      Naming::tjp_instance(code, loc);
      code << "._target = ";
      if (target_func(loc) && needs_this (*target_func(loc))) {
        code << " (";
        if (tjp.useAction ())
          code << "void*)";
        else {
          code << (is_dep ? "typename " : "") << "__TJP::Target*)";
        }
        switch (jptype) {
        case JPT_Construction:
        case JPT_Destruction:
        case JPT_Execution:
          code << "this";
          break;
        case JPT_Call:
          code << "&dst";
            break;
          default:
            code << " 0";
    	  }
      } else {
        code << " 0";
      }
      code << ";" << endl;
    }

    if (tjp.that() || tjp.useAction() ||
        (tjp.proceed () && proceed_needs_that (*loc))) {
      code << "  ";
      Naming::tjp_instance(code, loc);
      code << "._that = ";
      if (that_func (loc) && needs_this (*that_func (loc))) {
        code << " (";
        if (tjp.useAction ())
          code << "void*)";
        else {
          code << (is_dep ? "typename " : "")<< "__TJP::That*)";
        }
        switch (jptype) {
        case JPT_Construction:
        case JPT_Destruction:
        case JPT_Execution:
          code << "this";
          break;
        case JPT_Call:
          code << "srcthis";
            break;
          default:
            code << " 0";
        }
      } else {
        code << " 0";
      }
      code << ";" << endl;
    }
    
    if (tjp.useAction ()) {
      code << "  ";
      Naming::tjp_instance(code, loc);
      code << "._fptr = ";
      code << "0";
      code << ";" << endl;
    }
  }

}

void ThisJoinPoint::merge_flags (ACM_CodePlan &plan) {
  if (plan.has_next_level())
    merge_flags(*plan.get_next_level());
  typedef ACM_Container<ACM_CodeAdvice, true> Container;
  Container &before = plan.get_before();
  for (Container::iterator i = before.begin(); i != before.end(); ++i)
    TI_CodeAdvice::of(*(*i))->get_advice_info()->addTJPFlags(*this);
  if (plan.has_around()) {
    TI_CodeAdvice::of(*plan.get_around())->get_advice_info()->addTJPFlags(*this);
    if (*TI_CodeAdvice::of(*plan.get_around())->get_condition())
      conditional();
  }
  Container &after = plan.get_after();
  for (Container::iterator i = after.begin(); i != after.end(); ++i)
    TI_CodeAdvice::of(*(*i))->get_advice_info()->addTJPFlags(*this);
}

bool ThisJoinPoint::arg_needed (ACM_Code *loc) const {
  return (arg() || useAction() || (proceed () &&
    proceed_needs_args (*loc)));
}

void ThisJoinPoint::dump (ostream &out) const {
  out << "ThisJoinPoint (";
  out << "signature=" << (signature () ? "true" : "false");
  out << ", filename=" << (filename () ? "true" : "false");
  out << ", line=" << (line () ? "true" : "false");
  out << ", args=" << (args () ? "true" : "false");
  out << ", arg=" << (arg() ? "true" : "false");
  out << ", argtype=" << (argtype() ? "true" : "false");
  out << ", type=" << (type() ? "true" : "false");
  out << ", id=" << (id() ? "true" : "false");
  out << ", resulttype=" << (resulttype() ? "true" : "false");
  out << ", that=" << (that() ? "true" : "false");
  out << ", target=" << (target() ? "true" : "false");
  out << ", result=" << (result() ? "true" : "false");
  out << ", jptype=" << (jptype() ? "true" : "false");
  out << ", action=" << (action() ? "true" : "false");
  out << ", proceed=" << (proceed() ? "true" : "false");
  out << ")" << endl;
}
