/* Driver template for the LEMON parser generator.
** Copyright 1991-1995 by D. Richard Hipp.
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
** 
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the
** Free Software Foundation, Inc., 59 Temple Place - Suite 330,
** Boston, MA  02111-1307, USA.
**
** Modified 1997 to make it suitable for use with makeheaders.
*/
/* First off, code is include which follows the "include" declaration
** in the input file. */
#include <stdio.h>
#line 27 "cpp/PreParser.lem"

#include "Puma/List.h"
#include "Puma/Unit.h"
#include "Puma/Token.h"
#include "Puma/StrCol.h"
#include "Puma/PreExpr.h"
#include "Puma/CScanner.h"
#include "Puma/PreMacro.h"
#include "Puma/CTokens.h"
#include "Puma/PreParser.h"
#include "Puma/PreTree.h"
#include "Puma/UnitManager.h"
#include "Puma/TokenStream.h"
#include "Puma/ErrorStream.h"
#include "Puma/PrePredicate.h"
#include "Puma/ImportHandler.h"
#include "Puma/PreLevelState.h"
#include "Puma/CCommentTokens.h"
#include "Puma/PreParserState.h"
#include "Puma/PreMacroManager.h"
#include "Puma/PreFileIncluder.h"
#include "Puma/PreTreeToken.h"
#include "Puma/PreMacroExpander.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/PreTreeNodes.h"
#include "Puma/PrePredicateManager.h"
#include <stdlib.h>
#include <sstream>
#include <string>
using namespace std;


using namespace Puma;

// Definition of the Lemon generated parser interface.
void *PreParserAlloc (void *(*mallocProc)(...));
void PreParser       (void *pParser, int TokenID, PreTree *TokenValue);
void PreParserFree   (void *pParser, void (*freeProc)(...));
void PreParserTrace  (FILE *TraceFILE, char *zTracePrompt);

namespace Puma {


static PreprocessorParser*  parser;           // Current preprocessor parser.
static PreParserState*      state;            // Current parser.
static PreLevelState*       level;            // Current level state.
static ErrorStream*         err;              // Current error stream.

// Build a string from a subtree of the preprocessor syntax tree.
char* buildString (const PreTree* node)
 {
    if (! node) return (char*) 0;

    char *result, *ptr;
    string str;

    // If subtree isn't empty concatenate all tokens to a single string.
    if (node->sons ()) {
        PreTokenListPart* list = (PreTokenListPart*) node->son (0);
        
        // Fill the buffer.
        for (unsigned int i = 0; i < (unsigned int) list->sons (); i++) {
            str += ((PreTreeToken*) list->son (i))->token ()->text ();
        }
    } 
    
    // Create the return string buffer.
    ptr = result = new char[str.size() + 5];

    // Copy the buffer into the return buffer but skip newlines.
    for (unsigned int i = 0; i < str.size(); i++)
        if (str[i] != '\n') 
            *ptr++ = str[i];
        else
          cout << "newline " << i << endl;
            
    // Finish return buffer.
    *ptr = '\0';

    return result;
 }


// Evaluate a special answer string.
char* evalAnswerString (char* string, bool mode, PreParserState& state)
 {
    if (! string) return (char*) 0;

    // Scan the token string and build an unit.
    Unit *unit = new Unit ();
    parser->cscanner ().allow_macro_ops ();
    parser->cscanner ().fill_unit (string, *unit);
    parser->cscanner ().allow_macro_ops (false);
    
    // The call to an #assert (#unassert) directive expects the 
    // form: #assert PREDICATENAME (ANSWERTEXT). This function 
    // evaluates the answer token text to get the real answer text.
    
    Token* token;
    
    // Search the right parenthesis.
    for (token = (Token*) unit->first (); token;
         token = (Token*) unit->next (token))
        if (token->is_core () && token->type () == TOK_OPEN_ROUND)
            break;
    
    // No right parenthesis found -> parse error.
    if (! token) {
        if (mode) {
            *(state.err) << state.currToken->location () << sev_error
                << "Missing token list in `#assert'" 
                << endMessage; 
            return (char*) 0;
        } else { 
            string = new char[1]; 
            *string = '\0'; 
            return string; 
        }
    }
    
    // Delete all tokens before the right parenthesis.
    unit->kill ((Token*) unit->first (), token);
    
    // Search the left parenthesis.
    for (token = (Token*) unit->first (); token;
         token = (Token*) unit->next (token))
        if (token->is_core () && token->type () == TOK_CLOSE_ROUND)
            break;
    
    // No left parenthesis found -> parse error.
    if (! token) {
        *(state.err) << state.currToken->location ()
                     << sev_error << "Unterminated token list in ";
        if (mode) *(state.err) << "`#assert'";
        else      *(state.err) << "`#unassert'";
        *(state.err) << endMessage; 
        return (char*)0;
    }
    
    // Delete all tokens after the left parenthesis.
    unit->kill (token, (Token*) unit->last ());

    // Get the text between the parenthesis.
    char* answer = StrCol::buildString (unit);
    delete unit;
    
    // An empty answer causes a parse error.
    if (StrCol::onlySpaces (answer)) {
        *(state.err) << state.currToken->location () 
                     << sev_error << "Empty token list in ";
        if (mode) *(state.err) << "`#assert'";
        else      *(state.err) << "`#unassert'";
        *(state.err) << endMessage; 
        delete[] answer;
        return (char*) 0;
    }
    
    // Return the real answer text.
    return answer;
 }


// Update the bodies of some predefined macros.
void updatePredefined ()
 {
    TokenStream*     scanner = parser->scanner ();
    PreMacroManager* mm      = parser->macroManager ();
    
    if (scanner->length ()) {
        state->checkPredefined (mm->getIncLevelMacro (), scanner, *mm);
        state->checkPredefined (mm->getLineMacro (), scanner, *mm);
        state->checkPredefined (mm->getDateMacro (), scanner, *mm);
        state->checkPredefined (mm->getTimeMacro (), scanner, *mm);
        state->checkPredefined (mm->getFileMacro (), scanner, *mm);
    }
 }


// Send a newline token to the parser.
void sendNLToken ()
 {
    Token* token = state->newline;
    token->location (state->lastToken->location ());
    state->lastType = TOK_PRE_NEWLINE;
    PreTreeToken* synToken = new PreTreeToken (token);
    PreParser (state->pParser, state->lastType, synToken);
    PreParser (state->pParser, TOK_PRE_DIRECTIVE_END, (PreTree*) 0);
 }


// Return true if we are at the end of a directive.
bool atEndOfDirective (PreprocessorParser::PreMode mode)
 {
    int type = state->lastType;
    if (mode != PreprocessorParser::DEFINING &&
        mode != PreprocessorParser::ASSERT &&
        mode != PreprocessorParser::DIRECTIVE &&
        mode != PreprocessorParser::MACRO)
      if (type == TOK_PRE_ELSE    || type == TOK_PRE_ENDIF   || 
          type == TOK_PRE_ERROR   || type == TOK_PRE_WARNING || 
          type == TOK_PRE_TOKEN_LIST || type == TOK_PRE_RPAREN     || 
          type == TOK_PRE_ID) 
          return true;
    return false;
 }


// Prevent a special kind of parse error.
void preventEOFError (PreprocessorParser::PreMode mode)
 {
    // If the parser is parsing a directive and the next token
    // isn't a preprocessor token like expected there may be an
    // EOF at end of the directive. But the parser can't recognize
    // EOFs within directives, it would cause a parse error. But
    // it should be allowed to finish a directive with an EOF too
    // and not only with a newline. So it will be send a newline
    // to the parser if such an EOF is suspected.
    if (! state->currToken->is_comment () && state->lastToken) 
        if (atEndOfDirective (mode)) 
            sendNLToken ();
 }


// Prevent a special kind of parse error.
void preventInclEOFError (PreprocessorParser::PreMode mode)
 {
    // If an included file ends with a preprocessor directive the parser 
    // would expect that after the #include (that has included the file) 
    // is a newline. But if there is no newline but a new preprocessor 
    // directive a parse error would occure. In such a case we simply give 
    // the parser the expected newline token to prevent the error.
    if (state->lastToken)
        if (atEndOfDirective (mode)) 
        {
            int type = state->currType;
            if (type == TOK_PRE_DEFINE  || type == TOK_PRE_UNDEF    ||
                type == TOK_PRE_ASSERT  || type == TOK_PRE_UNASSERT ||
                type == TOK_PRE_WARNING || type == TOK_PRE_ERROR    ||
                type == TOK_PRE_IFDEF   || type == TOK_PRE_IFNDEF   ||
                type == TOK_PRE_IF      || type == TOK_PRE_ELIF     ||
                type == TOK_PRE_ELSE    || type == TOK_PRE_ENDIF    ||
                type == TOK_PRE_INCLUDE || type == TOK_PRE_INCLUDE_NEXT)
                sendNLToken ();
        }
 }


// Prevent a special kind of parse error.
bool directiveAtEOF ()
 {
    // An EOF within a directive is suspected. It's the same problem
    // like above.
    if (state->lastToken)
        if (parser->pre_mode () != PreprocessorParser::INITIAL) 
        {
            state->currToken = state->lastToken;
            state->currType = state->lastType;
            sendNLToken ();
            if (((TokenStream*) parser->scanner ())->length ()) 
            {
                // The last token of the main input file belongs to an
                // `#include' directive and nothing but EOF is following.
                // And now it is one more file to parse so the parser has
                // to be called again.
                state->lastToken = state->newline;
                state->lastToken->location (state->currToken->location ());
                state->lastType = TOK_PRE_NEWLINE;
                return true;
            }
        }
    return false;
 }


// Constructor of the preprocessor parser class.
PreprocessorParser::PreprocessorParser (ErrorStream* err, UnitManager* unitManager,
    UnitManager* locals, ostream& out, int max_depth) : _scanner (*err)
 {
    //PreParserTrace(stdout,"PRE_TRACE: ");
    _parserState                  = new PreParserState ();
    _parserState->maxIncludeDepth = max_depth;
    _parserState->err             = err;
    _parserState->pParser         = PreParserAlloc ((void* (*)(...))malloc);
    _silentMode                   = false;
    _out                          = &out;
    _unitManager                  = unitManager;
    _locals                       = locals;
    _levelState                   = new PreLevelState ();
    _macroStack                   = new Array<PreMacro*> ();
    _prescanned                   = new set<Token*>;
    _predicateManager             = new PrePredicateManager ();
    _macroManager                 = new PreMacroManager (err);
    _fileIncluder                 = new PreFileIncluder (*this);
    _pre_mode                     = INITIAL;    
    _support_gnu                  = true;
    _importHandler                = 0;
 }


PreprocessorParser::~PreprocessorParser ()
 {
    // Free the lemon generated parser.
    PreParserFree (_parserState->pParser, (void (*)(...))::free);
    delete _macroManager;
    delete _fileIncluder;
    delete _levelState;
    delete _parserState;
    delete _predicateManager;
    delete _macroStack;
    delete _prescanned;
    if (_importHandler)
      delete[] _importHandler;
 }

 
// Constructor of the preprocessor parser class.
void PreprocessorParser::reset (ErrorStream* err, UnitManager* unitManager,
                        UnitManager* locals, ostream& out, int max_depth)
 {
    delete _macroManager;
    delete _fileIncluder;
    delete _levelState;
    delete _parserState;
    delete _predicateManager;
    delete _macroStack;
    delete _prescanned;
    if (_importHandler)
      delete[] _importHandler;

    _parserState                  = new PreParserState ();
    _silentMode                   = false;
    _out                          = &out;
    _unitManager                  = unitManager;
    _locals                          = locals;
    _parserState->maxIncludeDepth = max_depth;
    _parserState->err             = err;
    _parserState->pParser         = PreParserAlloc ((void* (*)(...))malloc);
    _macroManager                 = new PreMacroManager (err);
    _fileIncluder                 = new PreFileIncluder (*this);
    _levelState                   = new PreLevelState ();
    _macroStack                   = new Array<PreMacro*> ();
    _prescanned                   = new set<Token*>;
    _predicateManager             = new PrePredicateManager ();
    _pre_mode                     = INITIAL;    
    _support_gnu                  = true;
    _importHandler                = 0;
 }


// Map scanner token types to parser token types
int PreprocessorParser::map_token (Token *token, bool &parse) {
  // default:
  int type = token->type ();
  parse = true;

  if (token->is_comment ()) {
    parse = false;
    return type;
  }

  // If the scope changes to another file, the mapping mode
  // must be reset due to a missing NEWLINE at end of the
  // previous file (In this case the mapping mode wouldn't
  // change correctly).
  if (state->lastToken)
    if (token->location ().filename ().name () !=
        state->lastToken->location ().filename ().name ())
      _pre_mode = INITIAL;

  if (_pre_mode == INITIAL) {
    if (token->is_preprocessor ()) {
      switch (type) {
        case TOK_PRE_DEFINE: 
          _pre_mode = DEFINING; 
          break;
        case TOK_PRE_ASSERT:
        case TOK_PRE_UNASSERT: 
          _pre_mode = ASSERT; 
          break;
        case TOK_PRE_IF:
        case TOK_PRE_ELIF:
        case TOK_PRE_WARNING:
        case TOK_PRE_ERROR:
        case TOK_PRE_INCLUDE:
        case TOK_PRE_INCLUDE_NEXT:
        case TOK_ELSE:
        case TOK_PRE_ENDIF:
          _pre_mode = TOKENLIST;
          break;
        case TOK_PRE_IFDEF:
        case TOK_PRE_IFNDEF:
        case TOK_PRE_UNDEF:
          _pre_mode = DIRECTIVE;
          break;
      }
    } else {
      parse = false;
    }
  } else {
    if (token->is_whitespace ()) {
      const char *c = token->text ();
      for (; *c && *c != '\n'; c++);

      if (*c == '\n') {
        type = TOK_PRE_NEWLINE;
        _pre_mode = INITIAL;
      } else if (_pre_mode == TOKENLIST) {
        type = TOK_PRE_TOKEN_LIST;
      } else if (_pre_mode == MAYBEMACRO) {
        type = TOK_PRE_TOKEN_LIST;
        _pre_mode = TOKENLIST;
      } else {
        parse = false;
      }
    } else {
      bool is_id = token->is_identifier ();

      type = Token::ID_UNKNOWN;
      switch (_pre_mode) {
        case TOKENLIST:
          type = TOK_PRE_TOKEN_LIST;
            break;
        case ASSERT:
        case DIRECTIVE:
          if (! state->passOnToken) {
            type = TOK_PRE_TOKEN_LIST;
            _pre_mode = TOKENLIST;
          } else if (is_id) {
            type = TOK_PRE_ID;
            _pre_mode = TOKENLIST;
          }
          break;
        case DEFINING:
          if (! state->passOnToken) {
            type = TOK_PRE_TOKEN_LIST;
            _pre_mode = TOKENLIST;
          } else if (is_id) {
            type = TOK_PRE_ID;
            _pre_mode = MAYBEMACRO;
          }
          break;
        case MAYBEMACRO:
          if (token->is_open ()) {
            type = TOK_PRE_LPAREN;
            _pre_mode = MACRO;
          }
          break;
        case MACRO:
          if (is_id) {
              type = TOK_PRE_ID;
          } else if (token->is_comma ()) {
            type = TOK_PRE_COMMA;
          } else if (token->is_close ()) {
            type = TOK_PRE_RPAREN;
            _pre_mode = TOKENLIST;
          } else if (token->type () == TOK_ELLIPSIS) {
            if (supportGNU ()) {
              type = TOK_PRE_REST;
            }
          } 
          break;
        case INITIAL: // to avoid compiler warnings
          break;
      }
      if (type == Token::ID_UNKNOWN) {
        _pre_mode = INITIAL;
      }
    }
  }
  return type;
}
    

// Parse a single token.
Token* PreprocessorParser::parseToken () {
  // Set the parser as the current parser.
  parser = this;
  state  = _parserState;
  level  = _levelState;
  err    = state->err;

  PreMacro*        macro;
  PreTreeToken* synToken;
  PreMode mode;

  // Create the macro expander.
  PreMacroExpander expander (parser);

  Unit *comment_unit = 0;
  Token *result = 0, *sl_comment = 0;
  bool done = false;
  
  while (! done) {
    done = true; // default: run once

    // Scan the input files. Get the next token.
    while ((state->currToken = scanner ()->next ())) {
      int type = state->currToken->type ();
      bool parse;

      // Finish a macro expansion.
      if (((TokenStream*) scanner ())->length () < state->number) {
        state->number = 0;
        
        // handle generated single line comments
        if (! sl_comment && state->sl_token && 
            (state->sl_token->type () == TOK_CCOMMENT ||
             state->sl_token->type () == TOK_CCSINGLE)) {
          sl_comment = state->sl_token;
          comment_unit = new Unit;
        }
      }
          
      // handle generated single line comments
      if (sl_comment) {
        Token *token;
        
        // single line comment finished
        if (state->currToken->location ().line () > 
            sl_comment->location ().line ()) {
          // Append the current token.
          token = (Token*)state->currToken->duplicate ();
          token->macro_generated ();
          comment_unit->append (*token);
          sl_comment = 0;

          // Push the unit containing the comment tokens on token stack. 
          ((TokenStream*)scanner ())->push (comment_unit);
          // Add the unit to the unit manager.
          _unitManager->addUnit (comment_unit);
          // Remember how many included headers lay on the stack.
          if (! state->number)
            state->number = ((TokenStream*) scanner ())->length ();
        // copy current token and mark it as comment token
        } else {
          token = state->currToken;
          token = new Token (TOK_CCOMMENT, Token::comment_id, token->text ());
          token->location (sl_comment->location ());
          token->macro_generated ();
          comment_unit->append (*token);
        }
        
        // scan next token
        continue;
      }
      
      mode = parser->pre_mode ();

      // recognize preprocessor state
      if (! sl_comment)
        type = map_token (state->currToken, parse);

      state->sl_token = state->currToken;
      state->currType = type;

      // If the scanned token isn't a preprocessor token, or a macro
      // is getting expanded and therefore preprocessor tokens have to
      // be ignored, do not parse the token.
      if (! parse || state->number) {
        preventEOFError (mode);
        break;
      }

      // Allow an included file to end with a preprocessor directive
      // and not only with a newline.
      preventInclEOFError (mode);

      // Remember the current token.
      state->lastToken = state->currToken;
      state->lastType = state->currType;

      // Encapsulate the current token so it can be part of the
      // preprocessor syntax tree.
      synToken = new PreTreeToken (state->currToken);

      // Give it to the parser.
      PreParser (state->pParser, type, synToken);

      // After a newline was parsed the parser has to wait until the
      // next token was parsed to be able to decide whether the end
      // of a directive really is reached (to be able to reduce the
      // current rule). But we have to do our work now, for example
      // after an #include directive we can't wait for the next
      // directive to include the file, we have to include it now.
      // Therefore a very special token will be send to the parser
      // after parsing a newline token. The parser recognizes this
      // very special token and the body of the #include-rule will
      // be executed right now.
      if (type == TOK_PRE_NEWLINE)
        PreParser (state->pParser, TOK_PRE_DIRECTIVE_END, (PreTree*) 0);
    }

    // If the current token isn't NULL process it.
    if (state->currToken) {
      // Go on only if we aren't within a `false' conditional branch and
      // no #error directive was parsed.
      if (state->passOnToken && ! state->errorDirective) {
        // Check every C++ identifier for macro calls.
        if (state->currToken->is_identifier ()) 
          if ((macro = _macroManager->getMacro (state->currToken->dtext ()))) {
            
            // Check whether a special predefined macro was called
            // and update it if necessary.
            macro = state->checkPredefined (macro, scanner (), *_macroManager);

            // Now it must be checked whether the macro is allowed
            // to be expanded. If it is a `self-referential' macro
            // call it will stay untouched.
            if (state->checkToExpand (state->currToken,
              macro, scanner (), *_macroStack, *_prescanned))
              // Try to expand the macro. It also could be that
              // this isn't a macro call at all, if so ignore it.
              if (expander.expandMacro (macro, state->currToken)) { 
                // After expansion the expanded text is pushed
                // on the input stream. Now we must be sure to
                // not expand again this macro inside of the
                // expanded text (self-referential macro call!).

                // Remember the number of real input files on
                // stack (the input stream) to be able to know
                // when the expansion text is scanned and removed
                // from input stream.
                if (! state->number) {
                  state->number = ((TokenStream*) scanner ())->length ();
                  state->stackPos = 0;
//                  _prescanned->clear ();
                } else
                  state->stackPos = ((TokenStream*) scanner ())->
                                    length () - state->number;

                // Put expanded macro on macro stack to prevent
                // it for be expanded again. The macro stack will
                // be cleared after expansion text is scanned.
                _macroStack->get (state->stackPos) = macro;

                // Call this function recursively again to skip
                // the macro call and parse the expanded text.
                done = false; 
              }        
          } 
          // If no macro call return the token unchanged.
          result = state->currToken;
            
      // The current token has to be ignored, so call this function
      // recursively again to skip the current token.
      } else
        done = false;
    }

    // The current token is NULL. That means all input files were parsed.
    else {      
      // Ensure that this is the last call to the parser.
      done = true;
      
      // Handle directive at EOF (missing newline). Same problem as above.
      directiveAtEOF ();

      // Create an EOF token needed if the EOF is within a directive
      // and caused a parse error. So the parser can print a error
      // message that report this error.
      state->currToken = new Token (Token::ID_END_OF_FILE);

      // Send the Lemon generated parser the signal to finish the
      // parse process.
      PreParser (state->pParser, 0, (PreTree*) 0);

      delete state->currToken;

      // Return NULL to signal all other parsers to finish their
      // parse processes.
      result = (Token*) 0; // return (Token*) 0;
    }
  }
  
  // remove the current token from the black list of self-referential macros
  // in prescanned macro argument lists
  if (result && result->is_identifier ())
    _prescanned->erase (result);

  return result;
}


} // namespace Puma

#line 711 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
/* Next is all token values, in a form suitable for use by makeheaders.
** This section will be null unless lemon is run with the -m switch.
*/
/* 
** These constants (all generated automatically by the parser generator)
** specify the various kinds of tokens (terminals) that the parser
** understands. 
**
** Each symbol here is a terminal symbol in the grammar.
*/
/* Make sure the INTERFACE macro is defined.
*/
#ifndef INTERFACE
# define INTERFACE 1
#endif
/* The next thing included is series of defines which control
** various aspects of the generated parser.
**    YYCODETYPE         is the data type used for storing terminal
**                       and nonterminal numbers.  "unsigned char" is
**                       used if there are fewer than 250 terminals
**                       and nonterminals.  "int" is used otherwise.
**    YYNOCODE           is a number of type YYCODETYPE which corresponds
**                       to no legal terminal or nonterminal number.  This
**                       number is used to fill in empty slots of the hash 
**                       table.
**    YYACTIONTYPE       is the data type used for storing terminal
**                       and nonterminal numbers.  "unsigned char" is
**                       used if there are fewer than 250 rules and
**                       states combined.  "int" is used otherwise.
**    PreParserTOKENTYPE     is the data type used for minor tokens given 
**                       directly to the parser from the tokenizer.
**    YYMINORTYPE        is the data type used for all minor tokens.
**                       This is typically a union of many types, one of
**                       which is PreParserTOKENTYPE.  The entry in the union
**                       for base tokens is called "yy0".
**    YYSTACKDEPTH       is the maximum depth of the parser's stack.
**    PreParserARGDECL       is a declaration of a 3rd argument to the
**                       parser, or null if there is no extra argument.
**    PreParserKRARGDECL     A version of PreParserARGDECL for K&R C.
**    PreParserANSIARGDECL   A version of PreParserARGDECL for ANSI C.
**    YYNSTATE           the combined number of states.
**    YYNRULE            the number of rules in the grammar
**    YYERRORSYMBOL      is the code number of the error symbol.  If not
**                       defined, then do no error processing.
*/
/*  */
#define YYCODETYPE unsigned char
#define YYNOCODE 51
#define YYACTIONTYPE unsigned char
#define PreParserTOKENTYPE PreTree*
typedef union {
  PreParserTOKENTYPE yy0;
  int yy101;
} YYMINORTYPE;
#define YYSTACKDEPTH 1000
#define PreParserARGDECL
#define PreParserXARGDECL
#define PreParserANSIARGDECL
#define YYNSTATE 110
#define YYNRULE 56
#define YYERRORSYMBOL 35
#define YYERRSYMDT yy101
#define YY_NO_ACTION      (YYNSTATE+YYNRULE+2)
#define YY_ACCEPT_ACTION  (YYNSTATE+YYNRULE+1)
#define YY_ERROR_ACTION   (YYNSTATE+YYNRULE)
/* Next is the action table.  Each entry in this table contains
**
**  +  An integer which is the number representing the look-ahead
**     token
**
**  +  An integer indicating what action to take.  Number (N) between
**     0 and YYNSTATE-1 mean shift the look-ahead and go to state N.
**     Numbers between YYNSTATE and YYNSTATE+YYNRULE-1 mean reduce by
**     rule N-YYNSTATE.  Number YYNSTATE+YYNRULE means that a syntax
**     error has occurred.  Number YYNSTATE+YYNRULE+1 means the parser
**     accepts its input.
**
**  +  A pointer to the next entry with the same hash value.
**
** The action table is really a series of hash tables.  Each hash
** table contains a number of entries which is a power of two.  The
** "state" table (which follows) contains information about the starting
** point and size of each hash table.
*/
struct yyActionEntry {
  YYCODETYPE   lookahead;   /* The value of the look-ahead token */
  YYACTIONTYPE action;      /* Action to take for this look-ahead */
  struct yyActionEntry *next; /* Next look-ahead with the same hash, or NULL */
};
static struct yyActionEntry yyActionTable[] = {
/* State 0 */
  {  44, 167, &yyActionTable[   1] }, /*   preprocessor_program accept */
  {  28,   1, 0                    }, /*       directive_groups shift  1 */
/* State 1 */
  {   0, 110, 0                    }, /*                      $ reduce 0 */
  {   1,  62, 0                    }, /*         TOK_PRE_ASSERT shift  62 */
  {   3,  70, 0                    }, /*         TOK_PRE_DEFINE shift  70 */
  {  35,   3, &yyActionTable[   4] }, /*                  error shift  3 */
  {  36,  56, 0                    }, /*             error_line shift  56 */
  {   8,  98, 0                    }, /*          TOK_PRE_ERROR shift  98 */
  {  38,   9, 0                    }, /*                if_line shift  9 */
  {  39,   6, 0                    }, /*                if_part shift  6 */
  {  40,  11, &yyActionTable[   7] }, /*             ifdef_line shift  11 */
  {  41,  13, 0                    }, /*            ifndef_line shift  13 */
  {  42,  42, &yyActionTable[  20] }, /*           include_line shift  42 */
  {  43,  44, &yyActionTable[  21] }, /*      include_next_line shift  44 */
  {  12,  32, 0                    }, /*         TOK_PRE_IFNDEF shift  32 */
  {  13,  58, 0                    }, /*        TOK_PRE_INCLUDE shift  58 */
  {  14,  60, 0                    }, /*   TOK_PRE_INCLUDE_NEXT shift  60 */
  {  47,  48, 0                    }, /*          unassert_line shift  48 */
  {  48,  52, 0                    }, /*             undef_line shift  52 */
  {  49,  54, 0                    }, /*           warning_line shift  54 */
  {  10,  21, 0                    }, /*             TOK_PRE_IF shift  21 */
  {  11,  28, 0                    }, /*          TOK_PRE_IFDEF shift  28 */
  {  20,  66, 0                    }, /*       TOK_PRE_UNASSERT shift  66 */
  {  21,  92, 0                    }, /*          TOK_PRE_UNDEF shift  92 */
  {  22,  96, 0                    }, /*        TOK_PRE_WARNING shift  96 */
  {  23,  46, 0                    }, /*            assert_line shift  46 */
  {  24,   5, 0                    }, /*      conditional_group shift  5 */
  {  25,   4, 0                    }, /*           control_line shift  4 */
  {  26,  50, 0                    }, /*            define_line shift  50 */
  {  27,   2, 0                    }, /*        directive_group shift  2 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
/* State 2 */
  {YYNOCODE,0,0}, /* Unused */
/* State 3 */
  {YYNOCODE,0,0}, /* Unused */
/* State 4 */
  {YYNOCODE,0,0}, /* Unused */
/* State 5 */
  {YYNOCODE,0,0}, /* Unused */
/* State 6 */
  {  32, 107, &yyActionTable[  39] }, /*              else_part shift  107 */
  {  28,   7, 0                    }, /*       directive_groups shift  7 */
  {  30, 100, 0                    }, /*              elif_part shift  100 */
  {YYNOCODE,0,0}, /* Unused */
/* State 7 */
  {YYNOCODE,0,0}, /* Unused */
  {   1,  62, 0                    }, /*         TOK_PRE_ASSERT shift  62 */
  {YYNOCODE,0,0}, /* Unused */
  {   3,  70, 0                    }, /*         TOK_PRE_DEFINE shift  70 */
  {YYNOCODE,0,0}, /* Unused */
  {   5,  36, 0                    }, /*           TOK_PRE_ELIF shift  36 */
  {   6,  38, 0                    }, /*           TOK_PRE_ELSE shift  38 */
  {   7,  40, 0                    }, /*          TOK_PRE_ENDIF shift  40 */
  {   8,  98, 0                    }, /*          TOK_PRE_ERROR shift  98 */
  {YYNOCODE,0,0}, /* Unused */
  {  10,  21, 0                    }, /*             TOK_PRE_IF shift  21 */
  {  11,  28, 0                    }, /*          TOK_PRE_IFDEF shift  28 */
  {  12,  32, 0                    }, /*         TOK_PRE_IFNDEF shift  32 */
  {  13,  58, 0                    }, /*        TOK_PRE_INCLUDE shift  58 */
  {  14,  60, 0                    }, /*   TOK_PRE_INCLUDE_NEXT shift  60 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {  20,  66, 0                    }, /*       TOK_PRE_UNASSERT shift  66 */
  {  21,  92, 0                    }, /*          TOK_PRE_UNDEF shift  92 */
  {  22,  96, 0                    }, /*        TOK_PRE_WARNING shift  96 */
  {  23,  46, 0                    }, /*            assert_line shift  46 */
  {  24,   5, 0                    }, /*      conditional_group shift  5 */
  {  25,   4, 0                    }, /*           control_line shift  4 */
  {  26,  50, 0                    }, /*            define_line shift  50 */
  {  27,   2, 0                    }, /*        directive_group shift  2 */
  {YYNOCODE,0,0}, /* Unused */
  {  29,  17, 0                    }, /*              elif_line shift  17 */
  {YYNOCODE,0,0}, /* Unused */
  {  31,  15, 0                    }, /*              else_line shift  15 */
  {YYNOCODE,0,0}, /* Unused */
  {  33,  19, 0                    }, /*             endif_line shift  19 */
  {  34,   8, 0                    }, /*             endif_part shift  8 */
  {  35,   3, 0                    }, /*                  error shift  3 */
  {  36,  56, 0                    }, /*             error_line shift  56 */
  {YYNOCODE,0,0}, /* Unused */
  {  38,   9, 0                    }, /*                if_line shift  9 */
  {  39,   6, 0                    }, /*                if_part shift  6 */
  {  40,  11, 0                    }, /*             ifdef_line shift  11 */
  {  41,  13, 0                    }, /*            ifndef_line shift  13 */
  {  42,  42, 0                    }, /*           include_line shift  42 */
  {  43,  44, 0                    }, /*      include_next_line shift  44 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {  47,  48, 0                    }, /*          unassert_line shift  48 */
  {  48,  52, 0                    }, /*             undef_line shift  52 */
  {  49,  54, 0                    }, /*           warning_line shift  54 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
/* State 8 */
  {YYNOCODE,0,0}, /* Unused */
/* State 9 */
  {   4,  10, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  10 */
/* State 10 */
  {YYNOCODE,0,0}, /* Unused */
/* State 11 */
  {   4,  12, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  12 */
/* State 12 */
  {YYNOCODE,0,0}, /* Unused */
/* State 13 */
  {   4,  14, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  14 */
/* State 14 */
  {YYNOCODE,0,0}, /* Unused */
/* State 15 */
  {   4,  16, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  16 */
/* State 16 */
  {YYNOCODE,0,0}, /* Unused */
/* State 17 */
  {   4,  18, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  18 */
/* State 18 */
  {YYNOCODE,0,0}, /* Unused */
/* State 19 */
  {   4,  20, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  20 */
/* State 20 */
  {YYNOCODE,0,0}, /* Unused */
/* State 21 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  22, 0                    }, /*             token_list shift  22 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 22 */
  {   4, 127, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 17 */
/* State 23 */
  {   4, 162, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 52 */
/* State 24 */
  {  16,  25, 0                    }, /*        TOK_PRE_NEWLINE shift  25 */
  {  19,  26, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  26 */
/* State 25 */
  {   4, 163, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 53 */
/* State 26 */
  {YYNOCODE,0,0}, /* Unused */
/* State 27 */
  {YYNOCODE,0,0}, /* Unused */
/* State 28 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  30, 0                    }, /*             TOK_PRE_ID shift  30 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  29, 0                    }, /*             token_list shift  29 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 29 */
  {   4, 128, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 18 */
/* State 30 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  31, 0                    }, /*             token_list shift  31 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 31 */
  {   4, 129, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 19 */
/* State 32 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  34, 0                    }, /*             TOK_PRE_ID shift  34 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  33, 0                    }, /*             token_list shift  33 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 33 */
  {   4, 130, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 20 */
/* State 34 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  35, 0                    }, /*             token_list shift  35 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 35 */
  {   4, 131, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 21 */
/* State 36 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  37, 0                    }, /*             token_list shift  37 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 37 */
  {   4, 132, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 22 */
/* State 38 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  39, 0                    }, /*             token_list shift  39 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 39 */
  {   4, 133, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 23 */
/* State 40 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  41, 0                    }, /*             token_list shift  41 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 41 */
  {   4, 134, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 24 */
/* State 42 */
  {   4,  43, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  43 */
/* State 43 */
  {YYNOCODE,0,0}, /* Unused */
/* State 44 */
  {   4,  45, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  45 */
/* State 45 */
  {YYNOCODE,0,0}, /* Unused */
/* State 46 */
  {   4,  47, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  47 */
/* State 47 */
  {YYNOCODE,0,0}, /* Unused */
/* State 48 */
  {   4,  49, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  49 */
/* State 49 */
  {YYNOCODE,0,0}, /* Unused */
/* State 50 */
  {   4,  51, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  51 */
/* State 51 */
  {YYNOCODE,0,0}, /* Unused */
/* State 52 */
  {   4,  53, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  53 */
/* State 53 */
  {YYNOCODE,0,0}, /* Unused */
/* State 54 */
  {   4,  55, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  55 */
/* State 55 */
  {YYNOCODE,0,0}, /* Unused */
/* State 56 */
  {   4,  57, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  57 */
/* State 57 */
  {YYNOCODE,0,0}, /* Unused */
/* State 58 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  59, 0                    }, /*             token_list shift  59 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 59 */
  {   4, 143, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 33 */
/* State 60 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  61, 0                    }, /*             token_list shift  61 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 61 */
  {   4, 144, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 34 */
/* State 62 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  64, 0                    }, /*             TOK_PRE_ID shift  64 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  63, 0                    }, /*             token_list shift  63 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 63 */
  {   4, 145, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 35 */
/* State 64 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  65, 0                    }, /*             token_list shift  65 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 65 */
  {   4, 146, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 36 */
/* State 66 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  68, 0                    }, /*             TOK_PRE_ID shift  68 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  67, 0                    }, /*             token_list shift  67 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 67 */
  {   4, 147, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 37 */
/* State 68 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  69, 0                    }, /*             token_list shift  69 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 69 */
  {   4, 148, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 38 */
/* State 70 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  72, 0                    }, /*             TOK_PRE_ID shift  72 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  71, 0                    }, /*             token_list shift  71 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 71 */
  {   4, 149, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 39 */
/* State 72 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  91, 0                    }, /*             token_list shift  91 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  15,  73, 0                    }, /*         TOK_PRE_LPAREN shift  73 */
/* State 73 */
  {  17,  76, &yyActionTable[ 247] }, /*           TOK_PRE_REST shift  76 */
  {  37,  79, &yyActionTable[ 244] }, /*        identifier_list shift  79 */
  {  18,  74, 0                    }, /*         TOK_PRE_RPAREN shift  74 */
  {   9,  90, 0                    }, /*             TOK_PRE_ID shift  90 */
/* State 74 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  75, 0                    }, /*             token_list shift  75 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 75 */
  {   4, 150, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 40 */
/* State 76 */
  {  18,  77, 0                    }, /*         TOK_PRE_RPAREN shift  77 */
/* State 77 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  78, 0                    }, /*             token_list shift  78 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 78 */
  {   4, 151, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 41 */
/* State 79 */
  {   2,  85, 0                    }, /*          TOK_PRE_COMMA shift  85 */
  {  17,  82, 0                    }, /*           TOK_PRE_REST shift  82 */
  {  18,  80, &yyActionTable[ 259] }, /*         TOK_PRE_RPAREN shift  80 */
  {YYNOCODE,0,0}, /* Unused */
/* State 80 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  81, 0                    }, /*             token_list shift  81 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 81 */
  {   4, 152, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 42 */
/* State 82 */
  {  18,  83, 0                    }, /*         TOK_PRE_RPAREN shift  83 */
/* State 83 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  84, 0                    }, /*             token_list shift  84 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 84 */
  {   4, 153, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 43 */
/* State 85 */
  {   9,  89, 0                    }, /*             TOK_PRE_ID shift  89 */
  {  17,  86, &yyActionTable[ 274] }, /*           TOK_PRE_REST shift  86 */
/* State 86 */
  {  18,  87, 0                    }, /*         TOK_PRE_RPAREN shift  87 */
/* State 87 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  88, 0                    }, /*             token_list shift  88 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 88 */
  {   4, 154, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 44 */
/* State 89 */
  {YYNOCODE,0,0}, /* Unused */
/* State 90 */
  {YYNOCODE,0,0}, /* Unused */
/* State 91 */
  {   4, 155, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 45 */
/* State 92 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {   9,  94, 0                    }, /*             TOK_PRE_ID shift  94 */
  {YYNOCODE,0,0}, /* Unused */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
  {YYNOCODE,0,0}, /* Unused */
  {  45,  93, 0                    }, /*             token_list shift  93 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {YYNOCODE,0,0}, /* Unused */
/* State 93 */
  {   4, 156, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 46 */
/* State 94 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  95, 0                    }, /*             token_list shift  95 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 95 */
  {   4, 157, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 47 */
/* State 96 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  97, 0                    }, /*             token_list shift  97 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 97 */
  {   4, 158, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 48 */
/* State 98 */
  {  16,  23, 0                    }, /*        TOK_PRE_NEWLINE shift  23 */
  {  45,  99, 0                    }, /*             token_list shift  99 */
  {  46,  24, 0                    }, /*        token_list_part shift  24 */
  {  19,  27, 0                    }, /*     TOK_PRE_TOKEN_LIST shift  27 */
/* State 99 */
  {   4, 159, 0                    }, /*  TOK_PRE_DIRECTIVE_END reduce 49 */
/* State 100 */
  {  28, 101, 0                    }, /*       directive_groups shift  101 */
/* State 101 */
  {YYNOCODE,0,0}, /* Unused */
  {   1,  62, 0                    }, /*         TOK_PRE_ASSERT shift  62 */
  {YYNOCODE,0,0}, /* Unused */
  {   3,  70, 0                    }, /*         TOK_PRE_DEFINE shift  70 */
  {YYNOCODE,0,0}, /* Unused */
  {   5,  36, 0                    }, /*           TOK_PRE_ELIF shift  36 */
  {   6,  38, 0                    }, /*           TOK_PRE_ELSE shift  38 */
  {   7,  40, 0                    }, /*          TOK_PRE_ENDIF shift  40 */
  {   8,  98, 0                    }, /*          TOK_PRE_ERROR shift  98 */
  {YYNOCODE,0,0}, /* Unused */
  {  10,  21, 0                    }, /*             TOK_PRE_IF shift  21 */
  {  11,  28, 0                    }, /*          TOK_PRE_IFDEF shift  28 */
  {  12,  32, 0                    }, /*         TOK_PRE_IFNDEF shift  32 */
  {  13,  58, 0                    }, /*        TOK_PRE_INCLUDE shift  58 */
  {  14,  60, 0                    }, /*   TOK_PRE_INCLUDE_NEXT shift  60 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {  20,  66, 0                    }, /*       TOK_PRE_UNASSERT shift  66 */
  {  21,  92, 0                    }, /*          TOK_PRE_UNDEF shift  92 */
  {  22,  96, 0                    }, /*        TOK_PRE_WARNING shift  96 */
  {  23,  46, 0                    }, /*            assert_line shift  46 */
  {  24,   5, 0                    }, /*      conditional_group shift  5 */
  {  25,   4, 0                    }, /*           control_line shift  4 */
  {  26,  50, 0                    }, /*            define_line shift  50 */
  {  27,   2, 0                    }, /*        directive_group shift  2 */
  {YYNOCODE,0,0}, /* Unused */
  {  29, 105, 0                    }, /*              elif_line shift  105 */
  {YYNOCODE,0,0}, /* Unused */
  {  31, 103, 0                    }, /*              else_line shift  103 */
  {YYNOCODE,0,0}, /* Unused */
  {  33,  19, 0                    }, /*             endif_line shift  19 */
  {  34, 102, 0                    }, /*             endif_part shift  102 */
  {  35,   3, 0                    }, /*                  error shift  3 */
  {  36,  56, 0                    }, /*             error_line shift  56 */
  {YYNOCODE,0,0}, /* Unused */
  {  38,   9, 0                    }, /*                if_line shift  9 */
  {  39,   6, 0                    }, /*                if_part shift  6 */
  {  40,  11, 0                    }, /*             ifdef_line shift  11 */
  {  41,  13, 0                    }, /*            ifndef_line shift  13 */
  {  42,  42, 0                    }, /*           include_line shift  42 */
  {  43,  44, 0                    }, /*      include_next_line shift  44 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {  47,  48, 0                    }, /*          unassert_line shift  48 */
  {  48,  52, 0                    }, /*             undef_line shift  52 */
  {  49,  54, 0                    }, /*           warning_line shift  54 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
/* State 102 */
  {YYNOCODE,0,0}, /* Unused */
/* State 103 */
  {   4, 104, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  104 */
/* State 104 */
  {YYNOCODE,0,0}, /* Unused */
/* State 105 */
  {   4, 106, 0                    }, /*  TOK_PRE_DIRECTIVE_END shift  106 */
/* State 106 */
  {YYNOCODE,0,0}, /* Unused */
/* State 107 */
  {  28, 108, 0                    }, /*       directive_groups shift  108 */
/* State 108 */
  {   1,  62, 0                    }, /*         TOK_PRE_ASSERT shift  62 */
  {  33,  19, &yyActionTable[ 380] }, /*             endif_line shift  19 */
  {  34, 109, 0                    }, /*             endif_part shift  109 */
  {  35,   3, &yyActionTable[ 385] }, /*                  error shift  3 */
  {  36,  56, 0                    }, /*             error_line shift  56 */
  {   3,  70, 0                    }, /*         TOK_PRE_DEFINE shift  70 */
  {  38,   9, 0                    }, /*                if_line shift  9 */
  {  39,   6, &yyActionTable[ 398] }, /*                if_part shift  6 */
  {  40,  11, &yyActionTable[ 399] }, /*             ifdef_line shift  11 */
  {  41,  13, 0                    }, /*            ifndef_line shift  13 */
  {  42,  42, &yyActionTable[ 408] }, /*           include_line shift  42 */
  {  43,  44, &yyActionTable[ 409] }, /*      include_next_line shift  44 */
  {  12,  32, 0                    }, /*         TOK_PRE_IFNDEF shift  32 */
  {  13,  58, 0                    }, /*        TOK_PRE_INCLUDE shift  58 */
  {  14,  60, 0                    }, /*   TOK_PRE_INCLUDE_NEXT shift  60 */
  {  47,  48, 0                    }, /*          unassert_line shift  48 */
  {  48,  52, 0                    }, /*             undef_line shift  52 */
  {  49,  54, 0                    }, /*           warning_line shift  54 */
  {   7,  40, 0                    }, /*          TOK_PRE_ENDIF shift  40 */
  {   8,  98, 0                    }, /*          TOK_PRE_ERROR shift  98 */
  {  20,  66, 0                    }, /*       TOK_PRE_UNASSERT shift  66 */
  {  21,  92, 0                    }, /*          TOK_PRE_UNDEF shift  92 */
  {  22,  96, 0                    }, /*        TOK_PRE_WARNING shift  96 */
  {  23,  46, 0                    }, /*            assert_line shift  46 */
  {  24,   5, 0                    }, /*      conditional_group shift  5 */
  {  25,   4, 0                    }, /*           control_line shift  4 */
  {  26,  50, 0                    }, /*            define_line shift  50 */
  {  27,   2, 0                    }, /*        directive_group shift  2 */
  {  10,  21, 0                    }, /*             TOK_PRE_IF shift  21 */
  {  11,  28, 0                    }, /*          TOK_PRE_IFDEF shift  28 */
  {YYNOCODE,0,0}, /* Unused */
  {YYNOCODE,0,0}, /* Unused */
/* State 109 */
  {YYNOCODE,0,0}, /* Unused */
};

/* The state table contains information needed to look up the correct
** action in the action table, given the current state of the parser.
** Information needed includes:
**
**  +  A pointer to the start of the action hash table in yyActionTable.
**
**  +  A mask used to hash the look-ahead token.  The mask is an integer
**     which is one less than the size of the hash table.  
**
**  +  The default action.  This is the action to take if no entry for
**     the given look-ahead is found in the action hash table.
*/
struct yyStateEntry {
  struct yyActionEntry *hashtbl; /* Start of the hash table in yyActionTable */
  int mask;                      /* Mask used for hashing the look-ahead */
  YYACTIONTYPE actionDefault;    /* Default action if look-ahead not found */
};
static struct yyStateEntry yyStateTable[] = {
  { &yyActionTable[0], 1, 111},
  { &yyActionTable[2], 31, 166},
  { &yyActionTable[34], 0, 112},
  { &yyActionTable[35], 0, 113},
  { &yyActionTable[36], 0, 114},
  { &yyActionTable[37], 0, 115},
  { &yyActionTable[38], 3, 111},
  { &yyActionTable[42], 63, 166},
  { &yyActionTable[106], 0, 116},
  { &yyActionTable[107], 0, 166},
  { &yyActionTable[108], 0, 119},
  { &yyActionTable[109], 0, 166},
  { &yyActionTable[110], 0, 120},
  { &yyActionTable[111], 0, 166},
  { &yyActionTable[112], 0, 121},
  { &yyActionTable[113], 0, 166},
  { &yyActionTable[114], 0, 122},
  { &yyActionTable[115], 0, 166},
  { &yyActionTable[116], 0, 124},
  { &yyActionTable[117], 0, 166},
  { &yyActionTable[118], 0, 126},
  { &yyActionTable[119], 3, 166},
  { &yyActionTable[123], 0, 166},
  { &yyActionTable[124], 0, 166},
  { &yyActionTable[125], 1, 166},
  { &yyActionTable[127], 0, 166},
  { &yyActionTable[128], 0, 165},
  { &yyActionTable[129], 0, 164},
  { &yyActionTable[130], 7, 166},
  { &yyActionTable[138], 0, 166},
  { &yyActionTable[139], 3, 166},
  { &yyActionTable[143], 0, 166},
  { &yyActionTable[144], 7, 166},
  { &yyActionTable[152], 0, 166},
  { &yyActionTable[153], 3, 166},
  { &yyActionTable[157], 0, 166},
  { &yyActionTable[158], 3, 166},
  { &yyActionTable[162], 0, 166},
  { &yyActionTable[163], 3, 166},
  { &yyActionTable[167], 0, 166},
  { &yyActionTable[168], 3, 166},
  { &yyActionTable[172], 0, 166},
  { &yyActionTable[173], 0, 166},
  { &yyActionTable[174], 0, 135},
  { &yyActionTable[175], 0, 166},
  { &yyActionTable[176], 0, 136},
  { &yyActionTable[177], 0, 166},
  { &yyActionTable[178], 0, 137},
  { &yyActionTable[179], 0, 166},
  { &yyActionTable[180], 0, 138},
  { &yyActionTable[181], 0, 166},
  { &yyActionTable[182], 0, 139},
  { &yyActionTable[183], 0, 166},
  { &yyActionTable[184], 0, 140},
  { &yyActionTable[185], 0, 166},
  { &yyActionTable[186], 0, 141},
  { &yyActionTable[187], 0, 166},
  { &yyActionTable[188], 0, 142},
  { &yyActionTable[189], 3, 166},
  { &yyActionTable[193], 0, 166},
  { &yyActionTable[194], 3, 166},
  { &yyActionTable[198], 0, 166},
  { &yyActionTable[199], 7, 166},
  { &yyActionTable[207], 0, 166},
  { &yyActionTable[208], 3, 166},
  { &yyActionTable[212], 0, 166},
  { &yyActionTable[213], 7, 166},
  { &yyActionTable[221], 0, 166},
  { &yyActionTable[222], 3, 166},
  { &yyActionTable[226], 0, 166},
  { &yyActionTable[227], 7, 166},
  { &yyActionTable[235], 0, 166},
  { &yyActionTable[236], 7, 166},
  { &yyActionTable[244], 3, 166},
  { &yyActionTable[248], 3, 166},
  { &yyActionTable[252], 0, 166},
  { &yyActionTable[253], 0, 166},
  { &yyActionTable[254], 3, 166},
  { &yyActionTable[258], 0, 166},
  { &yyActionTable[259], 3, 166},
  { &yyActionTable[263], 3, 166},
  { &yyActionTable[267], 0, 166},
  { &yyActionTable[268], 0, 166},
  { &yyActionTable[269], 3, 166},
  { &yyActionTable[273], 0, 166},
  { &yyActionTable[274], 1, 166},
  { &yyActionTable[276], 0, 166},
  { &yyActionTable[277], 3, 166},
  { &yyActionTable[281], 0, 166},
  { &yyActionTable[282], 0, 161},
  { &yyActionTable[283], 0, 160},
  { &yyActionTable[284], 0, 166},
  { &yyActionTable[285], 7, 166},
  { &yyActionTable[293], 0, 166},
  { &yyActionTable[294], 3, 166},
  { &yyActionTable[298], 0, 166},
  { &yyActionTable[299], 3, 166},
  { &yyActionTable[303], 0, 166},
  { &yyActionTable[304], 3, 166},
  { &yyActionTable[308], 0, 166},
  { &yyActionTable[309], 0, 111},
  { &yyActionTable[310], 63, 166},
  { &yyActionTable[374], 0, 117},
  { &yyActionTable[375], 0, 166},
  { &yyActionTable[376], 0, 123},
  { &yyActionTable[377], 0, 166},
  { &yyActionTable[378], 0, 125},
  { &yyActionTable[379], 0, 111},
  { &yyActionTable[380], 31, 166},
  { &yyActionTable[412], 0, 118},
};

/* The following structure represents a single element of the
** parser's stack.  Information stored includes:
**
**   +  The state number for the parser at this level of the stack.
**
**   +  The value of the token stored at this level of the stack.
**      (In other words, the "major" token.)
**
**   +  The semantic value stored at this level of the stack.  This is
**      the information used by the action routines in the grammar.
**      It is sometimes called the "minor" token.
*/
struct yyStackEntry {
  int stateno;       /* The state-number */
  int major;         /* The major token value.  This is the code
                     ** number for the token at this stack level */
  YYMINORTYPE minor; /* The user-supplied minor token value.  This
                     ** is the value of the token  */
};

/* The state of the parser is completely contained in an instance of
** the following structure */
struct yyParser {
  int idx;                            /* Index of top element in stack */
  int errcnt;                         /* Shifts left before out of the error */
  struct yyStackEntry *top;           /* Pointer to the top stack element */
  struct yyStackEntry stack[YYSTACKDEPTH];  /* The parser's stack */
};
typedef struct yyParser yyParser;

#ifndef NDEBUG
#include <stdio.h>
static FILE *yyTraceFILE = 0;
static char *yyTracePrompt = 0;

/* 
** Turn parser tracing on by giving a stream to which to write the trace
** and a prompt to preface each trace message.  Tracing is turned off
** by making either argument NULL 
**
** Inputs:
** <ul>
** <li> A FILE* to which trace output should be written.
**      If NULL, then tracing is turned off.
** <li> A prefix string written at the beginning of every
**      line of trace output.  If NULL, then tracing is
**      turned off.
** </ul>
**
** Outputs:
** None.
*/
void PreParserTrace(FILE *TraceFILE, char *zTracePrompt){
  yyTraceFILE = TraceFILE;
  yyTracePrompt = zTracePrompt;
  if( yyTraceFILE==0 ) yyTracePrompt = 0;
  else if( yyTracePrompt==0 ) yyTraceFILE = 0;
}

/* For tracing shifts, the names of all terminals and nonterminals
** are required.  The following table supplies these names */
static char *yyTokenName[] = { 
  "$",             "TOK_PRE_ASSERT",  "TOK_PRE_COMMA",  "TOK_PRE_DEFINE",
  "TOK_PRE_DIRECTIVE_END",  "TOK_PRE_ELIF",  "TOK_PRE_ELSE",  "TOK_PRE_ENDIF",
  "TOK_PRE_ERROR",  "TOK_PRE_ID",    "TOK_PRE_IF",    "TOK_PRE_IFDEF",
  "TOK_PRE_IFNDEF",  "TOK_PRE_INCLUDE",  "TOK_PRE_INCLUDE_NEXT",  "TOK_PRE_LPAREN",
  "TOK_PRE_NEWLINE",  "TOK_PRE_REST",  "TOK_PRE_RPAREN",  "TOK_PRE_TOKEN_LIST",
  "TOK_PRE_UNASSERT",  "TOK_PRE_UNDEF",  "TOK_PRE_WARNING",  "assert_line", 
  "conditional_group",  "control_line",  "define_line",   "directive_group",
  "directive_groups",  "elif_line",     "elif_part",     "else_line",   
  "else_part",     "endif_line",    "endif_part",    "error",       
  "error_line",    "identifier_list",  "if_line",       "if_part",     
  "ifdef_line",    "ifndef_line",   "include_line",  "include_next_line",
  "preprocessor_program",  "token_list",    "token_list_part",  "unassert_line",
  "undef_line",    "warning_line",
};
#define YYTRACE(X) if( yyTraceFILE ) fprintf(yyTraceFILE,"%sReduce [%s].\n",yyTracePrompt,X);
#else
#define YYTRACE(X)
#endif

/* 
** This function allocates a new parser.
** The only argument is a pointer to a function which works like
** malloc.
**
** Inputs:
** A pointer to the function used to allocate memory.
**
** Outputs:
** A pointer to a parser.  This pointer is used in subsequent calls
** to PreParser and PreParserFree.
*/
void *PreParserAlloc(void *(*mallocProc)(...)){
  yyParser *pParser;
  pParser = (yyParser*)(*mallocProc)( sizeof(yyParser), __FILE__, __LINE__ );
  if( pParser ){
    pParser->idx = -1;
  }
  return pParser;
}

/* The following function deletes the value associated with a
** symbol.  The symbol can be either a terminal or nonterminal.
** "yymajor" is the symbol code, and "yypminor" is a pointer to
** the value.
*/
static void yy_destructor(YYCODETYPE yymajor, YYMINORTYPE *yypminor){
  switch( yymajor ){
    /* Here is inserted the actions which take place when a
    ** terminal or non-terminal is destroyed.  This can happen
    ** when the symbol is popped from the stack during a
    ** reduce or during error processing or when a parser is 
    ** being destroyed before it is finished parsing.
    **
    ** Note: during a reduce, the only symbols destroyed are those
    ** which appear on the RHS of the rule, but which are not used
    ** inside the C code.
    */
    default:  break;   /* If no destructor action specified: do nothing */
  }
}

/*
** Pop the parser's stack once.
**
** If there is a destructor routine associated with the token which
** is popped from the stack, then call it.
**
** Return the major token number for the symbol popped.
*/
static int yy_pop_parser_stack(yyParser *pParser){
  YYCODETYPE yymajor;

  if( pParser->idx<0 ) return 0;
#ifndef NDEBUG
  if( yyTraceFILE && pParser->idx>=0 ){
    fprintf(yyTraceFILE,"%sPopping %s\n",
      yyTracePrompt,
      yyTokenName[pParser->top->major]);
  }
#endif
  yymajor = pParser->top->major;
  yy_destructor( yymajor, &pParser->top->minor);
  pParser->idx--;
  pParser->top--;
  return yymajor;
}

/* 
** Deallocate and destroy a parser.  Destructors are all called for
** all stack elements before shutting the parser down.
**
** Inputs:
** <ul>
** <li>  A pointer to the parser.  This should be a pointer
**       obtained from PreParserAlloc.
** <li>  A pointer to a function used to reclaim memory obtained
**       from malloc.
** </ul>
*/
void PreParserFree(
  void *p,               /* The parser to be deleted */
  void (*freeProc)(...)     /* Function used to reclaim memory */
){
  yyParser *pParser = (yyParser*)p;
  if( pParser==0 ) return;
  while( pParser->idx>=0 ) yy_pop_parser_stack(pParser);
  (*freeProc)(pParser, __FILE__, __LINE__);
}

/*
** Find the appropriate action for a parser given the look-ahead token.
**
** If the look-ahead token is YYNOCODE, then check to see if the action is
** independent of the look-ahead.  If it is, return the action, otherwise
** return YY_NO_ACTION.
*/
static int yy_find_parser_action(
  yyParser *pParser,        /* The parser */
  int iLookAhead             /* The look-ahead token */
){
  struct yyStateEntry *pState;   /* Appropriate entry in the state table */
  struct yyActionEntry *pAction; /* Action appropriate for the look-ahead */
 
  /* if( pParser->idx<0 ) return YY_NO_ACTION;  */
  pState = &yyStateTable[pParser->top->stateno];
  if( iLookAhead!=YYNOCODE ){
    pAction = &pState->hashtbl[iLookAhead & pState->mask];
    while( pAction ){
      if( pAction->lookahead==iLookAhead ) return pAction->action;
      pAction = pAction->next;
    }
  }else if( pState->mask!=0 || pState->hashtbl->lookahead!=YYNOCODE ){
    return YY_NO_ACTION;
  }
  return pState->actionDefault;
}

/*
** Perform a shift action.
*/
static void yy_shift(
  yyParser *yypParser,          /* The parser to be shifted */
  int yyNewState,               /* The new state to shift in */
  int yyMajor,                  /* The major token to shift in */
  YYMINORTYPE *yypMinor         /* Pointer ot the minor token to shift in */
){
  yypParser->idx++;
  yypParser->top++;
  if( yypParser->idx>=YYSTACKDEPTH ){
     yypParser->idx--;
     yypParser->top--;
#ifndef NDEBUG
     if( yyTraceFILE ){
       fprintf(yyTraceFILE,"%sStack Overflow!\n",yyTracePrompt);
     }
#endif
     while( yypParser->idx>=0 ) yy_pop_parser_stack(yypParser);
     /* Here code is inserted which will execute if the parser
     ** stack every overflows */
#line 715 "cpp/PreParser.lem"

  state->syntaxError = true;
  *err << state->lastToken->location () << sev_fatal 
       << "Giving up. Preprocessor stack overflow (>1000)" << endMessage;

#line 1683 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
     return;
  }
  yypParser->top->stateno = yyNewState;
  yypParser->top->major = yyMajor;
  yypParser->top->minor = *yypMinor;
#ifndef NDEBUG
  if( yyTraceFILE && yypParser->idx>0 ){
    int i;
    fprintf(yyTraceFILE,"%sShift %d\n",yyTracePrompt,yyNewState);
    fprintf(yyTraceFILE,"%sStack:",yyTracePrompt);
    for(i=1; i<=yypParser->idx; i++)
      fprintf(yyTraceFILE," %s",yyTokenName[yypParser->stack[i].major]);
    fprintf(yyTraceFILE,"\n");
  }
#endif
}

/* The following table contains information about every rule that
** is used during the reduce.
*/
static struct {
  YYCODETYPE lhs;         /* Symbol on the left-hand side of the rule */
  unsigned char nrhs;     /* Number of right-hand side symbols in the rule */
} yyRuleInfo[] = {
  { 44, 1 },
  { 28, 0 },
  { 28, 2 },
  { 28, 2 },
  { 27, 1 },
  { 27, 1 },
  { 24, 3 },
  { 24, 4 },
  { 24, 4 },
  { 39, 2 },
  { 39, 2 },
  { 39, 2 },
  { 32, 3 },
  { 32, 4 },
  { 30, 3 },
  { 30, 4 },
  { 34, 2 },
  { 38, 2 },
  { 40, 2 },
  { 40, 3 },
  { 41, 2 },
  { 41, 3 },
  { 29, 2 },
  { 31, 2 },
  { 33, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 25, 2 },
  { 42, 2 },
  { 43, 2 },
  { 23, 2 },
  { 23, 3 },
  { 47, 2 },
  { 47, 3 },
  { 26, 2 },
  { 26, 5 },
  { 26, 6 },
  { 26, 6 },
  { 26, 7 },
  { 26, 8 },
  { 26, 3 },
  { 48, 2 },
  { 48, 3 },
  { 49, 2 },
  { 36, 2 },
  { 37, 1 },
  { 37, 3 },
  { 45, 1 },
  { 45, 2 },
  { 46, 1 },
  { 46, 2 },
};

static void yy_accept(
  yyParser *yypParser           /* The parser */
  PreParserANSIARGDECL              /* Extra arguments (if any) */
);  /* Forward declaration */

/*
** Perform a reduce action and the shift that must immediately
** follow the reduce.
*/
static void yy_reduce(
  yyParser *yypParser,         /* The parser */
  int yyruleno                 /* Number of the rule by which to reduce */
  PreParserANSIARGDECL
){
  int yygoto;                     /* The next state */
  int yyact;                      /* The next action */
  YYMINORTYPE yygotominor;        /* The LHS of the rule reduced */
  struct yyStackEntry *yymsp;     /* The top of the parser's stack */
  int yysize;                     /* Amount to pop the stack */
  yymsp = yypParser->top;
  switch( yyruleno ){
  /* Beginning here are the reduction cases.  A typical example
  ** follows:
  **   case 0:
  **     YYTRACE("<text of the rule>");
  **  #line <lineno> <grammarfile>
  **     { ... }           // User supplied code
  **  #line <lineno> <thisfile>
  **     break;
  */
      case 0:
        YYTRACE("preprocessor_program ::= directive_groups")
#line 756 "cpp/PreParser.lem"
{
        state->syntaxTree = new PreProgram (yymsp[0].minor.yy0);
    }
#line 1802 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 1:
        YYTRACE("directive_groups ::=")
#line 765 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDirectiveGroups ();
    }
#line 1810 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 2:
        YYTRACE("directive_groups ::= directive_groups directive_group")
#line 769 "cpp/PreParser.lem"
{
        yymsp[-1].minor.yy0->add_son (yymsp[0].minor.yy0);
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1819 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 3:
        YYTRACE("directive_groups ::= directive_groups error")
#line 774 "cpp/PreParser.lem"
{
        yymsp[-1].minor.yy0->add_son (new PreError ());
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1828 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for error */
        break;
      case 4:
        YYTRACE("directive_group ::= control_line")
#line 784 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[0].minor.yy0;
    }
#line 1837 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 5:
        YYTRACE("directive_group ::= conditional_group")
#line 788 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[0].minor.yy0;
    }
#line 1845 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 6:
        YYTRACE("conditional_group ::= if_part directive_groups endif_part")
#line 797 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreConditionalGroup (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 1853 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 7:
        YYTRACE("conditional_group ::= if_part elif_part directive_groups endif_part")
#line 802 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreConditionalGroup (yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 1861 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 8:
        YYTRACE("conditional_group ::= if_part else_part directive_groups endif_part")
#line 807 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreConditionalGroup (yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 1869 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 9:
        YYTRACE("if_part ::= if_line TOK_PRE_DIRECTIVE_END")
#line 816 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1877 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 10:
        YYTRACE("if_part ::= ifdef_line TOK_PRE_DIRECTIVE_END")
#line 820 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1886 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 11:
        YYTRACE("if_part ::= ifndef_line TOK_PRE_DIRECTIVE_END")
#line 824 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1895 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 12:
        YYTRACE("else_part ::= directive_groups else_line TOK_PRE_DIRECTIVE_END")
#line 833 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreElsePart (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0);
    }
#line 1904 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 13:
        YYTRACE("else_part ::= elif_part directive_groups else_line TOK_PRE_DIRECTIVE_END")
#line 838 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreElsePart (yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0);
    }
#line 1913 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 14:
        YYTRACE("elif_part ::= directive_groups elif_line TOK_PRE_DIRECTIVE_END")
#line 847 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreElifPart ();
        ((PreElifPart*) (yygotominor.yy0))->addSons (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0);
    }
#line 1923 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 15:
        YYTRACE("elif_part ::= elif_part directive_groups elif_line TOK_PRE_DIRECTIVE_END")
#line 853 "cpp/PreParser.lem"
{
        ((PreElifPart*) (yymsp[-3].minor.yy0))->addSons (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0);
        yygotominor.yy0 = yymsp[-3].minor.yy0;
    }
#line 1933 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 16:
        YYTRACE("endif_part ::= endif_line TOK_PRE_DIRECTIVE_END")
#line 863 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 1942 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 17:
        YYTRACE("if_line ::= TOK_PRE_IF token_list")
#line 872 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIfDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A new conditional group starts, so go one level deeper.
        level->deeper ();

        if (! state->errorDirective)
            if (state->passOnToken) 
                {
                // Create a new macro expander to expand the macros that could
                // be part of the expression. 
                PreMacroExpander expander (parser);

                Location location = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location ();

                // Create a new expression evaluator.
                PreExpr expr (err, location);

                // Now add the special macro `defined' to the macro manager.
                PreTree* idl   = new PreIdentifierList (yymsp[-1].minor.yy0);
                PreMacro*   macro = new PreMacro ("defined", idl, "\n");
                delete idl;
                macro->location (location);
                parser->macroManager ()->addMacro (macro);
        
                // Update the predefined macros.
                updatePredefined ();
        
                // Build the expression string and then expand all macros within it.
                // Then we should have a string consisting only of 0s, 1s and some
                // operators like || or &&. Evaluate the string.
                char* str = buildString (yymsp[0].minor.yy0);
                char* exprstr = expander.expandMacros (str);
                expr.evaluatePreExpr (exprstr);
                    delete[] exprstr;
                delete[] str;

                // Remove the special macro `defined' from macro manager.
                parser->macroManager ()->removeMacro ("defined");

                // Add a daughter node containing the value of the expression.
                yygotominor.yy0->add_daughter (new PreCondSemNode (expr.value ()));

                // Now decide whether to go on or to ignore following C++ code.
                if (expr.value ()) level->setState (false);
                else state->passOnToken = false;
            } 
            else // Ignore further tokens.
            {
                    level->setPassOn (false);
                level->setState (false);
            } 
    }
#line 2001 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 18:
        YYTRACE("ifdef_line ::= TOK_PRE_IFDEF token_list")
#line 931 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIfdefDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A new conditional group starts, so go one level deeper.
        level->deeper ();

        // Ignore further tokens.
        if (! state->errorDirective)
        {
          level->setPassOn (false);
          level->setState (false);
        }
    }
#line 2019 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 19:
        YYTRACE("ifdef_line ::= TOK_PRE_IFDEF TOK_PRE_ID token_list")
#line 945 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIfdefDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A new conditional group starts, so go one level deeper.
        level->deeper ();
        
        if (! state->errorDirective)
            if (state->passOnToken) 
                {
                const char* name  = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->text ();
                PreMacro*   macro = parser->macroManager ()->getMacro (name);
        
                // Search the given macro name. If found the value of the 
                // conditional is true else it's false. Add a daughter node 
                // containing the value of the conditonal.
                if (macro)
                    yygotominor.yy0->add_daughter (new PreCondSemNode (true));
                else
                    yygotominor.yy0->add_daughter (new PreCondSemNode (false));
            
                // Now decide whether to go on or to ignore following C++ code.
                if (macro) level->setState (false);
                else state->passOnToken = false;
            } 
            else // Ignore further tokens.
            {
                    level->setPassOn (false);
                level->setState (false);
            } 
    }
#line 2054 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 20:
        YYTRACE("ifndef_line ::= TOK_PRE_IFNDEF token_list")
#line 981 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIfndefDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A new conditional group starts, so go one level deeper.
        level->deeper ();

        // Ignore further tokens.
        if (! state->errorDirective)
        {
          level->setPassOn (false);
          level->setState (false);
        }
    }
#line 2072 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 21:
        YYTRACE("ifndef_line ::= TOK_PRE_IFNDEF TOK_PRE_ID token_list")
#line 995 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIfndefDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A new conditional group starts, so go one level deeper.
        level->deeper ();
        
        if (! state->errorDirective)
            if (state->passOnToken) 
                {
                const char* name  = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->text ();
                PreMacro* macro = parser->macroManager ()->getMacro (name);
        
                // Search the given macro name. If found the value of the 
                // conditional is false else it's true. Add a daughter node 
                // containing the value of the conditonal.
                if (! macro)
                    yygotominor.yy0->add_daughter (new PreCondSemNode (true));
                else
                    yygotominor.yy0->add_daughter (new PreCondSemNode (false));
            
                // Now decide whether to go on or to ignore following C++ code.
                if (! macro) level->setState (false);
                else state->passOnToken = false;
            } 
            else // Ignore further tokens.
            {
                    level->setPassOn (false);
                level->setState (false);
            } 
    }
#line 2107 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 22:
        YYTRACE("elif_line ::= TOK_PRE_ELIF token_list")
#line 1031 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreElifDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective)
            if (state->passOnToken && ! level->state ())
                state->passOnToken = false;
            else if (! state->passOnToken && level->state ())
            {
                // Create a new macro expander to expand the macros that could
                // be part of the expression. 
                PreMacroExpander expander (parser);
        
                Location location = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location ();
        
                // Create a new expression evaluator.
                PreExpr expr (err, location);
        
                // Now add the special macro `defined' to the macro manager.
                PreTree* idl   = new PreIdentifierList (yymsp[-1].minor.yy0);
                PreMacro*   macro = new PreMacro ("defined", idl, "\n");
                delete idl;
                macro->location (location);
                parser->macroManager ()->addMacro (macro);
        
                // Update the predefined macros.
                updatePredefined ();
        
                // Build the expression string and then expand all macros within it.
                // Then we should have a string consisting only of 0s, 1s and some
                // operators like || or &&. Evaluate the string.
                char* str = buildString (yymsp[0].minor.yy0);
                char* exprstr = expander.expandMacros (str);
                expr.evaluatePreExpr (exprstr);
                    delete[] exprstr;
                delete[] str;
        
                // Remove the special macro `defined' from macro manager.
                parser->macroManager ()->removeMacro ("defined");
        
                // Add a daughter node containing the value of the expression.
                yygotominor.yy0->add_daughter (new PreCondSemNode (expr.value ()));
                
                // Now decide whether to go on or to ignore following C++ code.
                if (expr.value ()) 
                {
                    state->passOnToken = true;
                    level->setState (false);
                    } 
            }
    }
#line 2162 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 23:
        YYTRACE("else_line ::= TOK_PRE_ELSE token_list")
#line 1087 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreElseDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // Decide whether to go on or to ignore following C++ code.
        if (! state->errorDirective)
            if (state->passOnToken && ! level->state ()) 
                state->passOnToken = false;
            else if (! state->passOnToken && level->state ()) {
                state->passOnToken = true;
                level->setState (false);
            } 
    }
#line 2179 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 24:
        YYTRACE("endif_line ::= TOK_PRE_ENDIF token_list")
#line 1105 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreEndifDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // A conditional group ends. If no #error was parsed befor
        // go one level higher.
        
        // Decide whether to go on or to ignore following C++ code.
        if (! state->errorDirective) {
            if (level->passOn ())
                state->passOnToken = true;
            level->higher ();
        }
    }
#line 2197 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 25:
        YYTRACE("control_line ::= include_line TOK_PRE_DIRECTIVE_END")
#line 1124 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2205 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 26:
        YYTRACE("control_line ::= include_next_line TOK_PRE_DIRECTIVE_END")
#line 1128 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2214 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 27:
        YYTRACE("control_line ::= assert_line TOK_PRE_DIRECTIVE_END")
#line 1132 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2223 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 28:
        YYTRACE("control_line ::= unassert_line TOK_PRE_DIRECTIVE_END")
#line 1136 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2232 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 29:
        YYTRACE("control_line ::= define_line TOK_PRE_DIRECTIVE_END")
#line 1140 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2241 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 30:
        YYTRACE("control_line ::= undef_line TOK_PRE_DIRECTIVE_END")
#line 1144 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2250 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 31:
        YYTRACE("control_line ::= warning_line TOK_PRE_DIRECTIVE_END")
#line 1148 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2259 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 32:
        YYTRACE("control_line ::= error_line TOK_PRE_DIRECTIVE_END")
#line 1152 "cpp/PreParser.lem"
{
        yygotominor.yy0 = yymsp[-1].minor.yy0;
    }
#line 2268 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        /* No destructor defined for TOK_PRE_DIRECTIVE_END */
        break;
      case 33:
        YYTRACE("include_line ::= TOK_PRE_INCLUDE token_list")
#line 1161 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIncludeDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);

        if (! state->errorDirective && state->passOnToken) 
        { 
            // Calculate the number of files included.
            int num = state->number ? state->number - 2 : ((TokenStream*)
                      (parser->scanner ()))->length () - 1;

            if (((PreIncludeDirective*)yygotominor.yy0)->is_forced ())
            	state->forcedIncludes = true;
            else if (!((PreIncludeDirective*)yygotominor.yy0)->is_forced () && num == 0)
            	state->forcedIncludes = false;
            
            if (state->forcedIncludes)
            	num--;
            
            // Save current include depth.
            ((PreIncludeDirective*)yygotominor.yy0)->depth (num);

            // If the number of files included is higher than a maximal
            // count of maxIncludeDepth we should report an error and 
            // ignore the file to include.
            if (num >= state->maxIncludeDepth) {
                *err << ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location ()
                     << sev_error << "Include recursion too deep ("
                     << num << ")" << endMessage; 
            } 
            
            // Include the given file.
            else 
            {        // Update the predefined macros.
                updatePredefined ();
        
                // Include the file and add a daughter node with a 
                // reference to that file to the syntax tree.
                char* str = buildString (yymsp[0].minor.yy0);
                Unit* unit = 0;
                bool guarded = false;
                
                const char *txt = ((PreTreeToken*)yymsp[-1].minor.yy0)->token ()->text ();
                
                // compare last character of directive token to decide
                // whether this is an #include or #import directive
                if (txt && txt[strlen (txt)-1] == 't') {
                  // call the import handler with the arguments
                  // given at the #import line
                  ImportHandler ih (*parser, str);
                  unit = ih.IncludeFile ();
                // do an ordinary #include
                } else {
                  unit = parser->fileIncluder ()->includeFile (str);
                  guarded = parser->fileIncluder ()->guarded ();
                }
                
                yygotominor.yy0->add_daughter (new PreInclSemNode (unit, guarded));
                delete[] str;
            }
        }
    }
#line 2334 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 34:
        YYTRACE("include_next_line ::= TOK_PRE_INCLUDE_NEXT token_list")
#line 1227 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreIncludeDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);

        if (! state->errorDirective && state->passOnToken) 
        { 
            // Calculate the number of files included.
            int num = state->number ? state->number - 2 : ((TokenStream*)
                      (parser->scanner ()))->length () - 1;
            
            // If the number of files included is higher than a maximal
            // count of maxIncludeDepth we should report an error and 
            // ignore the file to include.
            if (num >= state->maxIncludeDepth) {
                *err << ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location ()
                     << sev_error << "Include recursion too deep ("
                     << num << ")" << endMessage; 
            } 
            
            // Include the next file with the given filename.
            else 
            {        // Update the predefined macros.
                updatePredefined ();
        
                // Include the next file and add a daughter node with
                // a reference to that file to the syntax tree.
                char* str = buildString (yymsp[0].minor.yy0);
                Unit *unit = parser->fileIncluder ()->includeNextFile (str);
                bool guarded = parser->fileIncluder ()->guarded ();
                yygotominor.yy0->add_daughter (new PreInclSemNode (unit, guarded));
                delete[] str;
            }
        }
    }
#line 2372 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 35:
        YYTRACE("assert_line ::= TOK_PRE_ASSERT token_list")
#line 1266 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreAssertDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 2380 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 36:
        YYTRACE("assert_line ::= TOK_PRE_ASSERT TOK_PRE_ID token_list")
#line 1270 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreAssertDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Get the answer text.
            char* str    = buildString (yymsp[0].minor.yy0);
            char* answer = evalAnswerString (str, true, *state);
            delete[] str;

            // If the answer isn't NULL get the name of the predicate.
            // If the predicate already exists add yymsp[-2].minor.yy0 new answer to that
            // predicate otherwise create yymsp[-2].minor.yy0 new predicate.
            if (answer) {
                const char* name = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->text ();
                if (PrePredicate* predicate = parser->predicateManager ()->
                                              getPredicate (name))
                    predicate->addAnswer (answer);
                else
                    parser->predicateManager ()->addPredicate (
                            new PrePredicate (name, answer));
                delete[] answer;
            }
        }
    }
#line 2410 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 37:
        YYTRACE("unassert_line ::= TOK_PRE_UNASSERT token_list")
#line 1301 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreUnassertDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 2418 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 38:
        YYTRACE("unassert_line ::= TOK_PRE_UNASSERT TOK_PRE_ID token_list")
#line 1305 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreUnassertDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);

        if (! state->errorDirective && state->passOnToken) 
        {
            // Get the answer text.
            char* str    = buildString (yymsp[0].minor.yy0);
            char* answer = evalAnswerString (str, false, *state);
            delete[] str;

            // If the answer string is empty remove the whole predicate.
            // Otherwise remove the answer for the predicate.
            if (answer) {
                const char* name = ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->text ();
                PrePredicate* predicate = parser->predicateManager ()->
                                          getPredicate (name);
                                          
                if (! strlen (answer)) 
                    parser->predicateManager ()->removePredicate (name);
                else if (predicate)
                    predicate->removeAnswer (answer);
                    
                delete[] answer;
            }
        }
    }
#line 2449 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 39:
        YYTRACE("define_line ::= TOK_PRE_DEFINE token_list")
#line 1337 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineConstantDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 2457 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 40:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID TOK_PRE_LPAREN TOK_PRE_RPAREN token_list")
#line 1342 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineFunctionDirective (yymsp[-4].minor.yy0, yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new function-like macro without arguments.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-3].minor.yy0)->
                              token ()->dtext (), (PreTree*) 0, str);
            delete[] str;
            macro->location (((PreTreeToken*) yymsp[-4].minor.yy0)->token ()->location ());
            
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2482 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 41:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID TOK_PRE_LPAREN TOK_PRE_REST TOK_PRE_RPAREN token_list")
#line 1364 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineFunctionDirective (yymsp[-5].minor.yy0, yymsp[-4].minor.yy0, yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new function-like macro with variable number 
            // of arguments.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-4].minor.yy0)->
                              token ()->dtext (), (PreTree*) 0, str, 
                              PreMacro::VAR_ARG_MACRO);
            delete[] str;
            macro->location (((PreTreeToken*) yymsp[-5].minor.yy0)->token ()->location ());
            
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2509 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 42:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID TOK_PRE_LPAREN identifier_list TOK_PRE_RPAREN token_list")
#line 1389 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineFunctionDirective (yymsp[-5].minor.yy0, yymsp[-4].minor.yy0, yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new function-like macro with arguments.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-4].minor.yy0)->
                              token ()->dtext (), yymsp[-2].minor.yy0, str);
            delete[] str;
            macro->location (((PreTreeToken*) yymsp[-5].minor.yy0)->token ()->location ());
        
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2534 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 43:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID TOK_PRE_LPAREN identifier_list TOK_PRE_REST TOK_PRE_RPAREN token_list")
#line 1412 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineFunctionDirective (yymsp[-6].minor.yy0, yymsp[-5].minor.yy0, yymsp[-4].minor.yy0, yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new function-like macro with arguments.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-5].minor.yy0)->
                              token ()->dtext (), yymsp[-3].minor.yy0, str, 
                              PreMacro::VAR_ARG_MACRO);
            delete[] str;
            macro->location (((PreTreeToken*) yymsp[-6].minor.yy0)->token ()->location ());
                    
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2560 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 44:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID TOK_PRE_LPAREN identifier_list TOK_PRE_COMMA TOK_PRE_REST TOK_PRE_RPAREN token_list")
#line 1436 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineFunctionDirective (yymsp[-7].minor.yy0, yymsp[-6].minor.yy0, yymsp[-5].minor.yy0, yymsp[-4].minor.yy0, yymsp[-3].minor.yy0, yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new function-like macro with arguments.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-6].minor.yy0)->
                              token ()->dtext (), yymsp[-4].minor.yy0, str, 
                              PreMacro::VAR_ARG_MACRO);
            delete[] str;
            macro->addArg ("__VA_ARGS__");
            macro->location (((PreTreeToken*) yymsp[-7].minor.yy0)->token ()->location ());
                    
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2587 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 45:
        YYTRACE("define_line ::= TOK_PRE_DEFINE TOK_PRE_ID token_list")
#line 1459 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreDefineConstantDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);

        if (! state->errorDirective && state->passOnToken) 
        {
            // Build a new simple macro.
            char*     str   = buildString (yymsp[0].minor.yy0);
            PreMacro* macro = new PreMacro (((PreTreeToken*) yymsp[-1].minor.yy0)->
                              token ()->dtext (), str);
            delete[] str;
            macro->location (((PreTreeToken*) yymsp[-2].minor.yy0)->token ()->location ());
            
            // Add the macro to the macro manager if its name is a valid
            // macro name.
            if (state->checkMacroName (macro)) 
                parser->macroManager ()->addMacro (macro);
            else
                delete macro;
        }
    }
#line 2612 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 46:
        YYTRACE("undef_line ::= TOK_PRE_UNDEF token_list")
#line 1485 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreUndefDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 2620 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 47:
        YYTRACE("undef_line ::= TOK_PRE_UNDEF TOK_PRE_ID token_list")
#line 1489 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreUndefDirective (yymsp[-2].minor.yy0, yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        // Remove the macro given by its name from macro manager.
        if (! state->errorDirective && state->passOnToken) 
            parser->macroManager ()->removeMacro (((PreTreeToken*) yymsp[-1].minor.yy0)->
                                                  token ()->dtext ());
    }
#line 2633 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 48:
        YYTRACE("warning_line ::= TOK_PRE_WARNING token_list")
#line 1503 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreWarningDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Print a warning message containing the text given as
            // argument of the #warning directive.
            char* message = buildString (yymsp[0].minor.yy0);
            *err << ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location () 
                 << "warning," << message << endMessage; 
            delete[] message;
        }
    }
#line 2651 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 49:
        YYTRACE("error_line ::= TOK_PRE_ERROR token_list")
#line 1522 "cpp/PreParser.lem"
{
        yygotominor.yy0 = new PreErrorDirective (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        
        if (! state->errorDirective && state->passOnToken) 
        {
            // Tell the parser to hide all following tokens from
            // other parsers in chain.
            state->errorDirective = true;
            
            // Print the text given as argument of the #error
            // directive.
            char* message = buildString (yymsp[0].minor.yy0);
            *err << ((PreTreeToken*) yymsp[-1].minor.yy0)->token ()->location () 
                 << "error," << message << endMessage; 
            delete[] message;
        }
    }
#line 2673 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 50:
        YYTRACE("identifier_list ::= TOK_PRE_ID")
#line 1545 "cpp/PreParser.lem"
{ 
        yygotominor.yy0 = new PreIdentifierList (yymsp[0].minor.yy0);
    }
#line 2681 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 51:
        YYTRACE("identifier_list ::= identifier_list TOK_PRE_COMMA TOK_PRE_ID")
#line 1549 "cpp/PreParser.lem"
{
        ((PreIdentifierList*) (yymsp[-2].minor.yy0))->addSons (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
        yygotominor.yy0 = yymsp[-2].minor.yy0;
    }
#line 2690 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 52:
        YYTRACE("token_list ::= TOK_PRE_NEWLINE")
#line 1559 "cpp/PreParser.lem"
{ 
        if (yymsp[0].minor.yy0->startToken () == state->newline) 
        {
            delete yymsp[0].minor.yy0;
            yygotominor.yy0 = new PreTokenList ();
        } 
        else
            yygotominor.yy0 = yymsp[0].minor.yy0;  
    }
#line 2704 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 53:
        YYTRACE("token_list ::= token_list_part TOK_PRE_NEWLINE")
#line 1569 "cpp/PreParser.lem"
{ 
        if (yymsp[0].minor.yy0->startToken () == state->newline) 
        {
            delete yymsp[0].minor.yy0;
            yygotominor.yy0 = new PreTokenList (yymsp[-1].minor.yy0);
        }
        else
            yygotominor.yy0 = new PreTokenList (yymsp[-1].minor.yy0, yymsp[0].minor.yy0);
    }
#line 2718 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 54:
        YYTRACE("token_list_part ::= TOK_PRE_TOKEN_LIST")
#line 1584 "cpp/PreParser.lem"
{ 
        yygotominor.yy0 = new PreTokenListPart (yymsp[0].minor.yy0); 
    }
#line 2726 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
      case 55:
        YYTRACE("token_list_part ::= token_list_part TOK_PRE_TOKEN_LIST")
#line 1588 "cpp/PreParser.lem"
{ 
        yymsp[-1].minor.yy0->add_son (yymsp[0].minor.yy0);
        yygotominor.yy0 = yymsp[-1].minor.yy0; 
    }
#line 2735 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
        break;
  };
  yygoto = yyRuleInfo[yyruleno].lhs;
  yysize = yyRuleInfo[yyruleno].nrhs;
  yypParser->idx -= yysize;
  yypParser->top -= yysize;
  yyact = yy_find_parser_action(yypParser,yygoto);
  if( yyact < YYNSTATE ){
    yy_shift(yypParser,yyact,yygoto,&yygotominor);
  }else if( yyact == YYNSTATE + YYNRULE + 1 ){
    yy_accept(yypParser PreParserARGDECL);
  }
}

/*
** The following code executes when the parse fails
*/
static void yy_parse_failed(
  yyParser *yypParser           /* The parser */
  PreParserANSIARGDECL              /* Extra arguments (if any) */
){
#ifndef NDEBUG
  if( yyTraceFILE ){
    fprintf(yyTraceFILE,"%sFail!\n",yyTracePrompt);
  }
#endif
  while( yypParser->idx>=0 ) yy_pop_parser_stack(yypParser);
  /* Here code is inserted which will be executed whenever the
  ** parser fails */
}

/*
** The following code executes when a syntax error first occurs.
*/
static void yy_syntax_error(
  yyParser *yypParser,           /* The parser */
  int yymajor,                   /* The major type of the error token */
  YYMINORTYPE yyminor            /* The minor type of the error token */
  PreParserANSIARGDECL               /* Extra arguments (if any) */
){
#define TOKEN (yyminor.yy0)
#line 721 "cpp/PreParser.lem"

    // Prevent the parser to report an error twice.
    if (state->syntaxError) {
        state->syntaxError = false;
    } 
    
    // Report the syntax error.
    else {
        Unit u;
        state->syntaxError = true;

        if (state->currType == Token::ID_END_OF_FILE) {
            *err << state->lastToken->location () << sev_error
                 << "End of file within directive" 
                 << endMessage; 
        } else if (state->lastType == TOK_PRE_NEWLINE) {
            Token* token = (Token*) (u.prev (state->lastToken));
            while (! token->is_preprocessor ())
                token = (Token*) (u.prev (token));
                *err << state->lastToken->location () << sev_error
                       << "Unterminated `" << token->text () 
                 << "' directive" << endMessage; 
        } else 
                *err << state->lastToken->location () << sev_error
                 << "Unexpected token `" 
                 << state->lastToken->text () << "'" << endMessage; 
    }

#line 2806 "/home/inf4/tartler/src/aspectc++/upstream/Puma/gen-release/step1/src/PreParser.cc"
}

/*
** The following is executed when the parser accepts
*/
static void yy_accept(
  yyParser *yypParser           /* The parser */
  PreParserANSIARGDECL              /* Extra arguments (if any) */
){
#ifndef NDEBUG
  if( yyTraceFILE ){
    fprintf(yyTraceFILE,"%sAccept!\n",yyTracePrompt);
  }
#endif
  while( yypParser->idx>=0 ) yy_pop_parser_stack(yypParser);
  /* Here code is inserted which will be executed whenever the
  ** parser accepts */
}

/* The main parser program.
** The first argument is a pointer to a structure obtained from
** "PreParserAlloc" which describes the current state of the parser.
** The second argument is the major token number.  The third is
** the minor token.  The fourth optional argument is whatever the
** user wants (and specified in the grammar) and is available for
** use by the action routines.
**
** Inputs:
** <ul>
** <li> A pointer to the parser (an opaque structure.)
** <li> The major token number.
** <li> The minor token number.
** <li> An option argument of a grammar-specified type.
** </ul>
**
** Outputs:
** None.
*/
void PreParser(
  void *yyp,                   /* The parser */
  int yymajor,                 /* The major token code number */
  PreParserTOKENTYPE yyminor       /* The value for the token */
  PreParserANSIARGDECL
){
  YYMINORTYPE yyminorunion;
  int yyact;            /* The parser action. */
  int yyendofinput;     /* True if we are at the end of input */
  int yyerrorhit = 0;   /* True if yymajor has invoked an error */
  yyParser *yypParser;  /* The parser */

  /* (re)initialize the parser, if necessary */
  yypParser = (yyParser*)yyp;
  if( yypParser->idx<0 ){
    /* Olaf: Empty files must be parse too!  if( yymajor==0 ) return; */
    yypParser->idx = 0;
    yypParser->errcnt = -1;
    yypParser->top = &yypParser->stack[0];
    yypParser->top->stateno = 0;
    yypParser->top->major = 0;
  }
  yyminorunion.yy0 = yyminor;
  yyendofinput = (yymajor==0);

#ifndef NDEBUG
  if( yyTraceFILE ){
    fprintf(yyTraceFILE,"%sInput %s\n",yyTracePrompt,yyTokenName[yymajor]);
  }
#endif

  do{
    yyact = yy_find_parser_action(yypParser,yymajor);
    if( yyact<YYNSTATE ){
      yy_shift(yypParser,yyact,yymajor,&yyminorunion);
      yypParser->errcnt--;
      if( yyendofinput && yypParser->idx>=0 ){
        yymajor = 0;
      }else{
        yymajor = YYNOCODE;
      }
    }else if( yyact < YYNSTATE + YYNRULE ){
      yy_reduce(yypParser,yyact-YYNSTATE PreParserARGDECL);
    }else if( yyact == YY_ERROR_ACTION ){
#ifndef NDEBUG
      if( yyTraceFILE ){
        fprintf(yyTraceFILE,"%sSyntax Error!\n",yyTracePrompt);
      }
#endif
#ifdef YYERRORSYMBOL
      /* A syntax error has occurred.
      ** The response to an error depends upon whether or not the
      ** grammar defines an error token "ERROR".  
      **
      ** This is what we do if the grammar does define ERROR:
      **
      **  * Call the %syntax_error function.
      **
      **  * Begin popping the stack until we enter a state where
      **    it is legal to shift the error symbol, then shift
      **    the error symbol.
      **
      **  * Set the error count to three.
      **
      **  * Begin accepting and shifting new tokens.  No new error
      **    processing will occur until three tokens have been
      **    shifted successfully.
      **
      */
      if( yypParser->errcnt<0 ){
        yy_syntax_error(yypParser,yymajor,yyminorunion PreParserARGDECL);
      }
      if( yypParser->top->major==YYERRORSYMBOL || yyerrorhit ){
#ifndef NDEBUG
        if( yyTraceFILE ){
          fprintf(yyTraceFILE,"%sDiscard input token %s\n",
             yyTracePrompt,yyTokenName[yymajor]);
        }
#endif
        yy_destructor(yymajor,&yyminorunion);
        yymajor = YYNOCODE;
      }else{
         while(
          yypParser->idx >= 0 &&
          yypParser->top->major != YYERRORSYMBOL &&
          (yyact = yy_find_parser_action(yypParser,YYERRORSYMBOL)) >= YYNSTATE
        ){
          yy_pop_parser_stack(yypParser);
        }
        if( yypParser->idx < 0 || yymajor==0 ){
          yy_destructor(yymajor,&yyminorunion);
          yy_parse_failed(yypParser PreParserARGDECL);
          yymajor = YYNOCODE;
        }else if( yypParser->top->major!=YYERRORSYMBOL ){
          YYMINORTYPE u2;
          u2.YYERRSYMDT = 0;
          yy_shift(yypParser,yyact,YYERRORSYMBOL,&u2);
        }
      }
      yypParser->errcnt = 3;
      yyerrorhit = 1;
#else  /* YYERRORSYMBOL is not defined */
      /* This is what we do if the grammar does not define ERROR:
      **
      **  * Report an error message, and throw away the input token.
      **
      **  * If the input token is $, then fail the parse.
      **
      ** As before, subsequent error messages are suppressed until
      ** three input tokens have been successfully shifted.
      */
      if( yypParser->errcnt<=0 ){
        yy_syntax_error(yypParser,yymajor,yyminorunion PreParserARGDECL);
      }
      yypParser->errcnt = 3;
      yy_destructor(yymajor,&yyminorunion);
      if( yyendofinput ){
        yy_parse_failed(yypParser PreParserARGDECL);
      }
      yymajor = YYNOCODE;
#endif
    }else{
      yy_accept(yypParser PreParserARGDECL);
      yymajor = YYNOCODE;
    }
  }while( yymajor!=YYNOCODE && yypParser->idx>=0 );
  return;
}
