// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Plan.h"
#include "AspectInfo.h"
#include "AdviceInfo.h"
#include "OrderInfo.h"
#include "IntroductionInfo.h"
#include "PointCutContext.h"
#include "PointCutEvaluator.h"
#include "PointCutExpr.h"
#include "JoinPointModel.h"
#include "JoinPointPlan.h"

#include "Puma/ACAspectInfo.h"
#include "Puma/ACTree.h"
#include "Puma/ErrorSink.h"

Plan::Plan (ErrorStream &e, JoinPointModel &jpm) : _err (e), _jpm (jpm) {

  // get all aspects from the join point model
  JoinPointModel::Selection all_aspects;
  _jpm.select (JoinPointLoc::Aspect, all_aspects);

  // sort them by inserting them into a set
  for (JoinPointModel::Selection::iterator iter = all_aspects.begin ();
       iter != all_aspects.end (); ++iter) {
    JPL_Aspect &jpl_aspect = (JPL_Aspect&)**iter;
    ACAspectInfo *acai = ((TI_Aspect*)jpl_aspect.transform_info ())->aspect_info();

    // Don't weave advice of abstract aspects
    if (acai->is_abstract ())
      continue;
    
    AspectInfo *aspect_info = addAspect (jpl_aspect);
    
    // traverse inheritance tree and collect advice declarations
    list<JPL_AdviceCode*> advices;
    jpl_aspect.collect_advice_codes (advices);

    // handle advice
    for (list<JPL_AdviceCode*>::const_iterator i = advices.begin ();
      i != advices.end (); ++i) {
      JPL_AdviceCode *code = *i;
      AdviceInfo *advice_info = addAdvice (*aspect_info, *code);
      aspect_info->advice_infos ().push_back (advice_info);
    }
  }
}

Plan::~Plan () {
  for (int i = 0; i < (int)_advice_infos.length (); i++)
    delete _advice_infos[i];
  for (int i = 0; i < (int)_exec_jpls.length (); i++)
    delete _exec_jpls[i]->plan ();
  for (int i = 0; i < (int)_call_jpls.length (); i++)
    delete _call_jpls[i]->plan ();
  for (int i = 0; i < (int)_cons_jpls.length (); i++)
    delete _cons_jpls[i]->plan ();
  for (int i = 0; i < (int)_dest_jpls.length (); i++)
    delete _dest_jpls[i]->plan ();
  for (list<IntroductionInfo*>::iterator i = _introduction_infos.begin ();
       i != _introduction_infos.end (); ++i)
    delete *i;
  for (list<OrderInfo*>::iterator i = _order_infos.begin ();
       i != _order_infos.end (); ++i)
    delete *i;
}

AspectInfo *Plan::addAspect (JPL_Aspect &a) {
  pair<AspectContainer::iterator, bool> res = _aspect_infos.insert (AspectInfo (a));
  return (AspectInfo*)&*res.first;
}

AdviceInfo *Plan::addAdvice (AspectInfo &ai, JPL_AdviceCode &code) {
  AdviceInfo *result = new AdviceInfo (ai, code);
  _advice_infos.append (result);
  return result;
}

IntroductionInfo *Plan::addIntroduction (JPL_Aspect &jpl_aspect, JPL_Introduction &intro) {
  IntroductionInfo *result = new IntroductionInfo (jpl_aspect, intro);
  _introduction_infos.push_back (result);
  return result;
}

OrderInfo *Plan::addOrder (JPL_Aspect &a, JPL_Order &o) {
  OrderInfo *result = new OrderInfo (a, o);
  _order_infos.push_back (result);
  return result;
}

// consider a join point and advice in the plan
void Plan::consider (JoinPointLoc *jpl, const Condition &cond, AdviceInfo *ai) {
  JPL_Code *jp_code = (JPL_Code*)jpl;
  if (!jp_code->is_pseudo ()) {
    JPP_Code *jp_plan = (JPP_Code*)jpl->plan ();
    if (!jp_plan) {
      switch (jpl->type ()) {
      case JoinPointLoc::Method:
        jp_plan = new JPP_Code;
        _exec_jpls.append (jpl);
        break;
      case JoinPointLoc::MethodCall:
        jp_plan = new JPP_Code;
        _call_jpls.append (jpl);
        break;
      case JoinPointLoc::Construction:
        jp_plan = new JPP_Code;
        _cons_jpls.append (jpl);
        break;
      case JoinPointLoc::Destruction:
        jp_plan = new JPP_Code;
        _dest_jpls.append (jpl);
        break;
      default:
        _err << sev_error
             << "internal problem, invalid join point type in plan" 
             << endMessage;
        return;
      }
      jpl->plan (jp_plan);
    }
    jp_plan->consider (ai, cond);
  }

  if (cond) {
    CObjectInfo *assoc_obj = TransformInfo::of (*jpl)->assoc_obj ();
    CFunctionInfo *that_func = (assoc_obj ? assoc_obj->FunctionInfo () : 0);
    CFunctionInfo *target_func = ((jpl->type () == JoinPointLoc::MethodCall) ?
      ((TI_MethodCall*)jpl->transform_info ())->called () : 0);

    cond.checks_for_that (_type_checks_true);
    cond.checks_for_target (_type_checks_true);

    StringSet check_names_that, check_names_target;

    if (that_func) {
      cond.names_for_that (check_names_that);
      for (StringSet::iterator iter = check_names_that.begin ();
           iter != check_names_that.end (); ++iter) {
        _type_checks_false.insert (TypeCheck (that_func->ClassScope (),
                                   *iter));
      }
    }

    if (target_func) {
      cond.names_for_target (check_names_target);
      for (StringSet::iterator iter = check_names_target.begin ();
	   iter != check_names_target.end (); ++iter) {
	_type_checks_false.insert (TypeCheck (target_func->ClassScope (), 
					      *iter));
      }
    }
  }
}

void Plan::consider (JoinPointLoc *jpl, const CFlow &cflow) {
  JPL_Code *jp_code = (JPL_Code*)jpl;
  if (jp_code->is_pseudo ())
    return;
    
  JPP_Code *jp_plan = (JPP_Code*)jpl->plan ();
  if (!jp_plan) {
    switch (jpl->type ()) {
    case JoinPointLoc::Method:
      jp_plan = new JPP_Code;
      _exec_jpls.append (jpl);
      break;
    case JoinPointLoc::MethodCall:
      jp_plan = new JPP_Code;
      _call_jpls.append (jpl);
      break;
    case JoinPointLoc::Construction:
      jp_plan = new JPP_Code;
      _cons_jpls.append (jpl);
      break;
    case JoinPointLoc::Destruction:
      jp_plan = new JPP_Code;
      _dest_jpls.append (jpl);
      break;
    default:
      _err << sev_error
           << "internal problem, invalid join point type in plan" 
           << endMessage;
      return;
    }
    jpl->plan (jp_plan);
  }
  jp_plan->consider (cflow);
}

// consider a join point for an introduction in the plan
JPP_Class *Plan::consider (JoinPointLoc *jpl, JPL_Introduction *intro) {
  JPP_Class *jp_plan = (JPP_Class*)jpl->plan ();
  if (!jp_plan) {
    switch (jpl->type ()) {
    case JoinPointLoc::Class:
    case JoinPointLoc::Aspect:
      jp_plan = new JPP_Class;
      _class_jpls.append (jpl);
      break;
    default:
      _err << sev_error
           << "internal problem, invalid join point type for intro in plan" 
           << endMessage;
      return 0;
    }
    jpl->plan (jp_plan);
  }
  
  jp_plan->consider (intro);
  return jp_plan;
}
  
// consider ordering information in the plan
void Plan::consider(JoinPointLoc* jpl, JPL_Aspect &hi, JPL_Aspect &lo) {
  JoinPointPlan* jp_plan = jpl->plan();
  if (jp_plan) {
    ACAspectInfo *h = TI_Aspect::of (hi)->aspect_info ();
    ACAspectInfo *l = TI_Aspect::of (lo)->aspect_info ();
    jp_plan->consider(*h, *l);
  }
}

// calculate the order for a single join points
void Plan::order (JoinPointLoc *jpl) {
  PointCutContext context (_jpm);

  const list<OrderInfo*> &orders = order_infos ();
  for (list<OrderInfo*>::const_iterator i = orders.begin ();
       i != orders.end (); ++i) {
    OrderInfo *order = *i;

    // check if the current join point location is matched by the pointcut
    // expression of the order advice
    context.concrete_aspect (order->aspect ());
    Binding binding;     // binding and condition not used for intros
    Condition condition;
    if (!order->jp_pce ()->evaluate (*jpl, context, binding, condition))
      continue;
    // remember the partial orderings in the plan for this join point
    const list<PointCutExpr*> &pces = order->pces ();
    list<JPL_Aspect*> *hi = 0;
    list<JPL_Aspect*> *lo = 0;
    bool start = true;
    for (list<PointCutExpr*>::const_iterator i2 = pces.begin ();
         i2 != pces.end (); ++i2) {
      PointCutExpr *pce = *i2;

      // match all relevant aspects with the current match expression
      list<JPL_Aspect*> *matched = new list<JPL_Aspect*>;
      // TODO: this is inefficient. Not all aspects are relevant for each jp
      // get all aspects from the join point model
      JoinPointModel::Selection all_aspects;
      _jpm.select (JoinPointLoc::Aspect, all_aspects);
      for (JoinPointModel::Selection::iterator iter = all_aspects.begin ();
        iter != all_aspects.end (); ++iter) {
        JPL_Aspect &jpl_aspect = (JPL_Aspect&)**iter;
        if (pce->evaluate (jpl_aspect, context, binding, condition))
          matched->push_back (&jpl_aspect);
      }
      if (matched->size () == 0) {
	delete matched;
      }
      else if (start) {
        lo    = matched; // will become high in the next loop
	start = false;
      }
      else {
        if (hi) delete hi;
        hi = lo;
        lo = matched;

        // now we can consider the ordering info in the plan
        for (list<JPL_Aspect*>::iterator ihi = hi->begin ();
             ihi != hi->end (); ++ihi) {
          for (list<JPL_Aspect*>::iterator ilo = lo->begin ();
               ilo != lo->end (); ++ilo) {
//              cout << (*ihi)->signature () << " -> " << (*ilo)->signature () << endl;
            consider (jpl, **ihi, **ilo);
          }
        }
      }
    }
    if (lo) delete lo;
  }
  
}
  
// calculate the order for all join points
void Plan::order () {
  for (int i = 0; i < (int)_exec_jpls.length (); i++)
    order (_exec_jpls[i]);
  for (int i = 0; i < (int)_call_jpls.length (); i++)
    order (_call_jpls[i]);
  for (int i = 0; i < (int)_cons_jpls.length (); i++)
    order (_cons_jpls[i]);
  for (int i = 0; i < (int)_dest_jpls.length (); i++)
    order (_dest_jpls[i]);
  for (int i = 0; i < (int)_class_jpls.length (); i++) 
    order (_class_jpls[i]);
}  

// check the plan for a specific join point
void Plan::check (JoinPointLoc *jpl) {
  JoinPointPlan *plan = jpl->plan ();
  if (plan) {
    if (!plan->check (_err))
      _err << " for " << jpl->type_str () << "(\"" << jpl->signature () << "\")"
           << TransformInfo::location (*jpl) << endMessage;
  }
}

// check the final plan -> messages to the error sink
void Plan::check () {
  for (int i = 0; i < (int)_exec_jpls.length (); i++)
    check (_exec_jpls[i]);
  for (int i = 0; i < (int)_call_jpls.length (); i++)
    check (_call_jpls[i]);
  for (int i = 0; i < (int)_cons_jpls.length (); i++)
    check (_cons_jpls[i]);
  for (int i = 0; i < (int)_dest_jpls.length (); i++)
    check (_dest_jpls[i]);
  for (int i = 0; i < (int)_class_jpls.length (); i++) 
    check (_class_jpls[i]);
}
