// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/InstantiationCandidate.h"
#include "Puma/CStructure.h"
#include "Puma/CClassInstance.h"
#include "Puma/CUnionInstance.h"
#include "Puma/CFctInstance.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CTemplateParamInfo.h"
#include "Puma/CTree.h"
#include "Puma/CCOverloading.h"
#include "Puma/CConstant.h"
#include "Puma/CCConversions.h"
#include "Puma/CUnit.h"
#include "Puma/CCParser.h"
#include "Puma/TokenStream.h"
#include "Puma/UnitManager.h"
#include "Puma/CClassDatabase.h"
#include "Puma/PreMacroManager.h"
#include "Puma/CTranslationUnit.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/CFileInfo.h"
#include "Puma/CSourceInfo.h"
#include "Puma/CCConversions.h"

namespace Puma {


#define SEM_ERROR(loc__,mesg__) \
  { \
    if (err) { \
      (*err) << sev_error << loc__->token ()->location () \
             << mesg__ << endMessage; \
    } \
  }


InstantiationCandidate::InstantiationCandidate () {
  obj_info = 0;
  tpl_info = 0;
  poi      = 0;
  err      = 0;
}


InstantiationCandidate::~InstantiationCandidate () {
// TEMPORARY HACK
//  for (long i = 0; i < darguments.length (); i++)
//    if (darguments[i])
//      delete darguments[i];
}


void InstantiationCandidate::reset () {
  obj_info = 0;
  tpl_info = 0;
  poi      = 0;
  err      = 0;

  for (long i = 0; i < darguments.length (); i++)
    if (darguments[i])
      delete darguments[i];

  arguments.reset ();
  darguments.reset ();
}


void InstantiationCandidate::initialize (ErrorSink *e) {
  err = e;
}


void InstantiationCandidate::initialize (CTree *p, CObjectInfo *o) {
  poi      = p;
  obj_info = o;
}


void InstantiationCandidate::initialize (CTree *p, CObjectInfo *o, CTemplateInfo *t, ErrorSink *e) {
  poi      = p;
  obj_info = o;
  tpl_info = t;
  err      = e ? e : err;
}


int InstantiationCandidate::getPosition (CTemplateParamInfo *param) const {
  for (unsigned i = 0; i < tpl_info->Parameters (); i++) 
    if (*param == *tpl_info->Parameter (i))
      return i;
  return -1;
}


// 14.8.2 deduce template arguments (if necessary)
bool InstantiationCandidate::deduceArguments (bool real_inst) {
  CTemplateParamInfo *param;
  CT_TemplateArgList *args;
  CTypeInfo *ptype, *atype;
  CObjectInfo *oinfo = 0;
  unsigned numargs;
  bool default_arg;
  CTree *arg, *sn, *name;
  
  // template arguments
  name = poi;
  if (name->NodeName () == CT_QualName::NodeId () ||
      name->NodeName () == CT_RootQualName::NodeId ())
    name = ((CT_QualName*)name)->Name ();
  if (name->NodeName () == CT_TemplateName::NodeId ()) {
    args = ((CT_TemplateName*)name)->Arguments ();
    numargs = args->Entries ();
  } else {
    args = 0;
    numargs = 0;
  }
  
  // get template parameter list information
  if (obj_info->ClassInfo ())
    tpl_info = obj_info->ClassInfo ()->TemplateInfo ();
  else if (obj_info->UnionInfo ())
    tpl_info = obj_info->UnionInfo ()->TemplateInfo ();
  else if (obj_info->FunctionInfo ())
    tpl_info = obj_info->FunctionInfo ()->TemplateInfo ();
  else if (obj_info->TemplateParamInfo ())
    tpl_info = obj_info->TemplateParamInfo ()->TemplateInfo ();
  if (! tpl_info) {
// TEMPORARY HACK --->
    if (real_inst) {
      SEM_ERROR (poi, "fatal error: missing template parameter information");
      return false;
    } else
      return true;
// TEMPORARY HACK <---
  }
  
  // check number of template arguments for function templates
  if (obj_info->FunctionInfo () && numargs > tpl_info->Parameters ()) {
// TEMPORARY HACK --->
    if (real_inst) {
      SEM_ERROR (poi, "wrong number of template arguments (" 
        << numargs << ", should be at most " << tpl_info->Parameters () << ")");
      return false;
    } else
      return true;
// TEMPORARY HACK <---
  }
    
  // compare type of arguments with type of parameters
  for (unsigned i = 0; i < tpl_info->Parameters (); i++) {
    param = tpl_info->Parameter (i);
    default_arg = false;

    if (i >= numargs) {
      // consider default template arguments for non-function templates
      if (! obj_info->FunctionInfo ()) {
        if (! param->DefaultArgument ()) {
// TEMPORARY HACK --->
          if (real_inst) {
            SEM_ERROR (poi, "missing default argument for parameter " << i+1);
            return false;
          } else
            return true;
// TEMPORARY HACK <---
        } else {
          arg = param->DefaultArgument ()->Entry (0);
          default_arg = true;
        }
      // function template parameters cannot have default arguments;
      // ommited arguments must be deducible from the function call
      // argument list
      } else 
        return deduceArguments (i);
    } else
      arg = args->Entry (i);
    
    // template template parameter: expect name of class base template
    if (param->isTemplate ()) {
// TEMPORARY HACK --->
      if (real_inst) {
        sn = arg;
        if (sn->NodeName () == CT_NamedType::NodeId ())
          sn = sn->Son (0)->Son (0);
        if (! (sn->NodeName () == CT_SimpleName::NodeId () ||
               sn->NodeName () == CT_RootQualName::NodeId () ||
               sn->NodeName () == CT_QualName::NodeId ()) ||
            ! sn->SemObject () || 
            ! (oinfo = sn->SemObject ()->Object ()) ||
            ! ((oinfo->TemplateParamInfo () && oinfo->TemplateParamInfo ()->isTemplate ()) ||
               (oinfo->Record () && oinfo->Record ()->isTemplate ()))) {
          SEM_ERROR (arg, "expected a class template as argument " << i+1);
          return false;
        }
        darguments.append (new DeducedArgument (param, 
          oinfo->TypeInfo ()->VirtualType (), arg, default_arg));
      } else {
        if (arg->SemObject () && (oinfo = arg->SemObject ()->Object ()) && oinfo->TypeInfo ())
          darguments.append (new DeducedArgument (param, 
            oinfo->TypeInfo ()->VirtualType (), arg, default_arg));
      }
// TEMPORARY HACK <---
    // template type parameter: expect type-id
    } else if (param->isTypeParam ()) {
// TEMPORARY HACK --->
      if (real_inst) {
        if (! (arg->NodeName () == CT_NamedType::NodeId ()) || ! arg->SemObject () || 
            ! (oinfo = arg->SemObject ()->Object ())) {
          SEM_ERROR (arg, "expected a type as argument " << i+1);
          return false;
        }
        darguments.append (new DeducedArgument (param, 
          oinfo->TypeInfo ()->VirtualType (), arg, default_arg));
      } else {
        if (arg->SemObject () && (oinfo = arg->SemObject ()->Object ()) && oinfo->TypeInfo ())
          darguments.append (new DeducedArgument (param, 
            oinfo->TypeInfo ()->VirtualType (), arg, default_arg));
      }
// TEMPORARY HACK <---
    // template non-type parameter: expect constant expression
    } else {
// TEMPORARY HACK --->
      if (real_inst) {
        // not a non-type argument
        if (arg->NodeName () == CT_NamedType::NodeId ()) {
          SEM_ERROR (arg, "expected a constant as argument " << i+1);
          return false;
        }
        // consider template non-type parameter as argument
        if (arg->Type () && arg->Type ()->TypeTemplateParam () &&
            arg->Type ()->TypeTemplateParam ()->isNonType ()) {
          darguments.append (new DeducedArgument (param, arg->Type (), arg, default_arg));
          continue;
        }
        // need value of constant
        if (! arg->Value () || ! arg->Value ()->Constant ()) {
          SEM_ERROR (arg, "expected a constant as argument " << i+1);
          return false;
        }
        
        // promote parameter type
        CCConversions cvs (*err);
        ptype = cvs.integralPromotion (param->ValueType ()->VirtualType ());
        atype = cvs.integralPromotion (arg->Type ()->VirtualType ());
        if (! ptype)
          ptype = param->ValueType ()->VirtualType ();
        if (! atype)
          atype = arg->Type ()->VirtualType ();
        CCConvSeq *seq = cvs.implicitConversions (ptype, atype, arg, 0);
        if (! seq) {
          SEM_ERROR (arg, "expected type `" << *param->ValueType ()
            << "', got type `" << *arg->Type () << "' for argument " << i+1);
          return false;
        } else {
          delete seq;
        }
        darguments.append (new DeducedArgument (param, 
          arg->Value ()->Constant (), arg, default_arg));
      } else {
        // consider template non-type parameter as argument
        if (arg->Type () && arg->Type ()->TypeTemplateParam () &&
            arg->Type ()->TypeTemplateParam ()->isNonType ()) {
          darguments.append (new DeducedArgument (param, 
            arg->Type (), arg, default_arg));
        } else if (arg->Value () && arg->Value ()->Constant ()) {
          darguments.append (new DeducedArgument (param, 
            arg->Value ()->Constant (), arg, default_arg));
        }
      }
// TEMPORARY HACK <---
    }
  }
  return true;
}


// 14.8.2 deduce template arguments from function call
bool InstantiationCandidate::deduceArguments (unsigned skip) {
  DeducedArgumentList dargs (tpl_info->Parameters ()-skip);
  CFunctionInfo *finfo;
  CTypeList *fargs;
  unsigned numargs;
  CTypeInfo *type;
  bool succeeded;
  CTree *arg;

  succeeded = true;
  finfo = obj_info->FunctionInfo ();
  fargs = finfo->TypeInfo ()->ArgTypes ();
  numargs = Arguments ();
  
  // compare number of function arguments and parameters
  if (fargs->Entries () > numargs) {
    numargs = fargs->Entries ();
  } else if (fargs->Entries () < numargs) {
    SEM_ERROR (poi, "too many arguments in call to function `"
      << finfo->Name () << "'");
    return false;
  }

  // prepare temporary deduced argument container
  for (unsigned i = 0; i < tpl_info->Parameters (); i++)
    dargs.append (0);

  // iterate and analyse arguments
  for (unsigned i = 0; i < numargs && succeeded; i++) {
    // consider default arguments for function parameters
    if (i >= Arguments ()) {
      arg = finfo->DefaultArgument (i);
      if (! arg) {
        SEM_ERROR (poi, "too few arguments in call to function `"
          << finfo->Name () << "'");
        return false;
      }
    } else
      arg = Argument (i);
    
    // 14.3.2.5 implicit argument type conversions
    type = arg->Type ()->VirtualType ();
    // array-to-pointer and function-to-pointer conversions
    if (type->isArray () || type->isFunction ()) {
      type = type->isArray () ? type->VirtualType ()->BaseType () : type;
      type = new CTypePointer (type->Duplicate ());
    } else {
      // qualification conversion
      if (type->TypeQualified ())
        type = type->UnqualType ()->Duplicate ();
      // integral promotion and conversion
      if (type->isInteger ()) {
        CCConversions cvs (*err);
        type = cvs.integralPromotion (type)->Duplicate ();
      }
    }
        
    succeeded = deduceArguments (fargs->Entry (i)->VirtualType (), type, dargs);

    if (type != arg->Type ()->VirtualType ())
      CTypeInfo::Destroy (type);
  }

  for (unsigned i = 0; i < (unsigned)dargs.length () && succeeded; i++) {
    if (! dargs[i]) {
      SEM_ERROR (poi, "could not deduce argument for parameter " << i+1+skip);
      succeeded = false;
    } else 
      darguments.append (dargs[i]);
  }
  
  return succeeded;
}


bool InstantiationCandidate::deduceArguments (CTypeInfo *ftype, CTypeInfo *atype, 
 DeducedArgumentList &dargs) {
  CTemplateInstance *fui, *aui, *fci, *aci;
  CObjectInfo *finfo, *ainfo;
  CTemplateParamInfo *param;
  CTypeList *ftl, *atl;
  CTypeInfo *type;
  int pos;

  ftype = ftype->UnqualType ();
  atype = atype->UnqualType ();
  
  if (ftype->TypeTemplateParam ()) {
    param = ftype->TypeTemplateParam ()->TemplateParamInfo ();
    pos = getPosition (param);
    if (pos != -1) {
      if (param->isTemplate ()) {
        if (atype->TypeRecord () && 
            (ainfo = atype->TypeRecord ()->Record ()) &&
            ainfo->isTemplateInstance ()) {
          if (ainfo->UnionInstance ())
            dargs[pos] = new DeducedArgument (param, 
              ainfo->TemplateInstance ()->Template ()->ObjectInfo ()->TypeInfo ());
          else if (ainfo->ClassInstance ())
            dargs[pos] = new DeducedArgument (param, 
              ainfo->TemplateInstance ()->Template ()->ObjectInfo ()->TypeInfo ());
        }
      } else if (param->isTypeParam ()) 
        dargs[pos] = new DeducedArgument (param, atype);
    }
    return true;
  } 
  
  if (ftype->Id () != atype->Id () ||
      ftype->TypeEmpty () || ftype->TypePrimitive ()) {
    return true;
  } 
  
  if (ftype->TypeMemberPointer ()) {
    param = ftype->TypeMemberPointer ()->TemplateParam ();
    if (param) {
      pos = getPosition (param);
      if (pos != -1 && param->isTypeParam () && atype->TypeMemberPointer ()->Record ()) 
        dargs[pos] = new DeducedArgument (param, 
          atype->TypeMemberPointer ()->Record ()->TypeInfo ());
    }
  } else if (ftype->TypeRecord ()) {
    finfo = ftype->TypeRecord ()->Record ();
    ainfo = atype->TypeRecord ()->Record ();
    if (finfo && ainfo && finfo->isTemplateInstance () &&
        ainfo->isTemplateInstance ()) {
      fui = finfo->UnionInstance () ? finfo->TemplateInstance () : 0; 
      aui = ainfo->UnionInstance () ? ainfo->TemplateInstance () : 0; 
      if (fui && aui && fui->DeducedArgs () == aui->DeducedArgs ()) {
        for (unsigned i = 0; i < fui->DeducedArgs (); i++)
          if ((type = fui->DeducedArg (i)->Type ())) {
            if (aui->DeducedArg (i)->Type ()) {
              if (! deduceArguments (type, aui->DeducedArg (i)->Type (), dargs))
                return false;
            } else if (type->TypeTemplateParam () &&
                     ! type->TypeTemplateParam ()->isTypeParam ()) {
              param = type->TypeTemplateParam ()->TemplateParamInfo ();
              if (param) {
                pos = getPosition (param);
                if (pos != -1) 
                  dargs[pos] = new DeducedArgument (param, aui->DeducedArg (i)->Value ());
              }
            }
          }
      }
      fci = finfo->ClassInstance () ? finfo->TemplateInstance () : 0; 
      aci = ainfo->ClassInstance () ? ainfo->TemplateInstance () : 0; 
      if (fci && aci && fci->DeducedArgs () == aci->DeducedArgs ()) {
        for (unsigned i = 0; i < fci->DeducedArgs (); i++)
          if ((type = fci->DeducedArg (i)->Type ())) {
            if (aci->DeducedArg (i)->Type ()) {
              if (! deduceArguments (type, aci->DeducedArg (i)->Type (), dargs))
                return false;
            } else if (type->TypeTemplateParam () &&
                     ! type->TypeTemplateParam ()->isTypeParam ()) {
              param = type->TypeTemplateParam ()->TemplateParamInfo ();
              if (param) {
                pos = getPosition (param);
                if (pos != -1) 
                  dargs[pos] = new DeducedArgument (param, aci->DeducedArg (i)->Value ());
              }
            }
          }
      }
    }
    return true;
  } else if (ftype->TypeArray ()) {
    if (ftype->TypeArray ()->DepDim ()) {
      param = ftype->TypeArray ()->DepDim ()->TemplateParamInfo ();
      if (param) {
        pos = getPosition (param);
        if (pos != -1 && ! param->isTypeParam () && atype->TypeArray ()->Dimension ())
          dargs[pos] = new DeducedArgument (param, atype->TypeArray ()->Dimension ());
      }
    }
  } else if (ftype->TypeFunction ()) {
    if (ftype->TypeFunction ()->ArgTypes ()->Entries () == 
        atype->TypeFunction ()->ArgTypes ()->Entries ()) {
      ftl = ftype->TypeFunction ()->ArgTypes ();
      atl = atype->TypeFunction ()->ArgTypes ();
      for (unsigned i = 0; i < ftl->Entries (); i++) {
        if (! deduceArguments (ftl->Entry (i), atl->Entry (i), dargs)) 
          return false;
      }
    }
  }
  
  return deduceArguments (ftype->BaseType (), atype->BaseType (), dargs);
}


// 14.5.4.1.2 try to match a partial specialization 
// against a given template argument list
bool InstantiationCandidate::match () {
  CT_TemplateArgList *args;
  CTemplateInfo *base;
  CTree *arg, *parg, *name;
  unsigned numargs;
  
  // template argument list
  name = poi;
  if (name->NodeName () == CT_QualName::NodeId () ||
      name->NodeName () == CT_RootQualName::NodeId ())
    name = ((CT_QualName*)name)->Name ();
  if (name->NodeName () == CT_TemplateName::NodeId ()) {
    args = ((CT_TemplateName*)name)->Arguments ();
    numargs = args->Entries ();
  } else {
    args = 0;
    numargs = 0;
  }
  
  // prepare deduced argument container
  darguments.reset ();
  for (unsigned i = 0; i < tpl_info->Parameters (); i++)
    darguments.append (0);

  // compare argument lists
  base = tpl_info->BaseTemplate ();
  for (unsigned i = 0; i < Arguments (); i++) {
    // current partial specialization argument
    parg = Argument (i); 

    // more partial specialization arguments than arguments
    // given at point of instantiation => consider default arguments
    if (i >= numargs) {
      if (base && base->Parameters () > i && base->Parameter (i)->DefaultArgument ()) {
        arg = base->Parameter (i)->DefaultArgument ()->Entry (0);
      } else {
        // oops, no default argument? this is an error
        return false;
      }
    } else
      arg = args->Entry (i);
      
    // try to match the current argument, must be an exact match!
    if (! matchArgument (parg, arg))
      return false;
  }
  
  // check if all template parameters could be deduced while matching
  for (unsigned i = 0; i < DeducedArgs (); i++)
    if (! DeducedArg (i))
      return false;
  
  return true;
}


bool InstantiationCandidate::sameType (int pos, CTypeInfo *type) {
  DeducedArgument *darg = darguments[pos];
  if (darg && darg->Value ())
    return false; // not a type
  CTypeInfo *otype = darg ? darg->Type () : 0;
  if (! type)
    return false;
  if (otype) {
    if (otype->TypeTemplateParam () && type->TypeTemplateParam ()) {
      CTemplateParamInfo *p1 = otype->TypeTemplateParam ()->TemplateParamInfo ();
      CTemplateParamInfo *p2 = type->TypeTemplateParam ()->TemplateParamInfo ();
      if (! p1 || ! p2 || *p1 != *p2)
        return false;
    } else if (*otype != *type)
      return false;
  }
  return true;
}


bool InstantiationCandidate::sameValue (int pos, CConstant *value) {
  DeducedArgument *darg = darguments[pos];
  if (darg && darg->Type ())
    return false; // not a value
  CConstant *ovalue = darg ? darg->Value () : 0;
  if (! value || (ovalue && (*ovalue != *value)))
    return false;
  return true;
}


bool InstantiationCandidate::matchArgument (CTree *pexpr, CTree *aexpr) {
  CConstant *value;

  // match against value, must be equal
  if (pexpr->Value () && pexpr->Value ()->Constant ()) {
    if (aexpr->Value () && (value = aexpr->Value ()->Constant ()) &&
        (*value == *pexpr->Value ()->Constant ())) 
      return true;
    return false;
  }

  CTypeInfo *ptype = 0, *atype = 0;
  if (pexpr->NodeName () == CT_NamedType::NodeId () &&
      pexpr->SemObject () && pexpr->SemObject ()->Object ())
    ptype = pexpr->SemObject ()->Object ()->TypeInfo ();
  if (aexpr->NodeName () == CT_NamedType::NodeId () &&
      aexpr->SemObject () && aexpr->SemObject ()->Object ())
    atype = aexpr->SemObject ()->Object ()->TypeInfo ();
  if (! ptype)
    ptype = pexpr->Type ();
  if (! atype)
    atype = aexpr->Type ();
  if (! ptype || ! atype)
    return false;

  return matchArgument (ptype, atype, aexpr);
}


bool InstantiationCandidate::matchArgument (CTypeInfo *ptype, CTypeInfo *atype, CTree *aexpr) {
  CTemplateInstance *pti, *ati;
  CTemplateInfo *ptinfo, *atinfo;
  CObjectInfo *pinfo, *ainfo;
  CTemplateParamInfo *param, *param2;
  CTypeList *ptl, *atl;
  CConstant *value, *value2;
  CTypeInfo *type, *type2;
  CRecord *prec, *arec;
  long pdim, adim;
  int pos;

  // match against template parameter
  if (ptype->TypeTemplateParam ()) {
    param = ptype->TypeTemplateParam ()->TemplateParamInfo ();
    pos = getPosition (param);
    if (pos != -1) {
      // template template parameter
      if (param->isTemplate ()) {
        if (atype->TypeTemplateParam ()) {
          if (sameType (pos, atype)) {
            darguments[pos] = new DeducedArgument (param, atype);
            return true;
          }
        } else if (atype->TypeRecord () && (ainfo = atype->TypeRecord ()->Record ()) &&
            ainfo->isTemplateInstance ()) {
          type = ainfo->TemplateInstance ()->Template ()->ObjectInfo ()->TypeInfo ();
          // template parameter must always deduce to the same type
          if (sameType (pos, type)) {
            darguments[pos] = new DeducedArgument (param, type);
            return true;
          }
        }
      // type template parameter
      } else if (param->isTypeParam ()) {
        if (sameType (pos, atype)) {
          darguments[pos] = new DeducedArgument (param, atype);
          return true;
        }
      // non-type template parameter
      } else if (atype->TypeTemplateParam ()) {
        if (atype->TypeTemplateParam ()->isNonTypeParam ()) {
          if (sameType (pos, atype)) {
            darguments[pos] = new DeducedArgument (param, atype);
            return true;
          }
        }
      // non-type template parameter
      } else if (aexpr) {
        if (aexpr->Value () && (value = aexpr->Value ()->Constant ()) &&
            sameValue (pos, value)) {
          darguments[pos] = new DeducedArgument (param, value);
          return true;
        }
      }
    } 
    return false;
  } 
  
  // non-template parameter arguments and non-value arguments 
  // must be equal in type
  if (ptype->Id () != atype->Id ())
    return false;

  // match against primitive type, no further analyses needed
  if (ptype->TypeEmpty () || ptype->TypePrimitive ())
    return true;
  
  // match against class or union
  if (ptype->TypeRecord ()) {
    pinfo = ptype->TypeRecord ()->Record ();
    ainfo = atype->TypeRecord ()->Record ();
    if (pinfo && ainfo) {
      if (pinfo->isTemplateInstance () && ainfo->isTemplateInstance ()) {
        pti = pinfo->TemplateInstance (); 
        ati = ainfo->TemplateInstance (); 
        if (pti && ati && pti->DeducedArgs () == ati->DeducedArgs ()) {
          ptinfo = pti->Template ();
          atinfo = ati->Template ();
          if (! ptinfo || ! atinfo)
            return false;
          ptinfo = ptinfo->BaseTemplate () ? ptinfo->BaseTemplate () : ptinfo;
          atinfo = atinfo->BaseTemplate () ? atinfo->BaseTemplate () : atinfo;
          if (*ptinfo != *atinfo)
            return false;
          for (unsigned i = 0; i < pti->DeducedArgs (); i++) {
            type = pti->DeducedArg (i)->Type ();
            type2 = ati->DeducedArg (i)->Type ();
            value = pti->DeducedArg (i)->Value ();
            value2 = ati->DeducedArg (i)->Value ();
            if (type) {
              if (type->TypeTemplateParam () &&
                  type->TypeTemplateParam ()->isNonTypeParam ()) {
                param = type->TypeTemplateParam ()->TemplateParamInfo ();
                if (! param || (pos = getPosition (param)) == -1)
                  return false;
                if (type2 && type2->TypeTemplateParam ()) {
                  if (! type2->TypeTemplateParam ()->isNonTypeParam () ||
                      ! sameType (pos, type2))
                    return false;
                  darguments[pos] = new DeducedArgument (param, type2);
                  continue;
                } else if (! value2 || ! sameValue (pos, value2)) 
                  return false;
                darguments[pos] = new DeducedArgument (param, value2);
              } else if (! type2 || ! matchArgument (type, type2, 0))
                return false;
            } else if (! value || ! value2 || *value != *value2)
              return false;
          }
          return true;
        }
      } else if (*pinfo == *ainfo)
        return true;
    } 
    return false;
  // match against pointer to class member
  } else if (ptype->TypeMemberPointer ()) {
    prec = ptype->TypeMemberPointer ()->Record ();
    arec = atype->TypeMemberPointer ()->Record ();
    param = ptype->TypeMemberPointer ()->TemplateParam ();
    if (param) {
      if (! arec)
        return false;
      type = arec->TypeInfo ();
      pos = getPosition (param);
      if (pos == -1 || ! param->isTypeParam () || ! sameType (pos, type)) 
        return false;
      darguments[pos] = new DeducedArgument (param, type);
      return true;
    }
    if ((bool)prec != (bool)arec || *prec != *arec)
      return false;
  // match against array
  } else if (ptype->TypeArray ()) {
    pdim = ptype->TypeArray ()->Dimension ();
    adim = atype->TypeArray ()->Dimension ();
    if (ptype->TypeArray ()->DepDim ()) {
      CConstant dim ((LONG_LONG)adim, &CTYPE_LONG);
      param = ptype->TypeArray ()->DepDim ()->TemplateParamInfo ();
      if (! param || param->isTypeParam () || 
          (pos = getPosition (param)) == -1)
        return false;
      if (atype->TypeArray ()->DepDim ()) {
        param2 = atype->TypeArray ()->DepDim ()->TemplateParamInfo ();
        if (! param2 || param2->isTypeParam () || 
            ! sameType (pos, param2->TypeInfo ()))
          return false;
        darguments[pos] = new DeducedArgument (param, param2->TypeInfo ());
      } else if (sameValue (pos, &dim)) {
        darguments[pos] = new DeducedArgument (param, adim);
      } else 
        return false;
    } else if (pdim != adim)
      return false;
  // match against function
  } else if (ptype->TypeFunction ()) {
    ptl = ptype->TypeFunction ()->ArgTypes ();
    atl = atype->TypeFunction ()->ArgTypes ();
    if (ptl->Entries () != atl->Entries ())
      return false;
    for (unsigned i = 0; i < ptl->Entries (); i++)
      if (! matchArgument (ptl->Entry (i), atl->Entry (i), 0)) 
        return false;
  }
  
  // match base types
  return matchArgument (ptype->BaseType (), atype->BaseType (), aexpr);
}


// 14.5.5.2 partial ordering rules
// return 0 if equal, 1 if more, and -1 if less specialized than other
int InstantiationCandidate::compare (InstantiationCandidate &other) {
  InstantiationCandidate cand1;
  cand1.initialize (other.TemplateInfo ()->SpecializationName (), ObjectInfo (), TemplateInfo ());
  for (unsigned i = 0; i < Arguments (); i++) 
    cand1.addArgument (Argument (i));
  
  InstantiationCandidate cand2;
  cand2.initialize (TemplateInfo ()->SpecializationName (), other.ObjectInfo (), other.TemplateInfo ());
  for (unsigned i = 0; i < other.Arguments (); i++) 
    cand2.addArgument (other.Argument (i));

  // match against the other
  bool matches1 = cand1.match (); 
  bool matches2 = cand2.match (); 
  
  // equal specialized
  if (matches1 && matches2) 
    return 0;
  // more specialized
  if (matches1) 
    return -1;
  // less specialized
  if (matches2) 
    return 1;
  // equal specialized
  return 0;
}


} // namespace Puma
