// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Tracing.h"
#include "Puma/CTokens.h"
#include "Puma/Token.h"
#include "Puma/CTree.h"
#include <iostream>
using namespace std;

namespace Puma {


ostream &Tracing::tos () const {
  return *_tos;
}

void Tracing::trace (ostream &os, bool trace_tokens) { 
  _tos = &os; 
  _trace = trace_tokens ? 2 : 1; 
}

void Tracing::trace (const char *rule, int n) { 
  if (! _trace) return;
  if (n == -1) _tdepth += n; 
  for (int i = 0; i < _tdepth; i++)
    *_tos << "| ";
  *_tos << rule << endl; 
  if (n == 1) _tdepth += n; 
}

void Tracing::trace (Token *token, const char *ok, const char *failed) { 
  if (! _trace) return;
  if (token) {
    *_tos << token->location () << ": " << ok 
          << ": " << token->text () << endl;
  } else 
    *_tos << failed << endl;
}

const char *Tracing::token_text (int token_type) const { 
  switch (token_type) {
    case TOK_ADVICE: return "`advice'";
    case TOK_ASPECT: return "`aspect'";
    case TOK_SLICE: return "`slice'";
    case TOK_POINTCUT: return "`pointcut'";
    case TOK_ADD_EQ: return "`+='";
    case TOK_AND: return "`&'";
    case TOK_AND_AND: return "`&&'";
    case TOK_AND_EQ: return "`&='";
    case TOK_ASM: return "`asm' | `__asm' | `__asm__'";
    case TOK_ASSIGN: return "`='";
    case TOK_AUTO: return "`auto'";
    case TOK_BOOL: return "`bool'";
    case TOK_BOOL_VAL: return "`true' | `false'";
    case TOK_BREAK: return "`break'";
    case TOK_CASE: return "`case'";
    case TOK_CATCH: return "`catch'";
    case TOK_CHAR: return "`char'";
    case TOK_CHAR_VAL: return "CHAR";
    case TOK_CLASS: return "`class'";
    case TOK_CLOSE_CURLY: return "`}'";
    case TOK_CLOSE_ROUND: return "`)'";
    case TOK_CLOSE_SQUARE: return "`]'";
    case TOK_COLON: return "`:'";
    case TOK_COLON_COLON: return "`::'";
    case TOK_COMMA: return "`,'";
    case TOK_CONST: return "`const' | `__const' | `__const__'";
    case TOK_CONST_CAST: return "`const_cast'";
    case TOK_CONTINUE: return "`continue'";
    case TOK_DECR: return "`--'";
    case TOK_DEFAULT: return "`default'";
    case TOK_DELETE: return "`delete'";
    case TOK_DIV: return "`/'";
    case TOK_DIV_EQ: return "`/='";
    case TOK_DO: return "`do'";
    case TOK_DOT: return "`.'";
    case TOK_DOT_STAR: return "`.*'";
    case TOK_DOUBLE: return "`double'";
    case TOK_DYN_CAST: return "`dynamic_cast'";
    case TOK_ELLIPSIS: return "`...'";
    case TOK_ELSE: return "`else'";
    case TOK_ENUM: return "`enum'";
    case TOK_EQL: return "`=='";
    case TOK_EXPLICIT: return "`explicit'";
    case TOK_EXPORT: return "`export'";
    case TOK_EXTERN: return "`extern'";
    case TOK_FLOAT: return "`float'";
    case TOK_FLT_VAL: return "FLOAT";
    case TOK_FOR: return "`for'";
    case TOK_FRIEND: return "`friend'";
    case TOK_GEQ: return "`>='";
    case TOK_GOTO: return "`goto'";
    case TOK_GREATER: return "`>'";
    case TOK_ID: return "ID";
    case TOK_IF: return "`if'";
    case TOK_INCR: return "`++'";
    case TOK_INLINE: return "`inline' | `__inline' | `__inline__'";
    case TOK_INT: return "`int'";
    case TOK_INT64: return "`__int64'";
    case TOK_INT_VAL: return "INT";
    case TOK_IOR_EQ: return "`|='";
    case TOK_LEQ: return "`<='";
    case TOK_LESS: return "`<'";
    case TOK_LONG: return "`long'";
    case TOK_LSH: return "`<<'";
    case TOK_LSH_EQ: return "`<<='";
    case TOK_MINUS: return "`-'";
    case TOK_MODULO: return "`%'";
    case TOK_MOD_EQ: return "`%='";
    case TOK_MUL: return "`*'";
    case TOK_MUL_EQ: return "`*='";
    case TOK_MUTABLE: return "`mutable'";
    case TOK_NAMESPACE: return "`namespace'";
    case TOK_NEQ: return "`!='";
    case TOK_NEW: return "`new'";
    case TOK_NOT: return "`!'";
    case TOK_OPEN_CURLY: return "`{'";
    case TOK_OPEN_ROUND: return "`('";
    case TOK_OPEN_SQUARE: return "`['";
    case TOK_OPERATOR: return "`operator'";
    case TOK_OR_OR: return "`||'";
    case TOK_OR: return "`|'";
    case TOK_PLUS: return "`+'";
    case TOK_PRIVATE: return "`private'";
    case TOK_PROTECTED: return "`protected'";
    case TOK_PTS: return "`->'";
    case TOK_PTS_STAR: return "`->*'";
    case TOK_PUBLIC: return "`public'";
    case TOK_QUESTION: return "`?'";
    case TOK_REGISTER: return "`register'";
    case TOK_REINT_CAST: return "`reinterpret_cast'";
    case TOK_RESTRICT: return "`restrict'";
    case TOK_RETURN: return "`return'";
    case TOK_ROOF: return "`^'";
    case TOK_RSH: return "`>>'";
    case TOK_RSH_EQ: return "`>>='";
    case TOK_SEMI_COLON: return "`;'";
    case TOK_SHORT: return "`short'";
    case TOK_SIGNED: return "`signed' | `__signed' | `__signed__'";
    case TOK_SIZEOF: return "`sizeof'";
    case TOK_STATIC: return "`static'";
    case TOK_STAT_CAST: return "`static_cast'";
    case TOK_STRING_VAL: return "STRING";
    case TOK_STRUCT: return "`struct'";
    case TOK_SUB_EQ: return "`-='";
    case TOK_SWITCH: return "`switch'";
    case TOK_TEMPLATE: return "`template'";
    case TOK_THIS: return "`this'";
    case TOK_THROW: return "`throw'";
    case TOK_TILDE: return "`~'";
    case TOK_TRY: return "`try'";
    case TOK_TYPEDEF: return "`typedef'";
    case TOK_TYPEID: return "`typeid'";
    case TOK_TYPENAME: return "`typename'";
    case TOK_UNION: return "`union'";
    case TOK_UNSIGNED: return "`unsigned'";
    case TOK_USING: return "`using'";
    case TOK_VIRTUAL: return "`virtual'";
    case TOK_VOID: return "`void'";
    case TOK_VOLATILE: return "`volatile' | `__volatile' | `__volatile__'";
    case TOK_WCHAR_T: return "`wchar_t' | `__wchar_t'";
    case TOK_WHILE: return "`while'";
    case TOK_XOR_EQ: return "`^='";
    case TOK_ZERO_VAL: return "`0'";
    // AspectC++ specific type specifier
    case TOK_UNKNOWN_T: return "`__unknown_t'";
    // Win specific declaration specifiers
    case TOK_CDECL: return "`_cdecl' || `__cdecl'";
    case TOK_STDCALL: return "`_stdcall' || `__stdcall'";
    case TOK_FASTCALL: return "`_fastcall' || `__fastcall'";
    default: return "UNKNOWN";
  }
}


} // namespace Puma
