// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/CEnumeratorInfo.h"
#include "Puma/CTemplateParamInfo.h"
#include "Puma/CClassDatabase.h"
#include "Puma/CAttributeInfo.h"
#include "Puma/CFunctionInfo.h"
#include "Puma/CArgumentInfo.h"
#include "Puma/CTypedefInfo.h"
#include "Puma/CTemplateInstance.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CObjectInfo.h"
#include "Puma/CScopeInfo.h"
#include "Puma/CUnionInfo.h"
#include "Puma/CClassInfo.h"
#include "Puma/CFileInfo.h"
#include "Puma/CEnumInfo.h"
#include "Puma/CTypeInfo.h"
#include "Puma/CUsingInfo.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/ErrorStream.h"
#include "Puma/CProject.h"
#include "Puma/Token.h"
#include "Puma/Unit.h"
#include <iostream>
#include <string.h>
#include <stdio.h>
using namespace std;

namespace Puma {


CClassDatabase::~CClassDatabase () {
  for (int i = 0; i < _Files.length (); i++)
    delete (CFileInfo*)_Files[i];
}

void CClassDatabase::Insert (CObjectInfo *info) {
  if (! info || ! info->Name ()) return;

  if (info->ClassInfo ())
    _Classes.append (info);
  else if (info->UnionInfo ())
    _Unions.append (info);
  else if (info->EnumInfo ())
    _Enums.append (info);
  else if (info->TypedefInfo ())
    _Typedefs.append (info);
  else if (info->FunctionInfo ())
    _Functions.append (info);
  else if (info->FileInfo ())
    _Files.append (info);
  else
    return; // Unknown!
 
  info->ClassDB (this);
}
  
void CClassDatabase::Remove (CObjectInfo *info) {
  if (! info) return;
  if (info->ClassInfo ()) {
    for (long i = ClassInfos (); i > 0; i--)
      if (_Classes.fetch (i-1) == info)
        _Classes.remove (i-1);
  } else if (info->UnionInfo ()) {
    for (long i = UnionInfos (); i > 0; i--)
      if (_Unions.fetch (i-1) == info)
        _Unions.remove (i-1);
  } else if (info->EnumInfo ()) {
    for (long i = EnumInfos (); i > 0; i--)
      if (_Enums.fetch (i-1) == info)
        _Enums.remove (i-1);
  } else if (info->TypedefInfo ()) {
    for (long i = TypedefInfos (); i > 0; i--)
      if (_Typedefs.fetch (i-1) == info)
        _Typedefs.remove (i-1);
  } else if (info->FunctionInfo ()) {
    for (long i = FunctionInfos (); i > 0; i--)
      if (_Functions.fetch (i-1) == info)
        _Functions.remove (i-1);
  } else if (info->FileInfo ()) {
    for (long i = FileInfos (); i > 0; i--)
      if (_Files.fetch (i-1) == info)
        _Files.remove (i-1);
  } else 
    return; // Unknown!
}
  
CObjectInfo *CClassDatabase::ObjectInfo (Token *token) const {
  unsigned infos = ObjectInfos ();
  for (unsigned i = 0; i < infos; i++) {
    CObjectInfo *info = ObjectInfo (i);
    if (! info->FileInfo ())
      if (*info->SourceInfo () == token)
        return info;  
  }
  return (CObjectInfo*)0;
}

CObjectInfo *CClassDatabase::ObjectInfo (CT_Token *token) const {
  unsigned infos = ObjectInfos ();
  for (unsigned i = 0; i < infos; i++) {
    CObjectInfo *info = ObjectInfo (i);
    if (! info->FileInfo ())
      if (*info->SourceInfo () == token)
        return info;  
  }
  return (CObjectInfo*)0;
}

CObjectInfo *CClassDatabase::ObjectInfo (unsigned i) const { 
  if (i < ClassInfos ()) return ClassInfo (i);
  else i = i - ClassInfos ();
  if (i < UnionInfos ()) return UnionInfo (i);
  else i = i - UnionInfos ();
  if (i < EnumInfos ()) return EnumInfo (i);
  else i = i - EnumInfos ();
  if (i < TypedefInfos ()) return TypedefInfo (i);
  else i = i - TypedefInfos ();
  if (i < FunctionInfos ()) return FunctionInfo (i);
  else i = i - FunctionInfos ();
  if (i < FileInfos ()) return FileInfo (i);
  return (CObjectInfo*)0; 
}


void CClassDatabase::Dump (ostream &out, int level) const {
  unsigned files = FileInfos ();
  CStructure *file;
  int count = 0;
  
  for (unsigned i = 0; i < files; i++) {
    file = FileInfo (i);
    out << "file " << count++ << " : " << file->Name () << endl;
    Dump (out, file, 0, level);
  }
}

void CClassDatabase::Dump (ostream &out, CStructure *scope, int ind, int level) const {
  unsigned num;
  
  if (! scope || (ind / 4) > level) 
    return; 
  
  // Template parameter (print on template scope only)
  num = scope->TemplateParams ();
  if (num && scope->TemplateInfo ()) {
    indent (out, ind+2); out << num << " template parameters:" << endl;
    for (unsigned j = 0; j < num; j++)
      DumpTemplateParam (out, scope->TemplateParam (j), ind);
  }
    
  // Types
  num = scope->Types ();
  if (num) {
    indent (out, ind+2); out << num << " types:" << endl;
    for (unsigned j = 0; j < num; j++)
      DumpType (out, scope->Type (j), ind);
  }
    
  // Namespaces
  num = scope->Namespaces ();
  if (num) {
    indent (out, ind+2); out << num << " namespaces:" << endl;
    for (unsigned j = 0; j < num; j++) 
      DumpNamespace (out, scope->Namespace (j), ind);
  }

  // Usings
  num = scope->Usings ();
  if (num) {
    indent (out, ind+2); out << num << " usings:" << endl;
    for (unsigned j = 0; j < num; j++) 
      DumpUsing (out, scope->Using (j), ind);
  }

  // Attributes
  num = scope->Attributes ();
  if (num) {
    indent (out, ind+2); out << num << " attributes: " << endl;
    for (unsigned j = 0; j < num; j++)
      DumpAttribute (out, scope->Attribute (j), ind);
  }

  // Functions
  num = scope->Functions ();
  if (num) {
    indent (out, ind+2); out << num << " functions:" << endl;
    for (unsigned j = 0; j < num; j++)
      DumpFunction (out, scope->Function (j), ind, level);
  }
  
  // Local scopes
  num = scope->Children ();
  if (num) {
    indent (out, ind+2); out << num << " scopes:" << endl;
    for (unsigned j = 0; j < num; j++) {
      CStructure *info = scope->Child (j)->Structure ();
//      if (info && info->Objects () &&
//          ! (info->FunctionInfo () && ! info->FunctionInfo ()->isFctDef ())) {
        DumpLocalScope (out, info, ind);
        Dump (out, info, ind+4, level);
//      }
    }
  }
}

void CClassDatabase::DumpLocalScope (ostream &out, CObjectInfo *info, int ind) const {
  indent (out, ind+4); 
  if (info->Record () || info->NamespaceInfo () || 
      info->FunctionInfo () || info->TemplateInfo ()) {
    if (info->UnionInfo ()) 
      out << "union";
    else if (info->ClassInfo ()) 
      out << (info->ClassInfo ()->isStruct () ? "struct" : "class");
    else if (info->NamespaceInfo ())
      out << "namespace";
    else if (info->FunctionInfo ())
      out << "function";
    else if (info->TemplateInfo ())
      out << "template";
    out << " : " << info->Name ();
  } else 
    out << "block";
  out << "\t";
  if (info->ClassInfo ()) {
    unsigned base_classes = info->ClassInfo ()->BaseClasses ();
    if (base_classes) {
      out << "[base classes: ";
      for (unsigned i = 0; i < base_classes; i++) {
        CClassInfo *cinfo = info->ClassInfo ()->BaseClass (i)->Class ();
        if (cinfo && cinfo->TypeInfo ()) {
          cinfo->TypeInfo ()->TypeText (out);
          if (i+1 < base_classes)
            out << ",";
        }
      }
      out << "] ";
    }
  }
  DumpQualities (out, info);
  out << endl;
}

void CClassDatabase::DumpFunction (ostream &out, CFunctionInfo *info, int ind, int level) const {
//  CArgumentInfo* arg;
//  unsigned num;
  
  indent (out, ind+4); 
  out << info->Name () << "\t (type: ";
  info->TypeInfo ()->TypeText (out);
  out << ") ";

  if (info->ObjectInfo()->QualifiedScope () && info->ObjectInfo()->QualifiedScope () != info->Scope ()) {
    out << " [member of ";
    DumpScopeName (out, info->ObjectInfo()->QualifiedScope ());
    out << "]";
  }
  if (info->isDestructor ()) 
    out << " [destructor]";
  else if (info->isConstructor ()) 
    out << " [constructor]";
  if (! info->isDefined () && ! info->isBuiltin ())
    out << " [undefined]";
  if (info->isOperator () || info->isConversion ())
    out << " [operator]";

  DumpQualities (out, info);
  out << endl;

  if (level == 0) 
    return;
  
  // dump function arguments
//  num = info->Arguments ();
//  if (num) {
//    indent (out, ind+6); out << num << " arguments:" << endl;
//    for (unsigned i = 0; i < num; i++) {
//      arg = info->Argument (i);
//      indent (out, ind+8); out << arg->Name () << "\t (type: ";
//      arg->TypeInfo ()->TypeText (out);
//      out << ") ";
//      DumpQualities (out, arg);
//      out << endl;
//    }
//  }
}

void CClassDatabase::DumpAttribute (ostream &out, CAttributeInfo *info, int ind) const {
  indent (out, ind+4); 
  out << info->Name () << "\t (type: ";
  info->TypeInfo ()->TypeText (out);
  out << ") ";

  if (info->ObjectInfo()->QualifiedScope () && info->ObjectInfo()->QualifiedScope () != info->Scope ()) {
    out << " [member of ";
    DumpScopeName (out, info->ObjectInfo()->QualifiedScope ());
    out << "]";
  }
  if (info->TypeInfo ()->isVarArray ()) 
    out << " [dim: variable]";
  else if ((info->TypeInfo ()->isArray () || info->TypeInfo ()->isBitField ()) && 
           info->TypeInfo ()->Dimension () != -1) 
    out << " [dim: " << info->TypeInfo ()->Dimension () << "]";
  else if (info->EnumeratorInfo ()) 
    out << " [value: " << info->EnumeratorInfo ()->Value () << "]";

  DumpQualities (out, info);
  out << endl;
}

void CClassDatabase::DumpType (ostream &out, CObjectInfo *info, int ind) const {
  indent (out, ind+4); 
  if (info->TypedefInfo ()) 
    out << "typedef ";
  else if (info->UnionInfo ()) 
    out << "union ";
  else if (info->EnumInfo ()) 
    out << "enum ";
  else if (info->ClassInfo ())
    out << (info->ClassInfo ()->isStruct () ? "struct " : "class ");
  out << ": " << info->Name () << "\t";
  if (info->TypedefInfo ()) {
    out << " (type: ";
    info->TypeInfo ()->TypeText (out);
    out << ") ";
  } else if (info->ClassInfo ()) {
    unsigned base_classes = info->ClassInfo ()->BaseClasses ();
    if (base_classes) {
      out << "[base classes: ";
      for (unsigned i = 0; i < base_classes; i++) {
        CClassInfo *cinfo = info->ClassInfo ()->BaseClass (i)->Class ();
        if (cinfo && cinfo->TypeInfo ()) {
          cinfo->TypeInfo ()->TypeText (out);
          if (i+1 < base_classes)
            out << ",";
        }
      }
      out << "] ";
    }
  }
  DumpQualities (out, info);
  out << endl;
}

void CClassDatabase::DumpTemplateParam (ostream &out, CTemplateParamInfo *info, int ind) const {
  indent (out, ind+4); 
  out << info->Name () << "\t";
  if (info->isTemplate ()) 
    out << " [template] ";
  else if (info->isTypeParam ()) 
    out << " [type] ";
  else 
    out << " [non-type] ";
  out << endl;
}

void CClassDatabase::DumpNamespace (ostream &out, CNamespaceInfo *info, int ind) const {
  CNamespaceInfo *nsinfo;
  indent (out, ind+4); 
  out << "namespace : " << info->Name () << "\t";
  if (info->isAlias ()) {
    nsinfo = info;
    while (nsinfo->isAlias ()) {
      nsinfo = (CNamespaceInfo*)nsinfo->NextObject ();
      if (nsinfo == info) // should never be true
        break;
    }
    out << " [alias of ";
    DumpScopeName (out, nsinfo);
    out << "]";
  }
  DumpQualities (out, info);
  out << endl;
}

void CClassDatabase::DumpUsing (ostream &out, CUsingInfo *info, int ind) const {
  CNamespaceInfo *nsinfo = info->Namespace ();
  indent (out, ind+4); 
  out << "using namespace " << ": ";
  DumpScopeName (out, nsinfo);
  out << endl;
}

void CClassDatabase::DumpQualities (ostream &out, CObjectInfo *info) const {
  if (info->isTemplate ())
    out << " [template]";
  if (info->TemplateInfo () && ! info->TemplateInfo ()->isBaseTemplate ())
    out << " [specialization of " << (void*)info->TemplateInfo ()->BaseTemplate () << "]";
  if (info->NamespaceInfo () && info->Name () && *info->Name ().c_str() == '%')
    out << " [instance scope]";
  if (info->isTemplateInstance ()) {
    if (info->TemplateInstance ()->isPseudoInstance ())
      out << " [pseudo instance]";
    else
      out << " [instance of " 
          << (void*)info->TemplateInstance ()->Template () << "]";
  }

  // dump member access level
  if (info->Protection () == CProtection::PROT_PROTECTED) 
    out << " [protected]";
  else if (info->Protection () == CProtection::PROT_PRIVATE) 
    out << " [private]";
//  else if (info->Protection () == CProtection::PROT_PUBLIC) 
//    out << " [public]";

  // dump linkage 
  if (! (info->Language () == CLanguage::LANG_UNDEFINED)) 
    out << " [language: " << info->Language ().Text () << "]";

  // dump type qualities
  if (info->TypeInfo () && info->TypeInfo ()->isVolatile ()) 
    out << " [volatile]";
  if (info->TypeInfo () && info->TypeInfo ()->isConst ()) 
    out << " [const]";

  if (info->isBuiltin ())
    out << " [builtin]";
  if (info->isInline ()) 
    out << " [inline]";
  if (info->isAnonymous ()) 
    out << " [anonymous]";

  if (info->isVirtual ()) 
    out << " [virtual]";
  if (info->isStatic ()) 
    out << " [static]";
  if (info->isExtern ()) 
    out << " [extern]";
  if (info->isMutable ()) 
    out << " [mutable]";
  if (info->isRegister ()) 
    out << " [register]";

  // dump storage class
  if (info->Storage () == CStorage::CLASS_DYNAMIC) 
    out << " [dynamic]";
//  else if (info->Storage () == CStorage::CLASS_STATIC) 
//    out << " [static]";
//  else if (info->Storage () == CStorage::CLASS_AUTOMATIC) 
//    out << " [automatic]";

  // dump address
  out << " [" << (void*)info << "]";
  // dump next linked object
  if (info->NextObject () != info)
    out << " [next: " << (void*)info->NextObject () << "]";
}

void CClassDatabase::DumpScopeName (ostream &out, CStructure *scope) const {
  if (scope->FileInfo ())
    return;

  CStructure *parent = scope->Parent ()->Structure ();
  if (parent->isRecord () || parent->isNamespace ()) {
    DumpScopeName (out, parent);
    out << "::" << scope->Name ();
  } else
    out << scope->Name ();
}

void CClassDatabase::indent (ostream &out, int ind) const {
  for (int i = 0; i < ind; i++)
    out << " ";
}

} // namespace Puma
