// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Semantic_h__
#define __Semantic_h__

#include "Puma/CClassDatabase.h"
#include "Puma/CObjectInfo.h"
#include "Puma/ErrorSink.h"
#include "Puma/PtrStack.h"
#include "Puma/Unit.h"
#include "Puma/BCStack.h"

namespace Puma {


class CTree;
class CFileInfo;
class CEnumInfo;
class CStructure;
class CTemplateInfo;
class CSemDeclSpecs;
class CT_DeclSpecSeq;
class CSemDeclarator;

/*DEBUG*/extern int TRACE_SCOPES;

class Semantic : public PtrStack<CObjectInfo> {
protected:
  CClassDatabase *_db;
  CFileInfo *_file;
  ErrorSink *_err;
  long _Anonymous;

  CEnumInfo *current_enum; 
  CStructure *current_fct; 
  CStructure *current_scope; 

  BCStack<CSemDeclSpecs*, 256> _sem_decl_specs;
  BCStack<CT_DeclSpecSeq*, 256> _decl_specs;
  BCStack<bool, 256> _in_param_decl_clause;
  BCStack<bool, 256> _in_decl;

protected:
  Semantic () {}
  virtual ~Semantic ();
  
public:
  virtual void init (CClassDatabase &, Unit &);
  virtual void configure (Config &) {}

  void error_sink (ErrorSink &);
  void undo (CTree *);
  CTree *PrivateName (); // create private (anonymous) name

  void save_state ();
  void forget_state ();
  void restore_state ();

public:
  // Scope information
  void enter_local_scope ();
  void leave_scopes ();
  virtual void enter_param_decl_clause ();
  virtual void leave_param_decl_clause ();
  bool in_param_decl_clause ();

public:
  void finish_decl ();
  void finish_declarator ();

protected:
  void common_settings (CObjectInfo *, CTree *);
  void common_settings (CObjectInfo *, CTree *, CSemDeclarator *);
  virtual void Delete ();
/*DEBUG*/void Push (CObjectInfo*);
  
  // Declarations
  bool in_decl () const;
  void decl_begin (CSemDeclSpecs *);
  void decl_end ();
  void decl_specs_begin (CT_DeclSpecSeq *);
  void decl_specs_end ();
  
  CSemDeclSpecs *sem_decl_specs () const;
  CT_DeclSpecSeq *decl_specs () const;
  
  // Scope information
  virtual void enter_scope (CStructure *);
  virtual void reenter_scope (CStructure *);
};

inline void Semantic::save_state () 
 { New (); }
inline void Semantic::forget_state () 
 { leave_scopes (); }
inline void Semantic::restore_state () 
 { Destroy (); }

inline bool Semantic::in_decl () const 
// { return _in_decl.length () > 0 ? _in_decl.top () : false; }
 { return _in_decl.empty () ? false : _in_decl.top (); }
inline void Semantic::finish_decl ()
 { if (in_decl ()) decl_end (); }

inline void Semantic::error_sink (ErrorSink &e) 
 { _err = &e; }

inline CSemDeclSpecs *Semantic::sem_decl_specs () const
 { return _sem_decl_specs.top (); }
inline CT_DeclSpecSeq *Semantic::decl_specs () const
 { return _decl_specs.top (); }
inline void Semantic::decl_specs_begin (CT_DeclSpecSeq *dss) 
 { _decl_specs.push (dss); }
inline void Semantic::decl_specs_end () 
 { _decl_specs.pop (); }


} // namespace Puma

#endif /* __Semantic_h__ */
