// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ACAspectInfo_h__
#define __ACAspectInfo_h__

#include <set>
using std::set;

#include "Puma/CClassInfo.h"
#include "Puma/CProtection.h"
#include "Puma/ACIntroductionInfo.h"

namespace Puma {


class ACPointcutInfo;
class ACAdviceInfo;
class CFunctionInfo;

/** \file 
 *  Representation of an AspectC++ 'aspect'.
 *  \see Puma::ACAspectInfo */

/** \class ACAspectInfo ACAspectInfo.h Puma/infos/acinfos/ACAspectInfo.h
 *  An ACAspectInfo object represents an AspectC++ 'aspect' declaration. 
 *  An aspect declaration is syntactically equal to a C++ class 
 *  declaration and also parsed like a C++ class. Additionally it
 *  contains pointcut and advice declarations. */
class ACAspectInfo {
  CClassInfo *_class;
  Array<ACPointcutInfo*> _pointcuts;
  int _intro_count;
  Array<ACIntroductionInfo*> _intro_nodes;
  int _advice_count;
  Array<CT_AdviceDecl*> _advice_nodes;
  Array<CT_AdviceDecl*> _order_nodes;
  CFunctionInfo *_aspectof;
  //ThisJoinPoint _aspectof_tjp;

  // determine the set of pure virtual functions or pointcuts of a class
  static void pure_virtual_functions (CClassInfo *cls,
                                      set<CFunctionInfo*> &res);
                                      
public:
  /** Construct an ACAspectInfo object from a CClassInfo.
   *  \param node The class info of the aspect declaration. */
  ACAspectInfo (CClassInfo *node);
  /** Destroy the aspect info. */
  ~ACAspectInfo ();

  /** Get the class info of the aspect declaration. */
  CClassInfo *ClassInfo () const;
  /** Get the name of the aspect. */
  const char *name () const;

  /** Return \e true if the aspect is declared abstract. */
  bool is_abstract () const;

  /** Return the number of pointcut declarations in the aspect. */
  int PointcutInfos () const { return _pointcuts.length (); }
  /** Get the n-th pointcut declaration of the aspect. 
   *  \param n The number of the pointcut declaration. 
   *  \return The pointcut info or \c NULL if \e n is invalid. */
  ACPointcutInfo *PointcutInfo (int n) const { 
    return _pointcuts.lookup (n); 
  }
  /** Add a pointcut info to the aspect. 
   *  \param p The pointcut info. */
  void add (ACPointcutInfo *p) { _pointcuts.append (p); }

  /** Return the number of introduction advices in the aspect. */
  int IntroNodes () const { return _intro_nodes.length (); }
  /** Get the n-th introduction advice of the aspect. 
   *  \param n The number of the introduction info. 
   *  \return The introduction info or \c NULL if \e n is invalid. */
  ACIntroductionInfo *IntroNode (int n) const { 
    return _intro_nodes.lookup (n); 
  }
  /** Add an introduction advice info to the aspect. 
   *  \param node The corresponding advice declaration node. 
   *  \param tp The protection of the advice (\e public, 
   *            \e protected, \e private). */
  void addIntroNode (CT_AdviceDecl *node, CProtection::Type tp) { 
    ACIntroductionInfo *acii = new ACIntroductionInfo (node, tp);
    _intro_nodes.append (acii); 
    acii->Aspect (this);
  }
  /** Increase the introduction advice counter and return
   *  the resulting count. */
  int IntroCount () { return _intro_count++; }

  /** Return the number of advice declaration nodes in the aspect. */
  int AdviceNodes () const { return _advice_nodes.length (); }
  /** Get the n-th advice declaration node of the aspect. 
   *  \param n The number of the advice declaration.
   *  \return The advice declaration node or \c NULL if \e n is invalid. */
  CT_AdviceDecl *AdviceNode (int n) const {
    return _advice_nodes.lookup (n);
  }
  /** Add an advice declaration node to the aspect info. 
   *  \param node The advice declaration syntax tree node. */
  void addAdviceNode (CT_AdviceDecl *node) { _advice_nodes.append (node); }
  /** Increase the advice declaration counter and return
   *  the resulting count. */
  int AdviceCount () { return _advice_count++; }

  /** Return the number of order advice declaration nodes in the aspect. */
  int OrderNodes () const { return _order_nodes.length (); }
  /** Get the n-th order advice declaration node of the aspect. 
   *  \param n The number of the advice declaration.
   *  \return The order advice declaration node or \c NULL if \e n is invalid. */
  CT_AdviceDecl *OrderNode (int n) const {
    return _order_nodes.lookup (n);
  }
  /** Add an order advice declaration node to the aspect info. 
   *  \param node The order advice declaration syntax tree node. */
  void addOrderNode (CT_AdviceDecl *node) { _order_nodes.append (node); }

  /** Set the aspectOf() function declared in the aspect. 
   *  \param a The function info of the aspectOf() function. */
  void aspect_of (CFunctionInfo *a) { _aspectof = a; }
  /** Get the function info of the aspectOf() function. */
  CFunctionInfo *aspect_of () const { return _aspectof; }
};

inline ACAspectInfo::ACAspectInfo (CClassInfo *node) : 
 _class (node), _intro_count (0), _advice_count (0), _aspectof (0) {}

inline CClassInfo *ACAspectInfo::ClassInfo () const 
 { return _class; }
inline const char *ACAspectInfo::name () const 
 { return _class->Name (); }


} // namespace Puma

#endif /* __ACAspectInfo_h__ */
