// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ACAdviceInfo_h__
#define __ACAdviceInfo_h__

#include "Puma/ACTree.h"
#include "Puma/CFunctionInfo.h"

#include <sstream>
using namespace std;

namespace Puma {


class ACAspectInfo;

/** \file 
 *  Representation of an AspectC++ 'advice'.
 *  \see Puma::ACAdviceInfo */

/** \class ACAdviceInfo ACAdviceInfo.h Puma/infos/acinfos/ACAdviceInfo.h
 *  An ACAdviceInfo object represents an AspectC++ 'advice' declaration. 
 *  An advice is part of an aspect declaration. */
class ACAdviceInfo {
public:
  /** The three advice types: before, around, and after advice. */
  enum advice_type { BEFORE, AROUND, AFTER };

private:
  advice_type _type;

  CT_AdviceDecl *_def_node;
  CT_AdviceDecl *_decl_node;
  std::ostringstream _name;
  ACAspectInfo *_aspect;
  bool _inherited;
  ACAdviceInfo *_parent;
  bool _uses_aspectOf;
  bool _have_advice_func;

public:
  /** Construct an advice info object. Also constructs the internal
   *  name of an advice.
   *  \param n A consecutive number used for the unique internal
   *           name of the advice. */
  ACAdviceInfo (int n) : _def_node (0), _decl_node (0), _aspect (0), 
   _inherited (false), _parent (0), _uses_aspectOf (false), 
   _have_advice_func (false) { _name << "__advice_" << n << ends; }

  /** Get the internal name of the advice. */
  const char *name () { return _name.str ().c_str (); }
  /** Return \e true if the advice is inherited from another aspect. */
  bool is_inherited () { return _inherited; }
  /** Set the \e inherited state of the advice info. 
   *  \param i \e true if the advice is inherited from another aspect. */
  void is_inherited (bool i) { _inherited = i; }
  /** Set the parent advice. The parent advice is the advice in the 
   *  base aspect from which this advice inherits. 
   *  \param p The advice info of the parent advice. */
  void parent (ACAdviceInfo *p) { _parent = p; }
  /** Get the parent advice info. If an advice is inherited from 
   *  another aspect, this method returns the info of the corresponding
   *  advice of the base aspect. 
   *  \return The advice info or \c NULL if advice is not inherited. */
  ACAdviceInfo *parent () { return _parent; }
  /** If an advice is inherited, this method returns the info of the 
   *  advice of the aspect first defining the advice.
   *  \return The advice info or \e this if the advice is not inherited. */
  ACAdviceInfo *root () {
    ACAdviceInfo *result = this;
    while (result->is_inherited ())
      result = result->parent ();
    return result;
  }
  /** Get the type of the advice. */
  advice_type type () { return _type; }
  /** Set the syntax tree node of the definition of the advice. 
   *  \param ad The advice definition syntax tree node. */
  void def_node (CT_AdviceDecl *ad) { _def_node = ad; }
  /** Get the advice definition syntax tree node. */
  CT_AdviceDecl *def_node () { return _def_node; }
  /** Set the syntax tree node of the declaration of the advice. 
   *  \param ad The advice declaration syntax tree node. */
  void decl_node (CT_AdviceDecl *ad) {
    _decl_node = ad;
  }
  /** Get the advice declaration syntax tree node. */
  CT_AdviceDecl *decl_node () { return _decl_node; }
  /** Set the aspect containing the advice. 
   *  \param ai The aspect info. */
  void Aspect (ACAspectInfo *ai) { _aspect = ai; }
  /** Get the aspect containing the advice. */
  ACAspectInfo *Aspect () { return _aspect; }
  /** Get the function info of the advice. Advice declarations
   *  are syntactically realized as function declaration. */
  CFunctionInfo *function();
  /** Return \e true if the advice uses the aspectOf() function. */
  bool uses_aspectOf () const {return _uses_aspectOf;}
  /** Set the state whether the advice uses the aspectOf() function. 
   *  \param u \e true if the advice uses the aspectOf() function. */
  void uses_aspectOf (bool u) { _uses_aspectOf = u;}
};


} // namespace Puma

#endif /* __ACAdviceInfo_h__ */
