#include <fcntl.h>
#include <unistd.h>

#include <asd.h>
#include <source-default.h>
#include <sample-convert.h>

SourceDefaultPrivate* source_default_alloc_private(gchar *name)
{
  SourceDefaultPrivate *p;

  g_assert(p = g_new0(SourceDefaultPrivate, 1));
  p->filename = name ? g_strndup(name, PATH_MAX) : NULL;
  p->is_open = FALSE;
  p->buffer = NULL;
  p->buffer_size = 0;
  p->error = FALSE;
  return p;
}

gboolean source_default_open_impl(Source *s)
{
  SourceDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);
  g_assert(p->filename && !p->is_open);

  g_assert(s->mode == SOURCE_DISABLED);

  if ((p->is_open = (p->fd = open(p->filename, O_RDONLY)) >= 0))
    {
      g_message("Source '%s' successfully opened.", s->shortname);
      
      if (!sample_type_equal(&s->sample_type, &default_sample_type))
	{
	  p->buffer_size = sample_convert_length(&default_sample_type, &s->sample_type, default_block_size, FALSE);
	  p->buffer = g_new(guint8, p->buffer_size);
	}
    }
  else
    g_message("Source '%s' open failed.", s->shortname);

  source_set_mode(s, p->is_open ? SOURCE_RUNNING : SOURCE_DISABLED);

  return p->is_open;
}

void source_default_close_impl(Source *s)
{
  SourceDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);
  
  if (p->is_open)
    {
      close(p->fd);
      p->is_open = FALSE;  
      g_message("Source '%s' closed.", s->shortname);
    }

  source_set_mode(s, SOURCE_DISABLED);
}

gboolean source_default_reopen_impl(Source *s)
{
  g_assert(s);
  if (s->close)
    s->close(s);
  
  if (s->open)
    return s->open(s);

  return FALSE;
}

glong source_default_read_impl(Source *s, guchar*d, gulong l)
{
  gulong i = 0;
  SourceDefaultPrivate *p;
  g_assert(s && d && (l > 0));
  g_assert(p = (SourceDefaultPrivate*) s->private_data);

  if (p->error)
    return -1;
  
  while (i < l)
    {
      ssize_t r = read(p->fd, d, l-i);
      
      if (r <= 0)
	{
	  if (s->reopen)
	    if (s->reopen(s))
	      continue;

	  p->error = TRUE;
	  break;
    	}
      
      d += r;
      i += r;
    }

  throughput_measure(s->throughput, i);

  return i;
}

Block* source_default_read_block_impl(Source *s)
{
  SourceDefaultPrivate *p;
  Block *b;
  gboolean ok = TRUE;

  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);

  b = block_new();
  pthread_cleanup_push(gc_ref_dec, b);

  g_assert(s->read);
  
  if (p->buffer) // Conversion needed
    {
      glong l = s->read(s, p->buffer, p->buffer_size);
      if (l > 0)
	{
	  sample_convert_copy(&s->sample_type, p->buffer, l, &default_sample_type, block_get_data(b), FALSE);
	  block_set_size(b, sample_convert_length(&s->sample_type, &default_sample_type, l, FALSE));
	}
      else
	ok = FALSE;
    }
  else
    {
      glong l = s->read(s, block_get_data(b), default_block_size);
      if (l > 0)
	block_set_size(b, l);
      else
	ok = FALSE;
    }

  pthread_cleanup_pop(0);
  
  if (!ok)
    {
      gc_ref_dec(b);
      return FALSE;
    }

  return b;
}

void source_default_free_private_impl(Source *s)
{
  SourceDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);

  g_free(p->filename);
  g_free(p->buffer);
  g_free(p);
  s->private_data = NULL;
}

void  source_default_save_private_impl(Source *s, xmlNodePtr node)
{
  g_error("Save not yet implemented.");
}
