#include <unistd.h>

#include <sample-convert.h>

#define FREQ_NEAR(f1,f2) (((f1)>(f2)*0.9)&&((f1)<(f2)*1.1)&&((f2)>(f1)*0.9)&&((f2)<(f1)*1.1))

static GStaticPrivate _buffer_private = G_STATIC_PRIVATE_INIT;

gboolean sample_convert_inplace_possible(SampleType *f, SampleType *t, gboolean h)
{
  return sample_type_sample_size(f) == sample_type_sample_size(t) && 
    ((h && (f->rate == t->rate)) || 
     (!h  && FREQ_NEAR(f->rate, t->rate)));
}

gulong sample_convert_length(SampleType *f, SampleType *t, gulong l, gboolean h)
{
  if ((h && (f->rate == t->rate)) | (!h && FREQ_NEAR(f->rate, t->rate)))
    return l*sample_type_sample_size(t)*sample_type_sample_size(f);
  else
    return l*sample_type_sample_size(t)*t->rate/sample_type_sample_size(f)/f->rate;
}

// Converts anything to 32/Unsigned/LE
inline static guint32 _convert_channel_from(SampleType *t, guint8 *p)
{
  guint32 r = 0;

  switch (t->bits)
    {
    case 8: // 8Bit
      if (t->sign)
        r = *((gint8*)p) ^ 0x80;
      else
        r = *((guint8*)p);

      r = r << 24;
      break;

    case 16: // 16Bit
      if (t->sign && t->be)
	r = (guint32) GINT16_FROM_BE(*((gint16*)p)) ^ 0x8000;
      else if (t->sign)
	r = (guint32) GINT16_FROM_LE(*((gint16*)p)) ^ 0x8000;
      else if (t->be)
	r = GUINT16_FROM_BE(*((guint16*)p));
      else
	r = GUINT16_FROM_LE(*((guint16*)p));
	
      r = r << 16;
      break;

    case 32: // 32Bit
      if (t->sign && t->be)
	r = (guint32) GINT32_FROM_BE(*((gint32*)p)) ^ 0x80000000;
      else if (t->sign)
	r = (guint32) GINT32_FROM_LE(*((gint32*)p)) ^ 0x80000000;
      else if (t->be)
	r = GUINT32_FROM_BE(*((guint32*)p));
      else
	r = GUINT32_FROM_LE(*((guint32*)p));
      break;

    default:
      g_error("%i bit not supported (yet?).", t->bits);
      break;
    }
      
   return r;
}

// Converts 32/Unsigned/LE from anything 
inline static void _convert_channel_to(SampleType *t, guint32 r, guint8 *p)
{
  switch (t->bits)
    {
    case 8: // 8Bit
      r = r >> 24;

      if (t->sign)
	*((gint8*) p) = (gint8) (r ^ 0x80);
      else
	*((guint8*) p) = (guint8) r;
      break;

    case 16: // 16 Bit
      r = r >> 16;
      
      if (t->sign && t->be)
	*((gint16*) p) = GINT16_TO_BE((gint16) (r ^ 0x8000));
      else if (t->sign)
	*((gint16*) p) = GINT16_TO_LE((gint16) (r ^ 0x8000));
      else if (t->be)
	*((guint16*) p) = GUINT16_TO_BE((guint16) r);
      else 
	*((guint16*) p) = GUINT16_TO_LE((guint16) r);
      break;

    case 32: // 32 Bit
      if (t->sign && t->be)
	*((gint32*) p) = GINT32_TO_BE((gint32) (r ^ 0x80000000));
      else if (t->sign)
	*((gint32*) p) = GINT32_TO_LE((gint32) (r ^ 0x80000000));
      else if (t->be)
	*((guint32*) p) = GUINT32_TO_BE((guint32) r);
      else 
	*((guint32*) p) = GUINT32_TO_LE((guint32) r);
      break;

    default:
      g_error("%i bit not supported (yet?).", t->bits);
      break;
    }
}

inline static void _convert_sample(SampleType *t1, guint l1, guint8 *d1, SampleType *t2, guint l2, guint8 *d2)
{
  if (t1->channels == t2->channels)
    {
      guint i;
      for (i = 0; i < t1->channels; i++)
	{
	  _convert_channel_to(t2, _convert_channel_from(t1, d1), d2);
	  d1 += l1;
	  d2 += l2;
	}
    }
  else if (t1->channels < t2->channels)
    {
      guint i;

      for (i = 0; i < t2->channels; i++)
	{
	  guchar *d;
          guint t = i*t1->channels/t2->channels;

	  d = d1 + t*l1;
	  _convert_channel_to(t2, _convert_channel_from(t1, d), d2);
	  d2 += l2;
	}
    }
  else
    g_error("Decreasing of channels not supported yet.");
}

/* void _hd(guint8* d, guint l) */
/* { */
/*   guint i; */

/*   for (i = 0; i < l; i++) */
/*     { */
/*       g_print("%02x ", *d); */
/*       if ((i & 0xF) == 0xF) g_print("\n"); */
/*       if ((i & 0xF) == 0x7) g_print(" "); */

/*       d++; */
/*     } */
/* } */


static void _resample(guint8*d1, gulong l1, guint8*d2, gulong l2, guint s)
{
  gulong i;
  g_assert(l1 != l2);

  for (i = 0; i < l2/s; i++)
    {
      guint t = i*l1/l2;

      memcpy(d2, d1 + t*s, s);
      d2 += s;       
    }
}

// Converts data from one format to another. no copying is done.
// Input:
//               t1: source sample type
//               d:  source and destination data
//               l1: data length
//               t2: destination sample type

void sample_convert_inplace(SampleType *t1, guint8* d, gulong l, SampleType *t2)
{
  guint32 s;
  guint i, sc1, sc2;
  
  g_assert(t1 && d && l && t2);

  if (sample_type_equal(t1, t2))
    return;

  s = sample_type_sample_size(t1);                 // Bytes per sample
  g_assert(s == sample_type_sample_size(t2));

  g_assert((t1->rate == t2->rate) || FREQ_NEAR(t1->rate, t2->rate)); // We won't do resampling

  sc1 = s/t1->channels;
  sc2 = s/t2->channels;

  for (i = 0; i < l; i += s)
    {
      _convert_sample(t1, sc1, d, t2, sc2, d);
      d += s;
    }
}


static guint8* _buffer(gulong l)
{
  guint8* d;

  if ((d = g_static_private_get(&_buffer_private)))
    if (*((guint32*)d) < l)
      {
        g_free(d);
        d = NULL;
      }
  
  if (!d)
    {
      d = g_new(guint8, l+sizeof(guint32));
      *((guint32*)d) = l;
      g_static_private_set(&_buffer_private, d, g_free);
    }

  return d+sizeof(guint32);
}


// Converts sample data from one format to another. Data is copied from memory area to another
// Input:
//              t1: source sample type
//              d1: source data
//              l1: source data length
//              t2: destination sample type
//              d2: destiantion data
//               h: Be hard on resampling (otherwise ignore 10% rate difference)
void sample_convert_copy(SampleType *t1, guint8 *d1, gulong l1, SampleType *t2, guint8 *d2, gboolean h)
{
  guint32 s1, s2;
  guint32 sc1, sc2;
  guint i;
  guint8 *buf, *b;
  gulong l2, bl;
  gboolean rs;

  g_assert(t1 && d1 && l1 && t2 && d2);

  s1 = sample_type_sample_size(t1);                 // Bytes per sample
  s2 = sample_type_sample_size(t2);

  sc1 = s1/t1->channels;                            // Bytes per channel
  sc2 = s2/t2->channels;

  l2 = sample_convert_length(t1, t2, l1, h);        // Destination buffer length
  bl = l1*s2/s1;                                    // Middle buffer length

  buf = _buffer(bl);

  rs = !((h && (t1->rate == t2->rate)) || (!h && FREQ_NEAR(t1->rate, t2->rate)));  // Resample?

  b = rs ? buf : d2;
  
  for (i = 0; i < l1; i += s1)
    {
      _convert_sample(t1, sc1, d1, t2, sc2, b);
      d1 += s1;
      b += s2;
    }

  if (rs)
    _resample(buf, bl, d2, l2, s2);
}

