#ifndef blockqueueh
#define blockqueueh

#include <glib.h>

#include <notify.h>

#include "block.h"

struct _BlockQueue;
typedef struct _BlockQueue BlockQueue;

typedef void (*BlockQueueNotifyProc) (BlockQueue *b, gpointer*userdata);

struct _BlockQueue
{
  GMutex *mutex;      // Lock all access

  guint size;         // Allocated entries in 'blocks[]'
  Block **blocks;     // A pointer to the first entry in queue

  guint push_index;   // An index to the next entry to be filled
  guint pop_index;    // An index to the next entry to be returned
  gboolean empty;     // A flag, which shows whether the queue is empty

  guint hold;         // The queue won't return blocks before a certain amount of blocks are in queue
  gboolean streaming; // The queue will continue to return blocks when this flag is set, regardless of 'hold'
  //  GTimeVal timeout;   // When this timeout is reached the queue will also return blocks, regardless of 'hold'

  gboolean push_wait;    // Shall we wait on push until item can be inserted?
  gboolean pop_wait;     // Shall we wait on pop until item can be inserted?
  gboolean pop_enabled;  // Someone is periodically popping blocks from the queue
  gboolean push_enabled; // Someone is periodically popping blocks from the queue

  Notify pop_notify;  // Notify when a block was popped from the queue
  Notify push_notify; // Notify when a block was pushed to the queue

  Notify *user_pop_notify;  // The same, user defineable
  Notify *user_push_notify; // The same, user defineable
};

// The constructor. Destruction is done with gc
BlockQueue* block_queue_new(guint l, guint hold);

// Dequeue a block
Block *block_queue_pop(BlockQueue *b);

// Enqueue a block. Returns TRUE when insertion was successful
gboolean block_queue_push(BlockQueue *b, Block *);

// Set the internal queue flags
void block_queue_set_hold(BlockQueue *b, guint hold);
void block_queue_set_push_wait(BlockQueue *b, gboolean push_wait);
void block_queue_set_pop_wait(BlockQueue *b, gboolean pop_wait);
void block_queue_set_pop_enabled(BlockQueue *b, gboolean e);
void block_queue_set_push_enabled(BlockQueue *b, gboolean e);

// Clear the queue
void block_queue_empty(BlockQueue *b);

// Return the number of blocks in queue
// You cannot not always rely on this
guint block_queue_length(BlockQueue *b);

// Is the queue empty?
gboolean block_queue_is_empty(BlockQueue *b);

// Resize the queue
void block_queue_resize(BlockQueue *b, guint s);

// Set notify handlers
void block_queue_set_push_notify(BlockQueue *b, Notify *notify);
void block_queue_set_pop_notify(BlockQueue *b, Notify *notify);

void block_queue_wait_until_empty(BlockQueue *b);

#endif
