#include <glib.h>
#include <string.h>

#include <recursive-lock.h>
#include <volume.h>

#include <block.h>
#include <asd.h>

static GStaticMutex _blocks_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _blocks_private = G_STATIC_PRIVATE_INIT;

#define _blocks_lock() recursive_lock(&_blocks_mutex, &_blocks_private)
#define _blocks_unlock() recursive_unlock(&_blocks_mutex, &_blocks_private)

static GMemChunk *_blocks = NULL;

guint block_count = 0;
Block* silence_block = NULL;

void block_init()
{
  gint a;

  a = sizeof(Block) + default_block_size;
  _blocks = g_mem_chunk_new("audioblocks", a, a*10, G_ALLOC_AND_FREE);

  silence_block = block_new();
  silence_block->size = default_block_size;
  silence_block->dynamic = FALSE;
  memset(block_get_data(silence_block), 0, default_block_size);
}

void block_done()
{
  gc_ref_dec(silence_block);

  g_mem_chunk_destroy(_blocks);

  if (block_count)
    g_message("WARNING: %i blocks still existing on exit.", block_count);
}

static void _block_free(gpointer p);

Block* block_new() 
{
  Block *b;
  
  _blocks_lock();
  g_assert(b = g_mem_chunk_alloc(_blocks));
  block_count++;
  _blocks_unlock();

  origin_reset(&b->origin);
  latency_reset(&b->latency);

  b->dynamic = TRUE;
  b->size = 0;

  gc_add(b, _block_free, "block");

  return b;
}

static void _block_free(gpointer p)
{
  Block *b;
  g_assert(b = (Block*) p);

  _blocks_lock();
  g_mem_chunk_free(_blocks, b);
  block_count--;
  _blocks_unlock();
}


Block* block_copy(Block *b)
{
  Block *newb;
  g_assert(b);

  newb = block_new();

  newb->size = b->size;
  newb->origin = b->origin;
  newb->latency = b->latency;
  newb->dynamic = b->dynamic;

  g_memmove(block_get_data(newb), block_get_data(b), b->size);

  return newb;
}

Block* block_mutable(Block *b)
{
  Block *n;
  g_assert(b);

  if (gc_ref_single(b))
    return b;

  n = block_copy(b);
  gc_ref_dec(b);
  return n;
}

Block* block_volume(Block *b, Volume *v)
{
  guint16 l, r;
  gulong i, j;
  gint16* p;

  g_assert(b && v);

  if ((b == silence_block) || volume_is_max(v))
    return b;

  if (volume_is_muted(v))
    {
      gc_ref_dec(b);
      return gc_ref_inc(silence_block);
    }

  g_assert(default_sample_type.channels == 2);
  g_assert(default_sample_type.bits == 16);
  g_assert(!default_sample_type.be);
  g_assert(default_sample_type.sign);

  b = block_mutable(b);

  l = v->factor[0];
  r = v->factor[1];

  j = b->size/2;
  p = (gint16*) block_get_data(b);
  for (i = 0; i < j; i+=2)
    {
      p[i]   = GINT16_TO_LE((gint16) ((((gdouble) GINT16_FROM_LE(p[i]))  *l)/0xFFFF));
      p[i+1] = GINT16_TO_LE((gint16) ((((gdouble) GINT16_FROM_LE(p[i+1]))*r)/0xFFFF));
    }

  return b;
}

Block* block_pad(Block *b)
{
  g_assert(b);
  
  if (b->size != default_block_size)
    {
      guchar c;

      if (default_sample_type.sign)
	c = 0;
      else 
	c = 0x7F;
	
      memset(block_get_data(b)+b->size, c, default_block_size - b->size);
    }

  return b;
}

Block* block_mix_array(Block *blocks[], guint l)
{
  Block *result = NULL;

  g_assert(blocks && (l > 0));
  
  g_assert(default_sample_type.bits == 16);
  g_assert(default_sample_type.sign);
  g_assert(!default_sample_type.be);

  if (l == 1)
    result = blocks[0];
  else if (l == 2)
    {
      guint i;
      gint16 *data2, *rdata;
      Block *block2;
      gulong size, samples;

      result = block_pad(block_mutable(blocks[0]));
      block2 =  block_pad(blocks[1]);

      size =  MAX(result->size, block2->size);
      samples = size/2; // 16bit

      rdata = (gint16*) block_get_data(result);
      data2 = (gint16*) block_get_data(block2);

      for (i = 0; i < samples; i++)
	rdata[i] = GINT16_TO_LE((gint16) CLAMP(((gint32)GINT16_FROM_LE(rdata[i])+
						(gint32)GINT16_FROM_LE(data2[i])), 
					       -32768, 32767));

      origin_merge(&result->origin, &block2->origin);
      latency_merge(&result->latency, &block2->latency);
      gc_ref_dec(block2);

      result->size = size;
    }
  else 
    {
      guint i, j;
      gint16* data[ASD_MIX_BLOCKS_MAX];
      gint16* rdata;
      guint samples = default_block_size/2; //16bit
      
      g_assert(l <= ASD_MIX_BLOCKS_MAX);
     
      result = block_pad(block_mutable(blocks[0]));
      rdata = (gint16*) block_get_data(result);
      
      for (j = 1; j < l; j++)
	data[j] = (gint16*) block_get_data(block_pad(blocks[j]));
      
      for (i = 0; i < samples; i++)
	{
	  gint32 sum;
	  
	  sum = GINT16_FROM_LE(rdata[i]);
	  for (j = 1; j < l; j++)
	    sum += GINT16_FROM_LE(data[j][i]);
	  
	  rdata[i] = GINT16_TO_LE((gint16) CLAMP(sum, -32768, 32767));
	}

      for (j = 1; j < l; j++)
        {
          origin_merge(&result->origin, &blocks[j]->origin);
          latency_merge(&result->latency, &blocks[j]->latency);
          gc_ref_dec(blocks[j]);
        }

      result->size = default_block_size;
    }

  return result;
}


void block_set_size(Block* b, gulong l) 
{ 
  g_assert(b && l > 0); 
  b->size = l; 
}
