/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __CDINFO_H__
#define __CDINFO_H__

#include <qvaluelist.h>
#include <qobject.h>
#include <qstring.h>

/*========================================================*/

class ArsonCdInfo : public QObject
{
	Q_OBJECT

public:
	ArsonCdInfo (QObject *parent = NULL, const char *name = NULL);
	~ArsonCdInfo (void);

	bool variousArtistDisk (void) const { return m_artist.isEmpty(); }
	bool valid (void) const { return !m_title.isEmpty(); }
	const QString &artist (void) const { return m_artist; }
	const QString &title (void) const { return m_title; }
	const QString &year (void) const { return m_year; }
	const QString &genre (void) const { return m_genre; }
	const QString &cddbID (void) const { return m_cddbID; }
	const QString &cdIndexID (void) const { return m_cdIndexID; }
	const QString &categ (void) const { return m_categ; }
	const QString &playOrder (void) const { return m_playorder; }
	const QString &comment (void) const { return m_ext; }
	uint leadout (void) const { return m_leadout; }
	int trackCount (void) const { return m_tracks.count(); }

	void setArtist (const QString &str) { m_artist = str; }
	void setTitle (const QString &str) { m_title = str; }
	void setCddbID (const QString &str) { m_cddbID = str; }
	void setCdIndexID (const QString &str) { m_cdIndexID = str; }
	void setLeadout (uint lo) { m_leadout = lo; }
	void setYear (const QString &year) { m_year = year; }
	void setGenre (const QString &genre) { m_genre = genre; }
	void setCateg (const QString &cat) { m_categ = cat; }
	void setPlayOrder (const QString &po) { m_playorder = po; }
	void setComment (const QString &cmt) { m_ext = cmt; }

	bool formatTrackName (int track, const QString &fmt, QString &result) const;
	uint totalSeconds (void) const;

	void merge (const ArsonCdInfo &info);
	void clear (void);

	virtual void edit (QWidget *parent, const char *name = NULL);

	struct Track
	{
		Track (void);

		struct Time {
			Time (int seconds = 0) : fract(0) { min = seconds / 60; sec = seconds % 60; }
			Time (int m, int s, int f = 0) : min(m), sec(s), fract(f) {}

			QString display (void) const { return QString().sprintf("%d:%02d.%02d", min, sec, fract); }
			int totalSeconds (void) const { return (min * 60) + sec + ((fract >= 50) ? 1 : 0); }

			int min;
			uchar sec;
			uchar fract;
		};

		void setArtist (const QString &str) { m_artist = str; }
		void setTitle (const QString &str) { m_title = str; }
		void setOffset (uint offset) { m_offset = offset; }
		void setComment (const QString &cmt) { m_ext = cmt; }
		void setTime (const Time &t) { m_time = t; }
		void setTrackNo (int track) { m_trackNo = track; }

		const QString &artist (const ArsonCdInfo *parent) const
		{	return m_artist.isEmpty() ? parent->artist() : m_artist; }
		const QString &title (void) const { return m_title; }
		const QString &comment (void) const { return m_ext; }
		const Time &time (void) const { return m_time; }
		int trackNo (void) const { return m_trackNo; }
		uint offset (void) const { return m_offset; }

		bool formatTrackName (const ArsonCdInfo *parent,
			const QString &fmt, QString &result) const;
		
	private:
		QString m_artist;	//	For multi-artist CDs
		QString m_title;
		QString m_ext;		//	EXTT
		uint m_offset;
		int m_trackNo;
		Time m_time;
	};

	Track &addTrack (const Track &track);

	const Track &track (int index) const
	{ return m_tracks[index]; }
	Track &track (int index)
	{ return m_tracks[index]; }

	typedef QValueList<Track> TRACKS;

	const TRACKS &tracks (void) const { return m_tracks; }
	void setTracks (const TRACKS &tk) { m_tracks = tk; }
	
#ifdef ARSONDBG
	void dump (void) const;
#endif	//	ARSONDBG

private:
	ArsonCdInfo (const ArsonCdInfo &info) { }
	ArsonCdInfo &operator= (const ArsonCdInfo &info) { return *this; }

	QString m_title;
	QString m_artist;		//	For single artist CDs
	QString m_cddbID;
	QString m_cdIndexID;
	QString m_year;
	QString m_genre;
	QString m_categ;		//	Freedb category
	QString m_ext;			//	EXTD
	QString m_playorder;
	uint m_leadout;
	TRACKS m_tracks;

public slots:
	void edit (void);

signals:
	void changed (void);
};

/*========================================================*/
#endif	/*	__CDINFO_H__	*/
