/* $Id: ArkWorldUpd.h,v 1.6 2002/11/16 16:40:02 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_WORLDUPDATER_H
#define ARK_WORLDUPDATER_H

#include <Ark/ArkWorld.h>

namespace Ark
{
   /**
    * When bound to a world, an updater is used to er... update it ! In fact,
    * WorldUpdater is an abstract class and the real task is done by its 
    * descendant. It can either do AI computations, to move entities (eg.
    * on server side, or in a single-player client), or it can connect to
    * a remote host to request an up-to-date state of the world each frame
    * (eg a multiplayer client).
    */
   class ARK_DLL_API WorldUpdater
   {
      public:
	 /// Create an empty world updater.
	 WorldUpdater() {}

	 /** Destroy the world updater. */
	 virtual ~WorldUpdater() {};


	 /** Load the world whose path is \c name.
	  */
	 virtual void LoadWorld (const String &name) = 0;

	 /**
	  * Return the previously loaded world, or nil if no world has
	  * been loaded.
	  */
	 virtual World *GetWorld() = 0;

	 /** Try to 'login' an entity. This means that the world part in
	  * which this entity lay is loaded, then all entity states are
	  * loaded ; finally the entity is returned. The \c password might
	  * be used for a networked game.
	  */
	 virtual Entity *Login (const String &login,
				const String &password = "") = 0;

	 /** This function updates the world, the position of the entities,
	  * the particle systems and so on. AI computations may also be
	  * done, or the updater might connect to remote system to have an
	  * up-to-date version of the world.
	  */
	 virtual bool Update (scalar delta_time) = 0;

      protected:
	 friend class World;
	 virtual void HintEntityRemoved (int entityID) = 0;
	 virtual void HintEntityAdded (int entityID) = 0;

      private:
	 /// Copy is disabled.
	 WorldUpdater &operator = (const WorldUpdater &WorldUpdater);

	 /// Copy is disabled.
	 WorldUpdater (const WorldUpdater &WorldUpdater);
   };

   class ARK_DLL_API WorldUpdaterFactory : public Factory
   {
      public:
         WorldUpdaterFactory();
	 virtual ~WorldUpdaterFactory(){}
	 virtual WorldUpdater *NewWorldUpdater(Cache *cache,
					       bool isServer = false) = 0;
	 static WorldUpdater *CreateWorldUpdater(const String &impl,
						 Cache *cache,
						 bool isServer = false,
						 FactoryList *faclist=0);
   };

}

#endif
