/* $Id: ArkWorld.cpp,v 1.37 2003/03/15 17:49:04 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkFactory.h>
#include <Ark/ArkLight.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkWorld.h>
#include <Ark/ArkWorldUpd.h>

#include <iostream>
#include <algorithm>


namespace Ark
{

   World::World (Cache *cache, WorldUpdater *updater) :
      m_Cache (cache),
      m_Updater (updater),
      m_Timer(),
      m_CurID (0),
      m_Quit (false)
   {
      assert (cache != 0);
   }


   // Delete all entities in the world.
   World::~World ()
   {
      m_Quit = true;

      EntityList::iterator i;
      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 Entity *e = (*i);
	 delete e;
      }
   }

   // Updates the entire world : calls the update function for each entities,
   // then do collision detection for all the world and send events to the
   // corresponding entities.
   // It also removes entities that are "DEAD" flagged.
   void World::Update (scalar delta)
   {
      // FIXME: if (delta == 0.0) delta = m_Timer.GetDelta();

      EntityList attached;

      // Update each entity
      EntityList::iterator i;
      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 if ((*i)->IsAttached())
	 {
	    attached.push_back(*i);
	    continue;
	 }

	 (*i)->Update (delta);
      }

      for (i = attached.begin(); i != attached.end(); ++i)
	 (*i)->Update (delta);

      // Detect collisions as the entities have moved..
      DetectCollisions();

      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 if (!((*i)->m_Flags & Entity::STATIC))
	 {
	     EntityCollisionListIterator it;
	    for (it=(*i)->m_Collisions.begin() ; it != (*i)->m_Collisions.end() ; ++it)
	    {
	       if (it->m_Potential)
	       {
		  (*i)->m_Path.SetLastCollision(*it);
		  break;
	       }
	    }
	 }
      }      

      // We should remove entities flagged as dead.
      for (i = m_Entities.begin(); i != m_Entities.end(); )
      {
	 if ((*i)->m_Flags & Entity::DEAD)
	 {
	    // Be careful about destroying the entity and removing it from the
	    // list. To be sure not to forget anything, have a look at 
	    // Entity::~Entity and World::Remove.
	    Entity *e = *i;
	    delete e;
	
	    // valid way to keep valid iterator while erasing
	    // for sequential containers
	    i = m_Entities.erase (i);
	 }
	 else
	 {
	    // we do not erase the element, incrementing iterator
	    ++i;
	 }
      }

      m_Timer.Update();
   }

   // Add an entity to the world.
   void
   World::Add (Entity *entity)
   {
      if (m_Quit)
	 return;

      assert (entity != 0);

      entity->m_World = this;
      
      if (entity->m_ID == 0)
	 entity->m_ID = ++m_CurID;
      else
		  m_CurID = std::max(entity->m_ID+1, m_CurID);

      // Worldwide entity
      m_Entities.push_back (entity);

      entity->m_Flags |= Entity::INWORLD;

      // Send an hint to our friend the World updater..
      if (m_Updater) m_Updater->HintEntityAdded (entity->m_ID);

   }

   // Add a light to the world.
   void
   World::Add (const Light &aLight)
   {
      // Worldwide light
      m_Lights.push_back (aLight);
   }

   LightList &
   World::GetLights()
   {
      return m_Lights;
   }

   // Remove an entity from the world
   void
   World::Remove (Entity *entity)
   {
      if (m_Quit)
	 return;

      if (entity == 0 || !(entity->m_Flags & Entity::INWORLD))
	 return;
      
      // Send an hint to our friend the World updater..
      if (m_Updater) m_Updater->HintEntityRemoved (entity->m_ID);

      // Remove entity from list
      if (!(entity->m_Flags & Entity::DEAD))
      {
	 // NB that if the flag DEAD is set, the entity has been destroyed
	 // by World::Update and is already removed.
	 m_Entities.erase
	    (std::remove(m_Entities.begin(), m_Entities.end(), entity), 
	     m_Entities.end() );
      }

      // Unlink it
      entity->m_World = 0;
      entity->m_Flags &= ~Entity::INWORLD;
   }

   /// Find the entity whose id is 'id'
   Entity *
   World::Find (int id) const
   {
      if (id == 0) return 0;

      EntityList::const_iterator i;

      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 if ((*i)->m_ID == id)
	    return *i;
      }

      return 0;
   }

   Entity *
   World::FindByName (const String &name) const
   {
      EntityList::const_iterator i;

      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 if ((*i)->m_ShortName == name)
	    return *i;
      }

      return 0;
   }

   EntityList*
   World::FindByClass (const String& className) const
   {
      EntityList::const_iterator i;
      EntityList* foundEntities = new EntityList;

#ifdef DEBUG
      assert(foundEntities);
#endif
      for (i = m_Entities.begin(); i != m_Entities.end(); ++i)
      {
	 if ( (*i)->m_Class == className )
	 {
	   foundEntities->push_back(*i);
	 }
      }

      return foundEntities;
   }

   ///////////////////////////////////////////////////////////////////

   VersionInfo World_VERSION = {"World", 0,1,0};
   WorldFactory::WorldFactory() : Factory(World_VERSION)
   {}

   World *
   WorldFactory::CreateWorld(const String &implname,
			     Cache *cache, WorldUpdater *updater,
			     FactoryList *faclist)
   {
      if (faclist == 0)
	 faclist = Ark::Sys()->Factories();
      
      Ark::WorldFactory *worldfac;
      if (!faclist->GetFactory(implname, &worldfac, World_VERSION))
	 Ark::Sys()->Fatal("Cannot create a world factory !");
      
      return worldfac->NewWorld (cache, updater);
   }

}
