/* $Id: ArkModel.cpp,v 1.46 2003/03/19 09:25:05 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sstream>

#include <Ark/ArkModel.h>
#include <Ark/ArkModelState.h>
#include <Ark/ArkModelBuild.h>
#include <Ark/ArkTexture.h>
#include <Ark/ArkRenderer.h>

namespace Ark
{

   Model::Model (const String &name) :
      Object (name, V_MODEL),
      m_Skeleton(NULL),
      m_Skin ("default"),
      m_CDModel (NULL),
      m_Optimized (false)
   {
      m_SharedCDModel = NULL;
   }

   Model::~Model()
   {
      if (m_SharedCDModel)
	 m_SharedCDModel->Unref();
      else if (m_CDModel)
	 delete m_CDModel;
   }

   BoneController *
   Model::GetController (int idx)
   {
      if (m_Skeleton == NULL)
	 return NULL;

      for (size_t i = 0; i < m_Skeleton->m_Controllers.size(); i++)
      {
	 if (m_Skeleton->m_Controllers[i].m_Index == idx)
	    return &m_Skeleton->m_Controllers[i];
      }
   
      return NULL;
   }

   /**
    * This function creates the collision model after the model has 
    * been completely loaded. 
    */
   bool
   Model::PostLoad (Cache* cache)
   {
      ColSystem *sys = cache->GetColSystem();

      if (!m_SharedCDModelName.empty())
      {
	 std::ostringstream os;
	 os << m_SharedCDModelName
	    << "?" << (m_BBox.m_Max.Y - m_BBox.m_Min.Y);

	 cache->Get(V_MODEL, os.str(), &m_SharedCDModel);
	 m_CDModel = m_SharedCDModel->m_CDModel;
      }
      else if (sys != NULL)
	 m_CDModel = sys->CreateModel (this);

      if (cache->IsServer()) 
      {
	 // Remove the geometry not used by the CD system.
	 for (SubModelLI i=m_SubModels.begin(); i!=m_SubModels.end(); ++i)
	    i->m_Meshes.clear();
      }

      return true;
   }

   void
   Model::SetupBBox (bool centered)
   {
      for (SubModelLI i=m_SubModels.begin(); i!=m_SubModels.end(); ++i)
	 m_BBox.AddBBox (i->m_BBox);

      if (!centered)
	 return;

      Vector3 center = m_BBox.m_Min + m_BBox.m_Max;
      center.Scale (-0.5f);

      Matrix44 m;
      m.MakeTranslation(center);

      Transform (m);
   }

   void
   Model::Transform (const Matrix44 &mat, bool scale)
   {
      for (SubModelLI i=m_SubModels.begin(); i!=m_SubModels.end(); ++i)
	 i->Transform (mat, scale);
    
      if (m_Skeleton && scale)
	 m_Scale = mat.M(0,0); // assume uniform scale.
   
      for (AttachmentMI it = m_Attachments.begin();
	   it != m_Attachments.end(); ++it)
      {
	 it->second.m_Origin = mat.Transform (it->second.m_Origin);
	 it->second.m_Vectors[0] = mat.Rotate (it->second.m_Vectors[0]);
	 it->second.m_Vectors[1] = mat.Rotate (it->second.m_Vectors[1]);
	 it->second.m_Vectors[2] = mat.Rotate (it->second.m_Vectors[2]);
      }

      m_BBox.Transform (mat);
   }

   void
   Model::Scale (float scale_factor)
   {
      Matrix44 mat;
      mat.MakeScale (Vector3 (scale_factor, scale_factor, scale_factor));

      Transform (mat, true);
   }

   void
   Model::Render (Ark::Renderer &renderer, const ModelState &state)
   {
      // Make sure the skin has the right number of materials.
      if (state.m_Skin ||
	  state.m_Skin->m_Materials.size() != m_Skin.m_Materials.size())
      {
	 // FIXME: print a warning ?
	 const_cast<ModelState*> (&state)->SetSkin(&m_Skin);
      }

      Skin& sk = (Skin&)*state.m_Skin;

      // IMPORTANT REMARK: if there is a skeleton, the matrix to transform
      // from object coordinate to world coordinate is already included
      // in the bone matrices => the renderer matrix remains unchanged.
      //
      // Otherwise, we need to multiply the current view matrix by the 
      // object->world matrix (state.m_Matrix).

      if (m_Skeleton)
      { 
	 for (SubModelList::iterator m = m_SubModels.begin(); m !=
		 m_SubModels.end(); ++m)
	 {
	    // Transform points
	    m->SkinTransform(state.m_BoneMatrices);
	    
	    renderer.SetActiveVB (m->m_VB);
	    renderer.LockVB (0, m->m_VB.Size());

	    /// Draw meshes
	    const SubModel::MeshList& meshes = m->m_Meshes;
	    for (SubModel::MeshList::const_iterator mesh=meshes.begin() ;
		 mesh!=meshes.end(); ++mesh)
	    {
	       /// Draw blocks
	       Material& mat = *sk.m_Materials[ mesh->m_Material ];
	       
	       const BlockList& blocks = mesh->m_Blocks;
	       
	       for (BlockLCI block=blocks.begin();block!=blocks.end(); ++block)
			   renderer.RenderBlock(mat, *block);
	    }

	    renderer.UnlockVB ();
	 }
      }
      else
      {
	 Matrix44 omtx = state.m_Matrix;

	 // Change the model view matrix
	 renderer.PushModelMatrix (omtx);

	 for (SubModelList::const_iterator m = m_SubModels.begin();
	      m != m_SubModels.end(); ++m)
	 {
	    renderer.SetActiveVB (m->m_VB);
	    renderer.LockVB (0, m->m_VB.Size());

	    /// Draw meshes
	    const SubModel::MeshList& meshes = m->m_Meshes;
	    for (SubModel::MeshList::const_iterator mesh=meshes.begin() ;
		 mesh!=meshes.end(); ++mesh)
	    {
	       /// Draw blocks
	       Material& mat = *sk.m_Materials[ mesh->m_Material ];
	       
	       const BlockList& blocks = mesh->m_Blocks;
	       
	       for (BlockLCI block=blocks.begin(); block!=blocks.end();++block)
		  renderer.RenderBlock(mat, *block);
	    }
	    
	    renderer.UnlockVB ();
	 }

	 // Restore the view matrix
	 renderer.PopModelMatrix();
      }

   }

// ============================================================================
// Sub model & building
// ============================================================================
   SubModel::SubModel ()
   {
      m_Builder = 0;
      m_SkinnedVB = 0;
   }

   SubModel::~SubModel ()
   {
      if (m_Builder)
      {
	 delete m_Builder;
	 m_Builder = 0;
      }
      
      if (m_SkinnedVB)
      {
	 delete m_SkinnedVB;
	 m_SkinnedVB = 0;
      }

   }

   ModelBuilder *
   SubModel::Builder()
   {
      if (m_Builder == 0)
	 m_Builder = new ModelBuilder (this);

      return m_Builder;
   }

   bool SubModel::Build()
   {
      if (m_Builder == NULL)
	 return false;

      bool res = m_Builder->Build();

      delete m_Builder;
      m_Builder = NULL;

      return res;
   }

   /**
    * Transform all points in this submodel by the \c transform
    * matrix. Rotate normals using the same matrix.
    */
   void SubModel::Transform (const Matrix44 &transform, bool scale)
   {
      const size_t sz = m_VB.Size();

      // Transform all points in the VB...
      for (size_t i = 0; i < sz; i++)
      {
	 Vector3 &n = m_VB.Normal(i);
	 Vector3 &c = m_VB.Coord(i);

	 if (!scale) n = transform.Rotate (n);
	 c = transform.Transform (c);
      }
   }

   void SubModel::SkinTransform (const Matrix44* boneMatrices)
   {
	   const int vertexCount = m_VB.Size();
	   if (!m_SkinnedVB)
	   {
		   m_SkinnedVB = new VertexBuffer();
		   m_SkinnedVB->SetFormat(VertexBuffer::VB_HAS_COORD | VertexBuffer::VB_HAS_NORMAL);
		   m_SkinnedVB->Resize(vertexCount);
		   for (int i=0 ; i<vertexCount ; ++i)
		   {
			   m_SkinnedVB->Coord(i) = m_VB.Coord(i);
			   m_SkinnedVB->Normal(i) = m_VB.Normal(i);
		   }
	   }

	   VertexBuffer& vb = *m_SkinnedVB;

	   /// Transform points
	   for (int i=0 ; i<vertexCount ; ++i)
	   {
		   const int bone = m_BoneBindings[i];
		   const Matrix44& transform = boneMatrices[bone];
		   m_VB.Coord(i) = transform.Transform(vb.Coord(i));
		   m_VB.Normal(i) = transform.Rotate(vb.Normal(i));
	   }
   }

   /// Returns a string describing the model
   String
   Model::Dump (bool long_version)
   {
       std::ostringstream os;
       os << "[mdl] " << m_Name;

       if (long_version)
       {
	   os << "\n\tnum_bones : " << (m_Skeleton ? m_Skeleton->m_Bones.size() : 0);
	   os << "\n\tnum_controllers : " << (m_Skeleton ? m_Skeleton->m_Controllers.size() : 0);
	   os << "\n\tnum_hitboxes : " << (m_Skeleton ? m_Skeleton->m_HitBoxes.size() : 0);    
	   os << "\n\tnum_submodels : " << m_SubModels.size();
	   os << "\n\tnum_attachments : "<< m_Attachments.size();
	   os << "\n\tnum_materials : " << m_Skin.m_Materials.size();                        
       }

       return os.str();
   }


} //namespace Ark
