#include "config.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "SnortLog.h"
#include "NetRangerLog.h"
#include "DragonLog.h"
#include "HttpUpload.h"
#include "misc.h"
#include "IDSLog.h"

#define SFCLEAN_VERSTRING "1.6.2"
#define AUTO_RETRY 20

extern char *optarg;
int verbose = 0;
int quiet = 0;
int curlverbose = 0;
char sUploadType[100];

void usage()
{

  fprintf(stderr, "SecurityFocus ARIS Extractor Version " SFCLEAN_VERSTRING "\n"
                  "Usage: extractor [options] -u <username> [-p password] <logfile>\n"
		  "       extractor [options] -a <username/password file> <logfile>\n"
		  "       extractor [options] -X <logfile>\n"
                  "       extractor [options] -L -u <username> [-p password] <xml file>\n"
                  "       extractor [options] -L -a <username/password file> <xml file>\n"
                  "Options:\n"
		  "  -u <username>\t\tUsername for upload server.\n"
	          "  -p <password>\t\tPassword for upload server.\n"
		      "\t\t\t(ARIS Extractor will prompt for password if omitted.)\n"
		  "  -a <username/password file>\tText file containing Username and Password for\n"
		      "\t\t\tUpload server (Newline separated)\n"
	          "  -f <portscan_logfile>\tOptionally specify a Snort portscan logfile to process.\n"
	          "  -c <mask1,mask2...>\tStrip IP addresses in specified netmasks from output.\n"
		      "\t\t\tMust be in CIDR block notation.\n"
	          "  -S <proxy_addr:port>\tUse a proxy server and optionally specify a port.\n"
                  "  -U <proxy_user>\tSpecify a username for authenticated proxy.\n"
                  "  -P <proxy_password>\tSpecify a password for authenticated proxy.\n"
	          "  -i <sensor_id>\tOptionally specify a unique sensor identifier.\n"
	          "  -X\t\t\tParse log only.  Do not upload results to server.\n"
	          "  -v\t\t\tDetailed output from cURL.\n"
	          "  -V\t\t\tDetailed output about log processing.\n"
	          "  -D <debug_logfile>\tDetailed upload information.\n"
	          "  -I <DD/MM/YYYY>\tOnly parse incidents after date input.\n"
	          "  -h <host>\t\tOverride default upload server with <host>.\n"
		  "  -n\t\t\tParse Cisco Secure IDS (Netranger) Logfile.\n"
		  "  -d\t\t\tParse Dragon IDS Logfiles. <logfile> should specify the\n"
		  "\t\t\tDB directory and driders.cfg file (Comma separated)\n"
                  "  -L\t\t\tUpload specified XML file to the server.\n"
                  "  -q\t\t\tQuiet logging. Everything except error messages are\n"
                  "  \t\t\tsupressed.\n"
                  "  -g\t\t\tIDS incident timestamps are in GMT. For use only with\n"
		  "\t\t\twith Snort and Dragon which log to local time by default\n"
                  "  -l\t\t\tIDS incident timestamps are in local time. For use\n"
		  "\t\t\tonly with NetRanger which logs to GMT by default.\n"
                  "  -N <255.255.255.0>\tSpecify netmask for your IDS. This helps SecurityFocus\n"
		  "\t\t\tkeep accurate statistics.\n"
	          "\n  <logfile> is the path to an input logfile.\n\n");
  exit(-1);
}

char *homedir = NULL;

void printTimestamp( double time )
{
    if( time == 0 )
    {
      printf( "No timestamp\n");
      return;
    }

    // convert to readable format
    int year, month, day, hour, minute, second, milli;

    CIDSLog::GetDate( time, year, month, day, hour, minute, second, milli);
    
    printf("%d:%.2d:%.2d  %.2d/%.2d/%d\n",hour,minute,second,day,month,year);
}

FILE *
openTimefile(char *sensor)
{
  char filename[1024];
  FILE *f;
  int n = 0;

  if(sensor)
    n = strlen(sensor);

  if(n > 100)
    return NULL;

  memset(filename, 0, sizeof(filename));
  snprintf(filename, 1023 - strlen("/lasttime-") - n, "%s/.sfclean", homedir);
  if(mkdir(filename, 0755) && (errno != EEXIST)) {
    fprintf(stderr, "Error creating directory: %s\n", filename);
    return NULL;
  }
  strcat(filename, "/lasttime");

  if(sensor) {
    strcat(filename, "-");
    strcat(filename, sensor);
  }

  if(!(f = fopen(filename, "r+")) && errno == ENOENT)
  {
    f = fopen(filename, "w+");
    if (!f)
      fprintf (stderr, "Could not create '%s'\n", filename); 
  }
  
  return f;
}


int 
loadTimestamp(double *ts, char *sensor)
{  
  FILE *f;
  char buf[100];

  if(!(f = openTimefile(sensor))) {
    fprintf(stderr, "Could not open timestamp file\n");
    return -1;
  }
  
  buf[0] = 0;
  fgets(buf, 100, f);
  if(buf[0] == 0) { // file is empty
    *ts = 0.0;
  }
  else {
    if(sscanf(buf, "%lf", ts) != 1) {
      fprintf(stderr, "Error reading timestamp. Timestamp file contains '%s'\n", buf);
      return -1;
    }
  }

  if(verbose)
  {
    printf("Read saved timestamp = ");
    printTimestamp( *ts );
  }
  return 0;
}
  
  
int
saveTimestamp(double ts, char *sensor)
{
  FILE *f;
  char buf[100];

  if(!(f = openTimefile(sensor))) {
    fprintf(stderr, "Could not open timestamp file\n");
    return(-1);
  }

  memset(buf, 0, sizeof(buf));
  snprintf(buf, 99, "%.8f", ts);
  fputs(buf, f);

  if(verbose)
  {
    printf("Saved timestamp = ");
    printTimestamp( ts ); 
  }
  return (0);
}


/* read username & password file */
int
readAuthFile (char *fname, char **username, char **password)
{
  char sAuth[32];
  char *newline = NULL;
  FILE *pAuthFile = fopen (fname, "r");
  if (!pAuthFile)
    return -1;
  if (!fgets (sAuth, 31, pAuthFile))
  {
    fclose (pAuthFile);
    return -1;
  }
  newline = strchr (sAuth, '\n');    
  if (newline)
    *newline = '\0';
  *username = strdup (sAuth);

  if (!fgets (sAuth, 31, pAuthFile))
  {
    free (*username);
    *username = NULL;
    fclose (pAuthFile);
    return -1;
  }
  newline = strchr (sAuth, '\n');    
  if (newline)
    *newline = '\0';
  *password = strdup (sAuth);

  fclose (pAuthFile);
  return 0;
}

// from Netmask.cpp
unsigned int GetNetmask();
FILE *pDebugLog;
int
main(int argc, char **argv)
{
  double last_date;
  double ts = 0.0;
  CIDSLog *myLog;
  HttpUpload *up;
  char op;
  char *pscanfile = NULL;
  int dopscan = 0;
  int parseonly = 0;
  int netranger = 0;
  int dragonlog = 0;
  int uploadonly = 0;
  int authfile = 0;
  int authcmdline = 0;
  int debuglog = 0;
  int useTimestamp = 0;
  int useGMT = 0;
  int useLocalTime = 0;
  unsigned long netmask = 0;
  double timestamp = 0.0;
  char *excludemasks = NULL;
  char *filenames = NULL;
  char *username = NULL;
  char *password = NULL;
  char *proxy_server = NULL;
  char *proxy_user = NULL;
  char *proxy_password = NULL;
  char *sensor_id = NULL;
  char *overridehost = NULL;
  char pathbuf[1024];
  char errbuf[256];
  struct stat stat_buf;
  int nRetry = AUTO_RETRY;
  bool bRetry = true;

  /* mask out SIGPIPEs */
  sigset_t set;
  sigemptyset (&set);
  sigaddset (&set, SIGPIPE);
  sigprocmask (SIG_BLOCK, &set, NULL);
  /* ================= */

  // get the HOME environment variable
  if(!homedir) {
    if(getenv("HOME"))
      homedir = strdup(getenv("HOME"));
    if( !homedir || !homedir[0]) {
      fprintf(stderr, "HOME environment variable must be set\n");
      return 0;
    }
  }

  while((op = getopt(argc, argv, "lgqndXvVLf:I:D:c:u:p:i:S:U:P:h:a:N:")) != -1) 
  {
    switch(op)
    {
    case 'd':
      dragonlog++;
      break;
    case 'n':
      netranger++;
      break;
    case 'L':
      uploadonly ++;
      break;
    case 'h':
      overridehost=optarg;
      break;
    case 'X':
      parseonly ++;
      break;
    case 'v':
      curlverbose ++;
      break;
    case 'V':
      verbose ++;
      break;
    case 'q':
      quiet ++;
      break;
    case 'g':
      useGMT++;
      break;
    case 'l':
      useLocalTime++;
      break;
    case 'I':
      // parse the date
      int year,month,day;
      if( (sscanf( optarg, "%d/%d/%d", &day, &month, &year )) == 3)
      {
        useTimestamp ++;

	// create a temporary log so that we can call GetDate
        timestamp = CIDSLog::GetDate( year, month, day, 0, 0, 0, 0, 0 ); // last 0 = no conversion to GMT
        if( quiet == 0 )
        {
          printf("Parsing incidents after: ");
          printTimestamp(timestamp);
        }
      }
      else
      {
        printf("Warning: Timestamp not formatted according to <DD/MM/YYYY>.\nParsing all incidents.\n" );
        timestamp = 0.0;
      }
      break;
    case 'N':
      netmask = inet_addr(optarg);
      break;
    case 'f':
      dopscan ++;
      pscanfile = optarg;
      break;
    case 'c':
      excludemasks = optarg;
      break;
    case 'a':
      if (authcmdline)			/* if the username or password are on the command line */
        break;				/* ignore the auth file */
      authfile ++;
      if (readAuthFile (optarg, &username, &password) == -1)
	authfile = -1;
      break;
    case 'u':
      authcmdline ++;
      if (authfile == 1) {              /* if the username & password are defined by the auth file */
        free (username);		/* override them */
        username = NULL;
        free (password);
        password = NULL;
      }
      authfile = 0;
      username = strdup(optarg);
      break;
    case 'p':
      authcmdline ++;
      if (authfile == 1) {		/* if the username & password are defined by the auth file */
	free (username);		/* override them */
	username = NULL;
	free (password);
	password = NULL;
      }
      authfile = 0;
      password = strdup(optarg);
      break;
    case 'D':
      debuglog++;
      // open the debug output file
      pDebugLog = fopen( optarg, "a+");
      if(pDebugLog == NULL )
      {
        printf("WARNING: Unable to open upload logging file %s.\n", optarg );
        break;
      }
      time_t timer;
      time(&timer);
      struct tm * t;
      t = gmtime(&timer);
      debug("********************************\n");
      debug(" Date: %s", asctime(t));
      debug("********************************\n");
      break;
    case 'S':
      proxy_server = optarg;
      break;
    case 'U':
      proxy_user = optarg;
      break;
    case 'P':
      proxy_password = optarg;
      break;
    case 'i':
      sensor_id = optarg;
      break;
    case '?':
    default:
      usage();
    }
  }

  if (authfile == -1)
  {
    fprintf(stderr, "Error reading username/password file\n\n");
    return -1;
  }
  
  if(username && !password) 
#ifdef HAVE_GETPASSPHRASE
    password = strdup(getpassphrase("Enter Password: "));
#else
    password = strdup(getpass("Enter Password: "));
#endif

  /* skip options */
  argc -= optind;
  argv += optind;

  if(argc != 1)
  {
    fprintf(stderr, "No file was specified, or invalid command line.\n\n");
    usage();
  }
  /* if nothing useful was specified on the command line */
  if (!parseonly && (!username || !password))
  {
    fprintf(stderr, "Invalid command line.\n\n");
    usage();
  }

  if(proxy_server)
  {
    if((proxy_user && !proxy_password) || (proxy_password && !proxy_user))
    {
      fprintf(stderr, "You must specify both the proxy username and the proxy password.\n\n");
      usage ();
    }
  }
  else if (proxy_user || proxy_password)
  {
    fprintf(stderr, "You must specify a proxy server when a proxy username and password are given.\n\n");
    usage ();
  } 

  if (parseonly)
  {
    if (uploadonly || username || password || proxy_user || proxy_password || proxy_server || overridehost)
    {
      fprintf (stderr, "You specified one or more options that are incompatible with -X.\n\n");
      usage ();
    }
  }
  else if (uploadonly)
  {
    if (dopscan || dragonlog || netranger || sensor_id || excludemasks)
    {
      fprintf (stderr, "You specified one or more options that are incompatible with -L.\n\n");
      usage ();
    }
  }
  else if (dopscan)
  {
    if (dragonlog || netranger)
    {
      fprintf (stderr, "The portscan option is not compatible with the -d or -n options.\n\n");
      usage ();
    }
  }
  else if (dragonlog)
  {
    if (netranger)
    {
      fprintf (stderr, "You cannot specify both the -d and -n options.\n\n");
      usage ();
    }
  }

  /* early file checks */
  if(!dragonlog && stat(argv[0], &stat_buf) == -1)
  {
    fprintf(stderr, "Could not open '%s'.\n\n", argv[0]);
    return -1;
  }
  if(!dragonlog && S_ISDIR(stat_buf.st_mode))
  {
    fprintf(stderr, "Input file '%s' is a directory.  It's supposed to be an IDS alert file\n", argv[0]);
    return -1;
  }
 
  if (uploadonly)
    strncpy (pathbuf, argv[0], 1024);
  else
  {
    if(dopscan)
    {
      if((filenames = (char *) malloc(strlen(argv[0]) + strlen(pscanfile) + 2)) == NULL)
      {
        fprintf(stderr, "Ack! No memory!\n");
        return -1;
      }
      strcpy(filenames, argv[0]);
      strcat(filenames, ",");
      strcat(filenames, pscanfile);
    }
    else
    {
      if ((filenames = strdup(argv[0])) == NULL)
      {
        fprintf(stderr, "Ack! No memory!\n");
        return -1;
      }
    }

    if( useTimestamp )
    { 
      ts = timestamp;
    }
    else
    {
      if(loadTimestamp(&ts, sensor_id) == -1)
        ts = 0.0;

      // do a sanity check on the timestamp
      // check if the timestamp is in the future
      time_t timer;
      time(&timer);
      struct tm * t;
      t = gmtime(&timer);

      double nowdouble = 0;
      nowdouble = CIDSLog::GetDate( 
	t->tm_year + EPOCH_YEAR,
	t->tm_mon + 1,
	t->tm_mday,
	t->tm_hour,
	t->tm_min,
	t->tm_sec,
	0,   //ms
	0 ); // already converted to GMT
      
      // set the timestamp to now for now
      if(nowdouble < ts)
        ts = nowdouble;
    }

    memset(pathbuf, 0, sizeof(pathbuf));
    if(netranger)
    {
      myLog = new CNetRangerLog;
      snprintf(pathbuf, sizeof(pathbuf) - 1, "%s/.sfclean/netranger.xml", homedir);
    }
    else if (dragonlog)
    {
      myLog = new CDragonLog;
      snprintf(pathbuf, sizeof(pathbuf) - 1, "%s/.sfclean/dragonlog.xml", homedir);
    }
    else
    {
      myLog = new CSnortLog;
      snprintf(pathbuf, sizeof(pathbuf) - 1, "%s/.sfclean/snortlog.xml", homedir);
    }

    if (excludemasks)
    { 
      if(!(myLog->SetIPMasks(excludemasks)))
      {
        fprintf(stderr, "Could not set excludemasks!\n");
        delete myLog;
        usage();
      }
    } else
       myLog->SetIPMasks(NULL);

    // currently unused for *nix version of Extractor
    // used in the xml header in IDSLog.cpp
    sprintf( sUploadType, "unknown" );
    fflush(stdout );
    
    // set the netmask variable

    if(netmask == 0)
      netmask = GetNetmask();

    myLog->m_nNetMask = netmask;

    // check whether we are using local or GMT
    if( netranger )
    {
      myLog->m_nLoggingInGMT = (!useLocalTime);
    }
    else
    {
      // either we're snort or dragon
      // which log to local time by default
      // check the -g flag to see if the default has been changed
      myLog->m_nLoggingInGMT = useGMT;      
    }

    //
    // Generate XML 
    if(!(myLog->Output(sensor_id, ts, 
		       filenames,
		       pathbuf, last_date)))
    {
      delete myLog;
      fprintf(stderr, "Error processing logfile\n");
      return -1;
    }
  
    myLog->FreeIPMasks();
    free(filenames);

    // XXX add printStats() to CNetRangerLog and CDragonLog
    if(verbose && !netranger && !dragonlog)
      ((CSnortLog *)myLog)->printStats();

    delete myLog;
  }

  if(parseonly)
  {
    debug( "Output saved in %s\nNo Upload specified, exiting.\n", pathbuf);
    fprintf(stderr, "Exiting without submitting log to server\n"
	            "Output saved in %s\n", pathbuf);
    return 0;
  }

#ifdef HAVE_CURL_GLOBAL_INIT
  curl_global_init(CURL_GLOBAL_ALL);
#endif
  
  // curl version
  debug( "Curl version: %s\n", curl_version() );
  
  bool bUploaded = false;
  while (bRetry)
  {
    if( debuglog )
    {
      FILE * temp = fopen( pathbuf, "r" );
      if( temp )
      {
        fseek( temp, 0, SEEK_END );
        long filesize = ftell( temp );
        debug("Upload file:   %s\nSize (bytes):  %ld\n\n", pathbuf, filesize );
        fclose( temp );
        if( filesize > 30000000) debug( "Warning! Uploading 30 megs");
      }
    }

    up = new HttpUpload(pathbuf, username, password, "/tmp/sfclean.XXXXXX");

    if(proxy_server) 
      up->setProxyInfo(proxy_server, proxy_user, proxy_password);
    
    if(overridehost)
      up->setAlternateHost(overridehost);

    bUploaded = up->doTransfer(); 
    int errInfo = up->errorInfo(errbuf, 256);

    delete up;

    if ((errInfo == HttpUpload::ERR_TIMEOUT) || (errInfo == HttpUpload::ERR_WRITE) || (errInfo == HttpUpload::ERR_SERVER) ||
        (errInfo == HttpUpload::ERR_SSL) || (errInfo == HttpUpload::ERR_SERVAUTH))
    {
      nRetry--;
      if (nRetry > 0)
      {	
        fprintf (stderr, " Retrying Upload...: %s\n", errbuf);
        continue;
      }
    }
    if (errInfo != HttpUpload::SUCCESS)
    {
      fprintf(stderr, "\nError uploading log: %s\n", errbuf);
    }
    bRetry = false;
  }

#ifdef HAVE_CURL_GLOBAL_INIT
  curl_global_cleanup();
#endif

  // if the debug log exists, close it
  if(pDebugLog)
    fclose(pDebugLog);

  if (bUploaded)
  {
    if (!uploadonly)
      saveTimestamp(last_date, sensor_id);
    if( quiet == 0 )
      fprintf(stderr, "Upload completed\n");
  }

  free (username);
  free (password);
  return 0;
}
