/*
 * A little test dealie to see if we can retrieve the broadcast addr
 *
 * Works on HP_UX, freebsd, linux
 * gcc -o /tmp/get_bcast ./get_bcast.c
 *
 * Works on solaris or Unixware (SVR4) with:
 * gcc -DBSD_COMP -o /tmp/get_bcast ./get_bcast.c -lsocket -lnsl
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>

// compile a stub GetNetmask() f'n only for solaris linux and win
// GETNETMASK defined for openbsd and solaris in configure script
#ifdef GETNETMASK

#include <errno.h>
#include <memory.h>
#include <unistd.h>

#include <sys/types.h>

#include <sys/ioctl.h>

#include <sys/socket.h>
#ifdef __SVR4
#include <sys/sockio.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>

// for sending debug 
#include "misc.h"

// defines the maximum number of interfaces we will poll for netmasks
#define MAXNUMINTERFACES 4

unsigned long GetNetmask()
{
	struct ifconf ifc; /* holds IOCTL return value for SIOCGIFCONF */

	// maximum number of interfaces
	int fd, n;
	struct ifreq *ifr; /* points to one interface returned from ioctl */
	struct sockaddr_in *sin;
	
	if ((fd = socket (PF_INET, SOCK_DGRAM, 0)) < 0)
		return 0;
	
	memset ((void*)&ifc, 0, sizeof(ifc));

	ifc.ifc_buf = NULL;
	ifc.ifc_len =  sizeof(struct ifreq) * MAXNUMINTERFACES;
	ifc.ifc_buf = (caddr_t)malloc(ifc.ifc_len);

	/* get the interface list, this should fail if we have more than 5 */

	if (ioctl(fd, SIOCGIFCONF, &ifc) < 0) {
		close(fd);
		debug("SIOCGIFCONF failed");
		return 0;
	}

	// interface counter
	int ii=0,jj=0;	

	// there are 2 properties of the interface we want to save 
	// to decide whether the interface is the one being
	// used by the IDS.
	int promiscuous[MAXNUMINTERFACES];
	unsigned long netmasks[MAXNUMINTERFACES];
	for(jj=0;jj<MAXNUMINTERFACES;jj++){promiscuous[jj]=0;netmasks[jj]=0;}

	/* loop through interfaces returned from SIOCGIFCONF */
	ifr = ifc.ifc_req;
	for (n=0; n < ifc.ifc_len; n+=sizeof(struct ifreq)) 
	{
		if (ioctl(fd,SIOCGIFADDR, ifr) != 0) 
		{
			close(fd);
			free (ifc.ifc_buf);
			debug("SIOCGIFADDR failed");
			return 0;
		}

		if (ifr->ifr_dstaddr.sa_family != AF_INET) 
		{
			ifr++;
			continue;
		}

		// we don't want to look at interfaces that have the following ips...
		// 127.*
		// 224.* and above.
		// use masks to remove the unwanted parts of the address for comparison porpoises
		sin = (struct sockaddr_in *) &ifr->ifr_dstaddr;
		unsigned long truncatedip1 = sin->sin_addr.s_addr & inet_addr("255.0.0.0");
		unsigned long truncatedip2 = sin->sin_addr.s_addr & inet_addr("255.255.0.0");

		if((truncatedip1 == inet_addr("127.0.0.0")) ||
			(truncatedip1 >= inet_addr("224.0.0.0")) )
		{
			ifr++;
			continue;
		}

		if (ioctl(fd,SIOCGIFNETMASK, ifr) != 0) 
		{
			close(fd);
			free (ifc.ifc_buf);
			debug("SIOCGIFNETMASK failed");
			return 0;
		}

		// check if the interface is in promiscuous mode
		if( ifr->ifr_ifru.ifru_flags & IFF_PROMISC )
			promiscuous[ii] = 1;

		// The next two ips would correspond to machines listening or local networks
 		// if this is the case, default to 255.255.255.0
		// 10.*
		// 192.168.*
		if( (truncatedip1 == inet_addr("10.0.0.0")) ||
			(truncatedip2 == inet_addr("192.168.0.0")))
		{
			netmasks[ii] = inet_addr("255.255.255.0");
		}
		else
		{
			// point the sin to the netmask of the interface
			sin = (struct sockaddr_in *) &ifr->ifr_dstaddr;
		
			// we will return only the most conservative netmask
			// store it for comparison later
			netmasks[ii] = *((unsigned long*)&sin->sin_addr);
		}
		
		// increment the polled interface count
		ii++;
	}

	// return the most conservtive netmask with the promiscuous bit set,
	// or if no promiscuous bits are set, return the most conservative
	// netmask (ie given the choice between 255.255.0.0 and 255.255.255.0
	// return 255.255.255.0)
	unsigned long returnnetmask = 0;
	for(jj=0;jj<ii;jj++)
	{
		if(promiscuous[jj])
			if(netmasks[jj] > returnnetmask)
				returnnetmask = netmasks[jj];
	}

	if(returnnetmask) return returnnetmask;

	// none of the interfaces are promiscuous, so return the most conservative
	for(jj=0;jj<ii;jj++)
	{
		if(netmasks[jj] > returnnetmask)
			returnnetmask = netmasks[jj];
	}

	return returnnetmask;
}


#else

#ifdef OPENBSD
// this portion of the code was written by
// Mario Van Velzen, mvelzen@securityfocus.com
// With thanks to Andrew Baker

#include <errno.h>
#include <memory.h>
#include <unistd.h>
#include <ifaddrs.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>

// for sending debug messages
#include "misc.h"

// GetNetmask function
// used by openbsd
unsigned long GetNetmask()
{

	struct ifaddrs *ifap, *ifa;
	in_addr_t config = inet_addr( "0.0.0.0" );
	in_addr_t config_rfc = inet_addr( "255.255.255.0" );
	in_addr_t config_mask = inet_addr( "0.0.0.0" );
	int ifpromisc_rfc = 0;
	int ifpromisc_mask = 0;

	if( getifaddrs( &ifap ) )
	{
	    return( inet_addr( "255.255.255.255" ) );
	}

	for(ifa = ifap; ifa; ifa=ifa->ifa_next)
	{
	    if ( ( ( ifa->ifa_flags & ( IFF_UP | IFF_RUNNING | IFF_LOOPBACK ) )
        	== ( IFF_UP | IFF_RUNNING ) ) &&
		( ifa->ifa_addr->sa_family == AF_INET ) )
	    {
		memcpy(&(config),
		    &(((struct sockaddr_in *)ifa->ifa_addr)->sin_addr),
		    sizeof(struct in_addr));

		unsigned long truncip1 = config & inet_addr("255.0.0.0");
		unsigned long truncip2 = config & inet_addr("255.255.0.0");

		memcpy(&(config),
		    &(((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr),
		    sizeof(struct in_addr));

		if( (truncip1 == inet_addr("10.0.0.0")) ||
			(truncip1 == inet_addr("127.0.0.0")) ||
			(truncip2 == inet_addr("192.168.0.0")) ||
			(truncip1 >= inet_addr("224.0.0.0")) )
		{
			if ( config > config_rfc )
			{
				config_rfc = config;
			}
			if ( ifa->ifa_flags & IFF_PROMISC ) {
			    ifpromisc_rfc = 1;
			}
			continue;
		}

		if ( ifa->ifa_flags & IFF_PROMISC ) {
		    ifpromisc_mask = 1;
		}

		if ( config > config_mask )
		{
			config_mask = config;
		}
	    }
	}

	if ( ifpromisc_mask == 1 )
	{
		return( config_mask );
	}

	if ( ifpromisc_rfc == 1 )
	{
		return( config_rfc );
	}
	if ( config_mask > 0 )
	{
		return( config_mask );
	}
	if ( config_rfc > 0 )
	{
		return( config_rfc );
	}

	return( inet_addr( "255.255.255.255" ) );
}

#else // #ifdef OPENBSD

// stub version of GetNetmask
// returns 0
unsigned long GetNetmask()
{
	return 0;
}
#endif //#ifdef OPENBSD
#endif // #ifdef GETNETMASK
