# Copyright (c) 2001 Carnegie Mellon University
# All rights reserved.
# See Copyright section at end of file use and distribution information.
#
# Net::Argus::HostPairs is a module that reads Argus data files (using ra)
# and extracts some basic statistics about which hosts talked to one
# another, and how much data was exchanged between them. See the POD  
# documentation for details, or the 'hostpairs' application for an
# example of how this module is used.
# Net::Argus::HostPairs requires the Net::Patricia, Text::Wrap, POSIX and
# Socket modules. Net::Patricia is available from www.cpan.org.  The Socket 
# Text::Wrap, and POSIX modules should be part of the regular perl5 
# distribution, and are also available from cpan.
#

package Net::Argus::HostPairs;

use strict;
use Net::Patricia;
use Socket;
use Text::Wrap;
use POSIX;

use vars qw($VERSION);

$VERSION = '1.0';

my $Debug = 0;

### don't change $base_filter or things will break:
my $base_filter = '(ip and not man)';
###

sub new {
    my $class = shift;
    my $self = {};
    $self->{'filedata'} = {};
    $self->{'argus_files'} = "";
    $self->{'pairs'} = {};
    $self->{'summary'} = {};
    $self->{'mode'} = 'hostpair';
    $self->{'filter'} = $base_filter;
    $self->{'filter_string'} = $base_filter;
    $self->{'summary_format'} = "";
    $self->{'hostpair_format'} = "";
    $self->{'portpair_format'} = "";
    $self->{'summary_sortby'} = "";
    $self->{'hostpair_sortby'} = "";
    $self->{'portpair_sortby'} = "";
    $self->{'summary_output'} = "";
    $self->{'hostpair_output'} = "";
    $self->{'portpair_output'} = "";
    $self->{'time_format'} = "";
    $self->{'print_header'} = 0;
    $self->{'print_port_names'} = 0;
    $self->{'summary_sort_reverse'} = {};
    $self->{'hostpair_sort_reverse'} = {};
    $self->{'portpair_sort_reverse'} = {};
    bless($self, $class);
    return $self;
}

sub ra_prog {
    my $self = shift;
    if (@_) {
        my $prog = shift;
        $self->{'ra_prog'} = $prog;
    }
    return  $self->{'ra_prog'};
}    

sub rarc {
    my $self = shift;
    if (@_) {
        $self->{'rarc'} = shift;
    }
    return $self->{'rarc'};
}    

sub ra_extra_args {
    my $self = shift;
    if (@_) { 
        my @args = @_;
        $self->{'ra_extra_args'} = \@args;
    }
    return ($self->{'ra_extra_args'}) ? @{$self->{'ra_extra_args'}} : undef;
}

sub hostset1 {
    my $self = shift;
    if (@_) {
        my $hoststring = shift;
        my $p = Net::Patricia->new();
        _parse_hoststring($p, $hoststring);
        $self->{'hostset1'} = $p;
        $self->{'hostset1_string'} = $hoststring;
    }    
    return $self->{'hostset1_string'};
}

sub hostset2 {
    my $self = shift;
    if (@_) {
        my $hoststring = shift;
        my $p = Net::Patricia->new();
        _parse_hoststring($p, $hoststring);
        $self->{'hostset2'} = $p;
        $self->{'hostset2_string'} = $hoststring;
    }
    return $self->{'hostset2_string'};
}    
    
sub portset1 {
    my $self = shift;
    if (@_) {
        my $portstring = shift;
        my @ports = ();
        my $portarray = _parse_portstring(\@ports, $portstring);
        $self->{'portset1'} = $portarray;
        $self->{'portset1_string'} = $portstring;
    }
    return $self->{'portset1_string'};
}    

sub portset2 {
    my $self = shift;
    if (@_) {
        my @ports = ();
        my $portstring = shift;
        my $portarray = _parse_portstring(\@ports, $portstring);
        $self->{'portset2'} = $portarray;
        $self->{'portset2_string'} = $portstring;
    }
    return $self->{'portset2_string'};
}

sub filter {
    my $self = shift;
    if (@_) {
        my $filt = shift;
        $self->{'filter'} = "($filt)"  . " and $base_filter";
        $self->{'filter_string'} = $self->{'filter'};
    }
    return $self->{'filter_string'};
}    

sub argus_files {
    my $self = shift;
    if (@_)    { 
        my @files = @_;
        $self->{'argus_files'} = \@files;
    }
    return  ($self->{'argus_files'}) ? @{$self->{'argus_files'}} : undef;
}

sub summary_sortby {
    my $self = shift;
    if (@_) {
        $self->{'summary_sortby'} = shift;
    }
    return $self->{'summary_sortby'};
}    

sub hostpair_sortby {
    my $self = shift;
    if (@_) {
        $self->{'hostpair_sortby'} = shift;
    }
    return $self->{'hostpair_sortby'};
}    

sub portpair_sortby {
    my $self = shift;
    if (@_) {
        $self->{'portpair_sortby'} = shift;
    }
    return $self->{'portpair_sortby'};
}    

sub get_filedata {
    my $self = shift;
    return $self->{'filedata'};
}

sub get_pairs {
    my $self = shift;
    return $self->{'pairs'};
}

sub get_summary {
    my $self = shift;
    return $self->{'summary'};
}


sub summary_output {
    my $self = shift;
    if (@_) {
        $self->{'summary_output'} = shift;
    }
    return $self->{'summary_output'};
}

sub hostpair_output {
    my $self = shift;
    if (@_) {
        $self->{'hostpair_output'} = shift;
    }
    return $self->{'hostpair_output'};
}

sub portpair_output {
    my $self = shift;
    if (@_) {
        $self->{'portpair_output'} = shift;
    }
    return $self->{'portpair_output'};
}

# format should be a format string suitable for giving to printf

sub summary_format { 
    my $self = shift;
    if (@_) {
        $self->{'summary_format'} = shift;
    }
    return $self->{'summary_format'};
}

sub hostpair_format { 
    my $self = shift;
    if (@_) {
        $self->{'hostpair_format'} = shift;
    }
    return $self->{'hostpair_format'};
}

sub portpair_format { 
    my $self = shift;
    if (@_) {
        $self->{'portpair_format'} = shift;
    }
    return $self->{'portpair_format'};
}

# format string should be something strftime understands
sub time_format { 
    my $self = shift;
    if (@_) {
        $self->{'time_format'} = shift;
    }
    return $self->{'time_format'};
}


#
# sort_order($mode, $field, n(ormal)|r(everse))
#

sub sort_order {
    my $self = shift;
    my $mode = shift;
    my $field = shift;
    my $order = shift;
    my $order_hash;
    if ($mode eq 'summary') {
        $order_hash = $self->{'summary_sort_reverse'};
    } elsif ($mode eq 'hostpair') {
        $order_hash = $self->{'hostpair_sort_reverse'};
    } elsif ($mode eq 'portpair') {
        $order_hash = $self->{'portpair_sort_reverse'};
    } else {
        die "Invalid sort order\n";
    }
    if ($order =~ m/^n(ormal)?$/) {
        delete($$order_hash{$field}) if exists($$order_hash{$field});
    } elsif ($order =~ m/^r(everse)?$/) {
        $$order_hash{$field} = 1;
    }
    return $order;
}

# do we output a header summary or not?  Values are 'yes' or 'no'.

sub print_header {
    my $self = shift;
    if (@_) {
        $self->{'print_header'} = shift;
    }
    return $self->{'print_header'};
}

sub print_port_names {
    my $self = shift;
    if (@_) {
        $self->{'print_port_names'} = shift;
    }
    return $self->{'print_port_names'};
}

sub h1_file {
    my $self = shift;
    if (@_) {
        my $file = shift;
        $self->{'h1_file'} = $file;
        my $host_string = _read_file($file);
        $self->hostset1($host_string);
        $self->{'hostset1_string'} = "FILE: $file";
    
    }
    return $self->{'h1_file'};
}    

sub h2_file {
    my $self = shift;
    if (@_) {
        my $file = shift;
        $self->{'h2_file'} = $file;
        my $host_string = _read_file($file);
        $self->hostset2($host_string);
        $self->{'hostset2_string'} = "FILE: $file";
    }
    return $self->{'h2_file'};
}    

sub p1_file {
    my $self = shift;
    if (@_) {
        my $file = shift;
        $self->{'p1_file'} = $file;
        my $port_string = _read_file($file);
        $self->portset1($port_string);
        $self->{'portset1_string'} = "FILE: $file";
    }
    return $self->{'p1_file'};
}    

sub p2_file {
    my $self = shift;
    if (@_) {
        my $file = shift;
        $self->{'p2_file'} = $file;
        my $host_string = _read_file($file);
        $self->portset2($host_string);
        $self->{'portset2_string'} = "FILE: $file";
    }
    return $self->{'p2_file'};
}    

sub filter_file {
    my $self = shift;
    if (@_) {
        my $file = shift;
        $self->{'filter_file'} = $file;
        my $filter = _read_filter_file($file);
        $self->filter($filter);
        $self->{'filter_string'} = "FILE: $file";
    }
    return $self->{'filter_file'};
}    

sub mode {
    my $self = shift;
    if (@_) {
        $self->{'mode'} = shift;
    }
    return $self->{'mode'};
}    

sub accumulate {
    my $self = shift;
    my $acc_fun_code = $self->_make_accumulator();
    print "accumulation function = $acc_fun_code\n" if ($Debug);
    my $acc_fun = eval $acc_fun_code;
    print STDERR "accumulate eval error: $@\n" if $@;
    &$acc_fun($self);
}    

# printout calls the print routine generators and prints things out
# for the mode

sub printout {
    my $self = shift;
    $self->_derive_fields();
    $self->_print_header() if ($self->print_header());
    my $mode = $self->mode();
    if ($mode eq 'summary') {
        my $sortfun_code = $self->_make_sortfun($mode);
        print "summary sortfun = $sortfun_code\n" if ($Debug);
        my $printfun_code = $self->_make_printfun($mode);
        print "summary printfun = $printfun_code\n" if ($Debug);
        my $sortfun = eval $sortfun_code;
        print STDERR "sortfun eval error: $@\n" if $@;
        my $printfun = eval $printfun_code;
        print STDERR"printfun eval error: $@\n" if $@;
        foreach my $hkey (&$sortfun($self,keys %{$self->{'summary'}})) {
            &$printfun($self, $hkey);
        }
        return;
    }
    if (($mode eq 'hostpair') || ($mode eq 'portpair')) {
        my $hostpair_sortfun_code = $self->_make_sortfun('hostpair');
        print "hostpair sortfun = $hostpair_sortfun_code\n" if ($Debug);
        my $hostpair_printfun_code = $self->_make_printfun('hostpair');
        print "hostpair printfun = $hostpair_printfun_code\n" if ($Debug);
        my $hostpair_sortfun = eval $hostpair_sortfun_code;
        print STDERR "hostpair sortfun eval error: $@\n" if $@;
        my $hostpair_printfun = eval $hostpair_printfun_code;
        print STDERR "hostpair printfun eval error: $@\n" if $@;
        my $portpair_printfun;
        my $portpair_sortfun;
        if ($mode eq 'portpair') {
            my $portpair_sortfun_code = $self->_make_sortfun('portpair');
            print "portpair sortfun = $portpair_sortfun_code\n" if ($Debug);
            my $portpair_printfun_code = $self->_make_printfun('portpair');
            print "portpair printfun = $portpair_printfun_code\n" if ($Debug);
            $portpair_sortfun = eval $portpair_sortfun_code;
            print STDERR "portpair sortfun eval error: $@\n" if $@;
            $portpair_printfun = eval $portpair_printfun_code;
            print STDERR "portpair printfun eval error: $@\n" if $@;
        }
        foreach my $hkey (&$hostpair_sortfun($self, keys %{$self->{'pairs'}})){
            &$hostpair_printfun($self,$hkey);
            if ($mode eq 'portpair') {
                foreach my $pkey (&$portpair_sortfun($self, $hkey,
                            keys %{$self->{'pairs'}->{$hkey}->{'ports'}})) {
                    &$portpair_printfun($self, $hkey, $pkey);
                }
            }
        }
    }    
}


sub _read_filter_file {
    my $filename = shift;
    my @filter;
    local $_;
    local *INFILE;
    open(INFILE, "$filename") or die "Cannot open $filename\n";
    while (<INFILE>) {
    chomp;
    # next if /whitespace # anything/ or /whitespace/
        next if ((/^(\s*\#.*$)/) || (/^\s*$/));
    # remove comments in non-whitespace, non-^# lines
        s/\#.*//;
    # remove leading/trailing whitespace;
        s/^\s+//;
        s/\s+$//;
        push(@filter, $_);
    }
    close INFILE;
    return join(' ',@filter);
}    


# this is pretty crude:  just read in the file and glue all the lines
# together with commas.  We'll then pass this string to the parse
# routines.

sub _read_file {
    my $filename = shift;
    my @stuff = ();
    local $_;
    local *INFILE;
    open(INFILE, "$filename") or die "Cannot open $filename\n";
    while (<INFILE>) {
        chomp;
    # next if /whitespace #anything/ or /whitespace/
        next if ((/^(\s*\#.*$)/) || (/^\s*$/));
    # remove comments in non-whitespace, non-^# lines
        s/\#.*//;
    # remove whitespace
        s/\s+//g;
        push(@stuff, $_)
    }
    close INFILE;
    return join(',',@stuff);
}    
    

# pass a ref to a patricia trie structure and comma-separated list of
# hostnames, ip numbers or network blocks (eg 128.2.0.0/16).

sub _parse_hoststring {
    my $pt = shift;
    my $hoststring = shift;
    return undef unless ($pt && $hoststring);
    foreach my $h (split /,/, $hoststring) {
        if ($h =~ m/^[\d\.\/]+$/) {            # assume ip or net block
            $pt->add_string($h);
        } else {                                # assume hostname
            my $ipnum = gethostbyname($h);
            die "Unable to resolve $h\n" unless $ipnum;
            $pt->add_string(inet_ntoa($ipnum));
            _add_to_hostcache($h,inet_ntoa($ipnum));
        }
    }
}    

sub _port_ok {
    my $p = shift;
    if (($p =~ m/^\d+$/) && ($p >= 0) && ($p <= 65535)) {
        return 1;
    } else {
        die "Invalid port number: $p\n";
    }
}



sub _parse_portstring {
    my @ports = @{shift()};
    my $portstring = shift;
    foreach my $p (split /,/, $portstring) {
        if (($p =~ m/^\d+$/) && _port_ok($p)) {
            $ports[$p] = 1;
        } elsif (($p =~ m/^(\d+)-(\d+)$/)) {
            my $p1 = $1;
            my $p2 = $2;
            die "Bad port range: $p1-$p2\n" unless ($p1 <= $p2);
            if (_port_ok($p1) && _port_ok($p2)) {
                for (my $i = $p1; $i <= $p2; $i++) {
                    $ports[$i] = 1;
                }
        } 
        } else {
            my $portnum = getservbyname($p, 'tcp');
            $portnum = getservbyname($p, 'udp')
                 unless defined($portnum);
            if ($portnum) {
                $ports[$portnum] = 1;
            } else { 
                die "Bad port specification: $portstring\n";
            }
        }  
    }
    return \@ports;
}

# whether or not we need optional field
# _need(mode,field)
#
sub _need_field {
    my $self = shift;
    my $mode = shift;
    my $field = shift;
    my %needed;
    if ($mode eq 'summary') {
        %needed = map {$_ => 1} (split(',', $self->summary_sortby()),
                                 split(',', $self->summary_output()));
    }
    if ($mode eq 'hostpair') {
        %needed = map {$_ => 1} (split(',', $self->hostpair_sortby()),
                                 split(',', $self->hostpair_output()));
    }
    if ($mode eq 'portpair') {
        %needed = map {$_ => 1} (split(',', $self->portpair_sortby()),
                                 split(',', $self->portpair_output()));
    }
    return (exists $needed{$field}) ? 1 : 0;
}


# these are the accessors for the summary/hostpair/portpair data

my %summary_fields = (
    h1_bytes => '{\'summary\'}->{\'hostkey\'}->{\'h1_bytes\'}',
    h2_bytes => '{\'summary\'}->{\'hostkey\'}->{\'h2_bytes\'}',
     h1_pkts => '{\'summary\'}->{\'hostkey\'}->{\'h1_pkts\'}',
     h2_pkts => '{\'summary\'}->{\'hostkey\'}->{\'h2_pkts\'}',
     records => '{\'summary\'}->{\'hostkey\'}->{\'records\'}',
       h1_ip => '{\'summary\'}->{\'hostkey\'}->{\'h1_ip\'}',
 total_bytes => '{\'summary\'}->{\'hostkey\'}->{\'total_bytes\'}',
  total_pkts => '{\'summary\'}->{\'hostkey\'}->{\'total_pkts\'}',
     h1_name => '{\'summary\'}->{\'hostkey\'}->{\'h1_name\'}',
   num_peers => '{\'summary\'}->{\'hostkey\'}->{\'num_peers\'}',
       peers => '{\'summary\'}->{\'hostkey\'}->{\'peers\'}->{\'h2_ip\'}');
       
my %hostpair_fields = (
   h1_bytes => '{\'pairs\'}->{\'hostkey\'}->{\'h1_bytes\'}',
   h2_bytes => '{\'pairs\'}->{\'hostkey\'}->{\'h2_bytes\'}',
   h1_pkts  => '{\'pairs\'}->{\'hostkey\'}->{\'h1_pkts\'}',
   h2_pkts  => '{\'pairs\'}->{\'hostkey\'}->{\'h2_pkts\'}',
   records  => '{\'pairs\'}->{\'hostkey\'}->{\'records\'}',
total_bytes => '{\'pairs\'}->{\'hostkey\'}->{\'total_bytes\'}',
total_pkts  => '{\'pairs\'}->{\'hostkey\'}->{\'total_pkts\'}',
     h1_ip  => '{\'pairs\'}->{\'hostkey\'}->{\'h1\'}',
     h2_ip  => '{\'pairs\'}->{\'hostkey\'}->{\'h2\'}',
  h1_ports  => '{\'pairs\'}->{\'hostkey\'}->{\'h1_ports\'}->{\'port\'}',
  h2_ports  => '{\'pairs\'}->{\'hostkey\'}->{\'h2_ports\'}->{\'port\'}',
      h1_pc => '{\'pairs\'}->{\'hostkey\'}->{\'h1_pc\'}',
      h2_pc => '{\'pairs\'}->{\'hostkey\'}->{\'h2_pc\'}',
   h1_name  => '{\'pairs\'}->{\'hostkey\'}->{\'h1_name\'}',
   h2_name  => '{\'pairs\'}->{\'hostkey\'}->{\'h2_name\'}');
         
my %portpair_fields = (
stime =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'stime\'}',
etime =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'etime\'}',
h1_bytes =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h1_bytes\'}',
h2_bytes =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h2_bytes\'}',
h1_pkts =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h1_pkts\'}',
h2_pkts =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h2_pkts\'}',
h1_port =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h1_port\'}',
h2_port =>
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'h2_port\'}',
proto => 
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'proto\'}',
records => 
    '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'records\'}',
total_bytes => 
  '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'total_bytes\'}',
total_pkts => 
  '{\'pairs\'}->{\'hostkey\'}->{\'ports\'}->{\'portkey\'}->{\'total_pkts\'}');

# _access(accessor, [field, variable], [field, variable],...);
# this is just used to make some of these long accessors a bit shorter,
# not to isolate all the access functions.

sub _access {
    my $accessor = shift;
    my @substitutions = @_;
    foreach my $s (@substitutions) {
        $accessor =~ s/\'$s->[0]\'/$s->[1]/;
    } 
    return '$self->' . $accessor;
}

sub _gen_counters {          # basic byte/packet/record counting
    my $self = shift;
    my $mode = shift;        # 'summary' or 'hostpair' or 'portpair'
    my $h1 = shift;          # 'src' or 'dst'
    my $h2 = shift;          # 'dst' or 'src'
    my %fields = (h1_pkts => '$' . $h1 . 'pkts',
                  h2_pkts => '$' . $h2 . 'pkts',
                  h1_bytes => '$' . $h1 . 'bytes',
                  h2_bytes => '$' . $h2 . 'bytes');

    my @subs = ();
    my $code = "";
    my %accessors;
    my $h1ip = '$' . $h1 . 'ip';
    my $h2ip = '$' . $h2 . 'ip';
    if ($mode eq 'summary') {
        %accessors = %summary_fields;
        @subs = (['hostkey',$h1ip]);
        if ($self->_need_field($mode, 'num_peers')) {
            $code .= _access($accessors{'peers'},
                     ['hostkey',$h1ip],['h2_ip', $h2ip]) . '= 1' . ";\n";
        }             

    } elsif ($mode eq 'hostpair') {
        %accessors = %hostpair_fields;
        @subs = (['hostkey','$hostkey']);
        if (($self->_need_field($mode,'h1_ports')) || 
            ($self->_need_field($mode,'h1_portlist')) ||
            ($self->_need_field($mode,'h1_pc'))) {
            $code .= _access($accessors{'h1_ports'},
                ['hostkey', '$hostkey'],['port','$'.$h1.'p']).'= 1'.";\n";
        }
        if (($self->_need_field($mode,'h2_ports')) ||
            ($self->_need_field($mode,'h2_portlist')) ||
            ($self->_need_field($mode,'h2_pc'))) {
            $code .= _access($accessors{'h2_ports'},
                ['hostkey', '$hostkey'],['port', '$'.$h2.'p']).'= 1'.";\n";
        }        
    } elsif ($mode eq 'portpair') {
        %accessors = %portpair_fields;
        @subs = (['hostkey','$hostkey'], ['portkey', '$portkey']);
            # no optional fields for portpairs.
    } else {
        print "Bad accessor in _gen_counters\n";
        exit(0);
    }
    foreach my $field (sort keys %fields) {
        $code .= _access($accessors{$field}, @subs) . '+=' .
                                        $fields{$field} . ";\n";
    }
    $code .= _access($accessors{'records'} , @subs) . '++' . ";\n";
    print "_gen_counters = $code\n" if ($Debug);
    return $code;
}    

sub _record_matcher {
    my $self = shift;
    my $f1 = shift;    # gets passed either 'src,dst' or 'dst,src'
    my $f2 = shift;
    my @cond = ();
    my $f1_ip = '$' . $f1 . 'ip';
    my $f2_ip = '$' . $f2 . 'ip';
    my $f1_p = '$' . $f1 . 'p';
    my $f2_p = '$' . $f2 . 'p';
    return '(1)' unless (($self->{'hostset1'}) || ($self->{'hostset2'}) ||
                     ($self->{'portset1'}) || ($self->{'portset2'}));
    push @cond, '($self->{\'hostset1\'}->match_string(' . $f1_ip . '))'
               if ($self->{'hostset1'});
    push @cond, '((' . $f1_p . '=~ m/^\d/o) && ($self->{\'portset1\'}->['
                 . $f1_p . ']))' if ($self->{'portset1'});
    push @cond, '($self->{\'hostset2\'}->match_string(' . $f2_ip . '))'
               if ($self->{'hostset2'});
    push @cond, '((' . $f2_p . '=~ m/^\d/o) && ($self->{\'portset2\'}->['
                 . $f2_p . ']))'   if ($self->{'portset2'});
    return '(' . join('&&', @cond) . ')';
}    
    

sub _make_accumulator {
    my $self = shift;
    my $mode = $self->mode();
    my $accumulator = "";
    die "Unknown mode: $mode\n" unless  (($mode eq 'summary') ||
                                         ($mode eq 'hostpair') ||
                                         ($mode eq 'portpair'));
    my $make_hostkey = <<'_END_';
    if ($srcip lt $dstip) {
        $hostkey = join('-', $srcip, $dstip);
    } else {
        $hostkey = join('-', $dstip, $srcip);
    }
      
_END_

    my $make_portkey_h1_is_src = <<'_END_';
    if (($srcp =~ m/^\d/o) && ($dstp =~ m/^\d/o)) {
        $portkey = join('-', $srcp, $dstp, $typ);
    } else {
        $portkey = join('-', '77777','77777', $typ);  # bogus port number
    }        

_END_

    my $make_portkey_h2_is_src = <<'_END_';
    if (($srcp =~ m/^\d/o) && ($dstp =~ m/^\d/o)) {
        $portkey = join('-', $dstp, $srcp,  $typ);
    } else {
        $portkey = join('-', '77777', '77777', $typ);
    }        
_END_

    my $init_port_timevals = <<'_END_';
    unless ($self->{'pairs'}->{$hostkey}->{'ports'}->{$portkey}->{'stime'}) {
        $self->{'pairs'}->{$hostkey}->{'ports'}->{$portkey}->{'stime'} =
            $s_time;
        $self->{'pairs'}->{$hostkey}->{'ports'}->{$portkey}->{'etime'} =
            $e_time;
    }
_END_

    $accumulator .= <<'_END_';
    
sub {
    my $MAX_TIME = 2147483647;  
    my $MIN_TIME = 0;
    print "# No files to process.\n"
            unless ($self->{'argus_files'});
    die "Need RA program with rarc specified\n"
         unless (($self->ra_prog()) && ($self->rarc()));
    $self->{'time'} = time();
    foreach my $file (@{$self->{'argus_files'}}) {
        my $first_match = $MAX_TIME;
        my $last_match = $MIN_TIME; 
        my $file_match_count = 0;  
        my $file_rec_count = 0;
        local *RA;
        local $_;
        my $ra = $self->{'ra_prog'};        
        my @ra_args = ('-F', $self->rarc());  
        push(@ra_args, $self->ra_extra_args())if ($self->ra_extra_args());
        push(@ra_args, '-', $self->{'filter'});
        push(@ra_args, '-r', $file);
        my $hostkey;
        my $portkey;
        my $pid = open(RA, "-|");
        if ($pid) {
            while (<RA>) {  
                 my ($s_time,$e_time, $typ, $srcip, $srcp, $dir, 
                     $dstip,$dstp, $srcpkts, $dstpkts, $srcbytes, $dstbytes,
                     $state) = m/^([^,]*),([^,]*),([^,]*),([^,]*),
                           ([^,]*),([^,]*),([^,]*),([^,]*),([^,]*),
                           ([^,]*),([^,]*),([^,]*),([^,]*$)/ox;
                 $file_rec_count++;
_END_
# summary mode gets handled a bit differently than the hostpair mode since
# both h1 and h2 may match and both need to have byte counts totaled up
    if ($mode eq 'summary') {
        $accumulator .= 'if' . $self->_record_matcher('src','dst') . '{';
        $accumulator .= '$first_match = $s_time if ($s_time < $first_match);';
        $accumulator .= '$last_match = $e_time if ($last_match < $e_time);';
        $accumulator .= '$file_match_count++;';
        $accumulator .= $self->_gen_counters($mode,'src','dst') . '}';
        $accumulator .= 'if' . $self->_record_matcher('dst','src') . '{';
        $accumulator .= '$first_match = $s_time if ($s_time < $first_match);';
        $accumulator .= '$last_match = $e_time if ($last_match < $e_time);';
        $accumulator .= '$file_match_count++;';
        $accumulator .= $self->_gen_counters($mode,'dst','src') . '}';
    }
# end of summary mode-specific code

    if (($mode eq 'hostpair') || ($mode eq 'portpair')) {
        $accumulator .= 'if' . $self->_record_matcher('src','dst') . '{';
### src matches h1/p1 and dst matches h2/p2
        $accumulator .= $make_hostkey;
        $accumulator .= <<'_END_';
        $first_match = $s_time if ($s_time < $first_match);
        $last_match = $e_time if ($last_match < $e_time);
        $file_match_count++;
        if (! exists($self->{'pairs'}->{$hostkey})) {
            $self->{'pairs'}->{$hostkey}->{'h1'} = $srcip;
            $self->{'pairs'}->{$hostkey}->{'h2'} = $dstip;
        }
_END_
        $accumulator .= $self->_gen_counters('hostpair', 'src', 'dst');
        if ($mode eq 'portpair') {
            $accumulator .= $make_portkey_h1_is_src;
            $accumulator .= $self->_gen_counters('portpair', 'src' , 'dst');
            $accumulator .= $init_port_timevals;
            $accumulator .=
            _access($portpair_fields{'stime'},['hostkey' , '$hostkey'],
                                              ['portkey', '$portkey']) .
            ' = $s_time if (' .
            _access($portpair_fields{'stime'},['hostkey' , '$hostkey'],
                                              ['portkey','$portkey'])  .
            ' > $s_time);';
            $accumulator .= 
            _access($portpair_fields{'etime'},['hostkey' , '$hostkey'],
                                             ['portkey', '$portkey']) .
            ' = $e_time if (' .
            _access($portpair_fields{'etime'},['hostkey' , '$hostkey'],
                                             ['portkey','$portkey'])  .
            ' < $e_time);';
        }
### now handle case where dst matches h1/p1 and src matches h2/p1
        $accumulator .= '} elsif '. $self->_record_matcher('dst','src') . '{';
### 
        $accumulator .= $make_hostkey;
        $accumulator .= <<'_END_';
        $first_match = $s_time if ($s_time < $first_match);
        $last_match = $e_time if ($last_match < $e_time);
        $file_match_count++;
        if (! exists($self->{'pairs'}->{$hostkey})) {
            $self->{'pairs'}->{$hostkey}->{'h1'} = $dstip;
            $self->{'pairs'}->{$hostkey}->{'h2'} = $srcip;
        }
_END_
        $accumulator .= $self->_gen_counters('hostpair', 'dst', 'src');
        if ($mode eq 'portpair') {
            $accumulator .= $make_portkey_h2_is_src;
            $accumulator .= $self->_gen_counters('portpair','dst', 'src');
            $accumulator .= $init_port_timevals;
            $accumulator .=
            _access($portpair_fields{'stime'},['hostkey' , '$hostkey'],
                                              ['portkey', '$portkey']) .
            ' = $s_time if (' . 
            _access($portpair_fields{'stime'},['hostkey' , '$hostkey'],
                                              ['portkey' , '$portkey'])  .
            ' > $s_time);';
            $accumulator .= 
            _access($portpair_fields{'etime'},['hostkey' , '$hostkey'],
                                              ['portkey', '$portkey']) .
            ' = $e_time if (' .
            _access($portpair_fields{'etime'},['hostkey' , '$hostkey'],
                                              ['portkey',  '$portkey'])  .
            ' < $e_time);';
            
        }
        $accumulator .= '}';  # end of elsif
    }

### at this point, just need to clean up and store end-of-file statistics.

    $accumulator .= '}';   # end of 'while'
    $accumulator .= <<'_END_';
        close RA;
        $self->{'filedata'}->{$file}->{'first_match'} = $first_match;
        $self->{'filedata'}->{$file}->{'last_match'} = $last_match;
        $self->{'filedata'}->{$file}->{'matches'} = $file_match_count;
        $self->{'filedata'}->{$file}->{'records'} = $file_rec_count;
     } else {      # child
         exec($ra, @ra_args)
             or die "Cannot exec $ra @ra_args : $! \n";
        } 
_END_
    $accumulator .= '}';  # end of foreach
    $accumulator .= '};';  # end of accumulator sub
    
}  # end of _make_accumulator


sub _print_header {    # print "meta information" (file names, etc)
    my $self = shift;
    my $now = localtime($self->{'time'});  # perhaps this should be $then
    my $mode = $self->mode();
    print "# HostPairs (mode: $mode) output generated:  $now\n";
    print "# Files processed:\n";
    foreach my $file (@{$self->{'argus_files'}}) {
        my $fm = localtime(int $self->{'filedata'}->{$file}->{'first_match'});
        my $lm = localtime(int $self->{'filedata'}->{$file}->{'last_match'});
        my $fmc = $self->{'filedata'}->{$file}->{'matches'};
        my $frc = $self->{'filedata'}->{$file}->{'records'};
        print "#    File = $file\n";
        print "#        Number of records processed after filtering = $frc\n";
        print "#        Number of matching records = $fmc\n";
        if ($fmc) {
            print "#        Start time of first matching record = $fm\n";
            print "#        End time of last matching record = $lm\n";
        }
        print "#\n";
    }
    my $h1 = $self->hostset1();
    my $h2 = $self->hostset2();
    my $p1 = $self->portset1();
    my $p2 = $self->portset2();
    my $filter = $self->filter();
    if ($h1) {
        my $text = _format_paramset('#*H1*=*','#**', $h1);
        print "$text\n";
    }    

    if ($h2) {
        my $text = _format_paramset('#*H2*=*','#**', $h2);
        print "$text\n";
    }
    if ($p1) {
        my $text = _format_paramset('#*P1*=*','#**', $p1);
        print "$text\n";
    }    
    if ($p2) {
        my $text = _format_paramset('#*P2*=*','#**', $p2);
        print "$text\n";
    }
    if ($filter) {
        my $text = wrap('# Filter = ', 
                        '#          ', $filter);
        print "$text\n";
    }
    my $out;
    if ($mode eq 'summary') {
        $out = $self->summary_output();
    } else {
        $out = $self->hostpair_output();
      }
    $out =~ s/,/ /g;
    my $text = wrap('# Host Output = ', 
                    '#               ', $out);
    print "$text\n";
    
    if ($mode eq 'portpair') {
        my $portpair_out = $self->portpair_output();
        $portpair_out =~ s/,/ /g;
        my $portpair_text = wrap('# Port Output = ', 
                                 '#               ', $portpair_out);
        print "$portpair_text\n";
    }   
    print "\n";
}        

# need to do a bit of fu to make long port/host/filter specs look nice

sub _format_paramset {
    my $header = shift;
    my $footer = shift;
    my $params = shift;
    if ($params =~ m/FILE:/) {
        my $text =  wrap($header, $footer, $params);
        $text =~ s/\*/ /g;
        return $text;
    } else {
    
        $params =~ s/,/ /g;
        my $text = wrap($header, $footer, $params);
        $text =~ s/ /,/g;
        $text =~ s/\n/,\n/g;
        $text =~ s/\*/ /g;
        return $text;
    }        
}

# It is possible to sort and display data that is derived from the data
# present after the argus files are read (eg hostnames, total_bytes,
# num hosts, etc).  Missing values will be filled in here according to
# needed sort/print fields, in order to make things easier for the sort 
# and printing routines.  This code has grown pretty ugly as fields have
# been added.
# Ideally, the sort routine would just generate these as it goes and also
# cache them for later printing out.  



sub _derive_fields {
    my $self = shift;
    my $mode = $self->mode();
    
    if ($mode eq 'summary') {
        my $need_total_bytes = $self->_need_field($mode,'total_bytes');
        my $need_total_pkts = $self->_need_field($mode,'total_pkts');
        my $need_num_peers = $self->_need_field($mode,'num_peers');
        my $need_h1_name = $self->_need_field($mode,'h1_name');
        my $need_h1_ip = $self->_need_field($mode, 'h1_ip');
        if ($need_total_bytes || $need_total_pkts || $need_num_peers ||
            $need_h1_name || $need_h1_ip) {
            foreach my $ip (keys %{$self->{'summary'}}) {
                $self->{'summary'}->{$ip}->{'total_bytes'} =
                    $self->{'summary'}->{$ip}->{'h1_bytes'} +
                    $self->{'summary'}->{$ip}->{'h2_bytes'}
                    if ($need_total_bytes);
                $self->{'summary'}->{$ip}->{'total_pkts'} =
                    $self->{'summary'}->{$ip}->{'h1_pkts'} +
                    $self->{'summary'}->{$ip}->{'h2_pkts'} 
                                                if ($need_total_pkts);
                # sorta silly but it makes the sort function more consistant.
                $self->{'summary'}->{$ip}->{'h1_ip'} = $ip if ($need_h1_ip);
                $self->{'summary'}->{$ip}->{'num_peers'} =
                    scalar keys(%{$self->{'summary'}->{$ip}->{'peers'}})
                                                    if ($need_num_peers);
                if ($need_h1_name) {
                    my $h = _resolve_hosts($ip);
                    $self->{'summary'}->{$ip}->{'h1_name'} = $$h{$ip};
                }
            }
        }             
        return;
    }
    
    if (($mode eq 'hostpair') || ($mode eq 'portpair')) {
        
        my $hpneed_total_bytes = $self->_need_field('hostpair','total_bytes');
        my $hpneed_total_pkts = $self->_need_field('hostpair','total_pkts');
        my $hpneed_h1_pc = $self->_need_field('hostpair','h1_pc');
        my $hpneed_h2_pc = $self->_need_field('hostpair','h2_pc');
        my $hpneed_h1_pl = $self->_need_field('hostpair','h1_portlist');
        my $hpneed_h2_pl = $self->_need_field('hostpair','h2_portlist');
        my $hpneed_h1_name = $self->_need_field('hostpair','h1_name');
        my $hpneed_h2_name = $self->_need_field('hostpair','h2_name');
        my $pp_need_total_bytes = (($mode eq 'portpair') &&
            $self->_need_field('portpair', 'total_bytes'));
        my $pp_need_total_pkts =  (($mode eq 'portpair') &&
            $self->_need_field('portpair', 'total_pkts'));
        my $pp_need_h1_port =  (($mode eq 'portpair') &&
            $self->_need_field('portpair', 'h1_port'));
        my $pp_need_h2_port =  (($mode eq 'portpair') &&
            $self->_need_field('portpair', 'h2_port'));
        my $pp_need_proto =  (($mode eq 'portpair') &&
            $self->_need_field('portpair', 'proto'));
        if ($hpneed_h1_pc || $hpneed_h2_pc || $hpneed_h1_name ||
            $hpneed_h2_name || $hpneed_total_bytes || $hpneed_total_pkts ||
            $pp_need_total_bytes || $pp_need_total_pkts || $hpneed_h1_pl ||
            $hpneed_h2_pl || $pp_need_h1_port || $pp_need_h2_port || 
            $pp_need_proto) {
            foreach my $pair (keys %{$self->{'pairs'}}) {
                if ($hpneed_h1_pc || $hpneed_h2_pc || $hpneed_h1_pl ||
                    $hpneed_h2_pl) {  # remove bogus entry from porthash
                      delete($self->{'pairs'}->{$pair}->{'h1_ports'}->{""}) 
                          if ($hpneed_h1_pc || $hpneed_h1_pl);
                      delete($self->{'pairs'}->{$pair}->{'h2_ports'}->{""})
                          if ($hpneed_h2_pc || $hpneed_h2_pl);
                }
                $self->{'pairs'}->{$pair}->{'total_bytes'} =
                    $self->{'pairs'}->{$pair}->{'h1_bytes'} +
                    $self->{'pairs'}->{$pair}->{'h2_bytes'} 
                    if ($hpneed_total_bytes);
                    
                $self->{'pairs'}->{$pair}->{'total_pkts'} = 
                    $self->{'pairs'}->{$pair}->{'h1_pkts'} +
                    $self->{'pairs'}->{$pair}->{'h2_pkts'}
                    if ($hpneed_total_pkts);
                    
                $self->{'pairs'}->{$pair}->{'h1_pc'} =
                    keys(%{$self->{'pairs'}->{$pair}->{'h1_ports'}})
                    if ($hpneed_h1_pc);
                        
                $self->{'pairs'}->{$pair}->{'h2_pc'} =
                    keys(%{$self->{'pairs'}->{$pair}->{'h2_ports'}})
                    if ($hpneed_h2_pc);
                     
                if ($hpneed_h1_name) {
                    my $h = _resolve_hosts($self->{'pairs'}->{$pair}->{'h1'});
                    $self->{'pairs'}->{$pair}->{'h1_name'} = 
                        $$h{$self->{'pairs'}->{$pair}->{'h1'}};
                }

                if ($hpneed_h2_name) {
                    my $h = _resolve_hosts($self->{'pairs'}->{$pair}->{'h2'});
                    $self->{'pairs'}->{$pair}->{'h2_name'} = 
                        $$h{$self->{'pairs'}->{$pair}->{'h2'}};
                }

                if ($pp_need_total_bytes || $pp_need_total_pkts ||
                    $pp_need_proto || $pp_need_h1_port || $pp_need_h2_port) {
                    foreach my $pp
                        (keys %{$self->{'pairs'}->{$pair}->{'ports'}}) {
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'total_bytes'} =
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'h1_bytes'} +
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'h2_bytes'}
                            if ($pp_need_total_bytes);
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'total_pkts'} =
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'h1_pkts'} +
                        $self->{'pairs'}->{$pair}->{'ports'}
                                ->{$pp}->{'h2_pkts'}
                            if ($pp_need_total_pkts);
                        if ($pp_need_proto || $pp_need_h1_port || 
                            $pp_need_h2_port) {
                            my ($port_h1, $port_h2, $proto) = split(/-/,$pp);
                            $self->{'pairs'}->{$pair}->{'ports'}->{$pp}->
                                        {'h1_port'} = $port_h1;
                            $self->{'pairs'}->{$pair}->{'ports'}->{$pp}->
                                        {'h2_port'} = $port_h2;
                            $self->{'pairs'}->{$pair}->{'ports'}->{$pp}->
                                        {'proto'} = $proto;
                        }                                            
                   }
                }

            }                    
        }

    }
}    





# eval up a method to print out the fields we need in the format
# we want.  for summary and hostpair mode, new method will take single
# unique key for arg.  for portpair mode, new method will take both
# unique keys needed (hostpair key and portpair key). method will print
# out single line.

sub _make_printfun {
    my $self = shift;
    my $mode = shift;
    my $printstring;
    my $formatstring;
    my $access;
    if ($mode eq 'summary') {
        $printstring = $self->summary_output();
        $formatstring = $self->summary_format();
        $access = \%summary_fields;
    } elsif ($mode eq 'hostpair') {
        $printstring = $self->hostpair_output();
        $formatstring = $self->hostpair_format();
        $access = \%hostpair_fields;
    } else {
        $printstring = $self->portpair_output();
        $formatstring = $self->portpair_format();
        $access = \%portpair_fields;
    }
    my @printfields = split /,/, $printstring;
    my @fields = ();
    my $fun = 'sub { my $self = shift; my $hostkey = shift;';
    $fun .= 'my $portkey = shift;' if ($mode eq 'portpair');
    foreach my $f (@printfields) {
        if ($mode eq 'summary') {
            push @fields, _access($$access{$f},['hostkey', '$hostkey']);
        } elsif ($mode eq 'hostpair') {
            if (($f eq 'h1_portlist') || ($f eq 'h2_portlist')) {
                my $h1_portlistgen =
           '_portlist(keys %{$self->{\'pairs\'}->{$hostkey}->{\'h1_ports\'}})';
                my $h2_portlistgen = 
            '_portlist(keys %{$self->{\'pairs\'}->{$hostkey}->{\'h2_ports\'}})';
                push @fields, $h1_portlistgen if ($f eq 'h1_portlist');
                push @fields, $h2_portlistgen if ($f eq 'h2_portlist');
            } else {
                push @fields, _access($$access{$f},['hostkey', '$hostkey']);
            }
        } else {                            # portpair mode
            if (($f eq 'stime') || ($f eq 'etime')) {
                push(@fields, 'POSIX::strftime(' . '"' .
                  $self->{'time_format'} . '", localtime(int ' .
                  _access($$access{$f}, ['hostkey', '$hostkey'],
                                        ['portkey', '$portkey']) . '))');
            } elsif (($f eq 'h1_port') || ($f eq 'h2_port')) {
                push @fields, '_port_name($self,' . 
                _access($$access{$f},['hostkey', '$hostkey'],
                                     ['portkey', '$portkey']) . ',' .
                _access($$access{'proto'},['hostkey', '$hostkey'],
                                          ['portkey', '$portkey']) . ')';
            } else {
                push @fields, _access($$access{$f},['hostkey', '$hostkey'],
                                                   ['portkey', '$portkey']);
            }
       }
    }
    $fun .= 'printf ' . join(',', '"'. $formatstring . '"', @fields) . ';' ;
    $fun .= '};';
    return $fun;
}    


# some things need 'cmp' and some things need '<=>'. 

my %comparators = (
h1_name => 
    '$Net::Argus::HostPairs::a->[#] cmp $Net::Argus::HostPairs::b->[#]',
h2_name =>
    '$Net::Argus::HostPairs::a->[#] cmp $Net::Argus::HostPairs::b->[#]',
h1_ip => 
    '$Net::Argus::HostPairs::a->[#] cmp $Net::Argus::HostPairs::b->[#]',
h2_ip =>
    '$Net::Argus::HostPairs::a->[#] cmp $Net::Argus::HostPairs::b->[#]',
proto =>
    '$Net::Argus::HostPairs::a->[#] cmp $Net::Argus::HostPairs::b->[#]',
h1_bytes =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h2_bytes =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h1_pkts =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h2_pkts =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
total_bytes =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
total_pkts =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
records =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h1_pc =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h2_pc =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h1_port =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
h2_port =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
stime =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
etime =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]',
num_peers =>
    '$Net::Argus::HostPairs::a->[#] <=> $Net::Argus::HostPairs::b->[#]');


#
# _make_sortfun is a general schwartzian transform sort function maker.
# It takes an argument specifying which type of data (summary, hostpairs
# or portpairs) is going to get sorted.
# 

sub _make_sortfun {
    my $self = shift;
    my $mode = shift;
    my $sortstring;
    my $access;
    my $sortorder;
    if ($mode eq 'summary') {
        $sortstring = $self->summary_sortby();
        $access = \%summary_fields;
        $sortorder = $self->{'summary_sort_reverse'};
    } elsif ($mode eq 'hostpair') {
        $sortstring = $self->hostpair_sortby();
        $access = \%hostpair_fields;
        $sortorder = $self->{'hostpair_sort_reverse'};
    } else {
        $sortstring = $self->portpair_sortby();
        $access = \%portpair_fields;
        $sortorder = $self->{'portpair_sort_reverse'};        
    }
    my @sortfields = split(/,/, $sortstring);
    die "Oooops...no sortfields\n" unless (@sortfields);
    my @keycache = ();
    my $numfields = $#sortfields + 1;
    push @keycache, '$_';
    my $sortfunct = 'sub { my $self = shift; ';
    $sortfunct .= 'my $hostpair = shift;' if ($mode eq 'portpair');
    $sortfunct .= 'map $_->[0] => sort {';
    my $accessor;
    for (1..$numfields) {
        my $field = $sortfields[$_ - 1];
        my $compare = $comparators{$field};
        $compare =~ s/(.+?)(a->)(.+?)(b->)/$1$4$3$2/ if ($$sortorder{$field});
        $compare =~ s/\#/$_/g;
        $sortfunct .= $compare;
        $sortfunct .= ' || ' if ($_ < $numfields);
        if (($mode eq 'summary') || ($mode eq 'hostpair')) {
            if (($field eq 'h1_ip') || ($field eq 'h2_ip')) {
                push @keycache, 
                    'inet_aton(' . _access($$access{$field},['hostkey', '$_']) 
                                                       .')';
            } else {
                push @keycache, _access($$access{$field},['hostkey', '$_']);
              }
        } else {
                push @keycache, _access($$access{$field},
                     ['hostkey', '$hostpair'], ['portkey', '$_']);
          }
    }
    my $cache = '[' . join(',', @keycache) . ']';
    $sortfunct .= '}';
    my $sorter = $sortfunct . 'map' . $cache . '=> @_;}';
    return $sorter;
}    


sub _port_name {
    my $self = shift;
    my $p = shift;
    my $proto = shift;
    my $name;
    if ($p == 77777) {                        #  bogus port number == no port
        return '*';
    } elsif ($self->print_port_names()) {
        $name = getservbyport($p, $proto) if ($p && $proto);
        return $name ? $name : $p;
    } else {
        return $p;
    }
}

sub _portlist {
    my @ports = sort {$a <=> $b} @_;
    return '*' unless @ports;
    my @interlist = ();
    my $anchor = shift @ports;
    my $seq_last = $anchor;
    return $anchor unless @ports;
    my $next_num;
    do {
        $next_num = shift @ports;
        if (! $next_num) {
            my $seq = ($anchor == $seq_last) ?
                                  $anchor : $anchor . '-' . $seq_last;
            push @interlist, $seq;
        } elsif ($next_num == ($seq_last + 1)) {
            $seq_last = $next_num;
        } else {
            my $seq = ($anchor == $seq_last) ?
                       $anchor : $anchor . '-' . $seq_last;
             push @interlist, $seq;
             $anchor = $seq_last = $next_num;
        }    

    } while ($next_num);
    return join(',', @interlist);
}


{

my %host_cache;

sub _resolve_hosts {
    my @hosts = @_;
    my $h = {};
    return $h unless (@hosts);
    foreach my $host (@hosts) {
        if (exists $host_cache{$host}) {
            $$h{$host} = $host_cache{$host};
        } else {
            my $hname = gethostbyaddr((inet_aton $host), AF_INET);
            $hname = (defined $hname) ? $hname : '???';
            $host_cache{$host} = uc($hname);
            $$h{$host} = uc($hname);
        }

    }
    return $h;
}

sub _add_to_hostcache {
    my $hname = shift;
    my $ip = shift;
    $host_cache{uc($hname)} = $ip;
}    

}

1;
__END__

=head1 NAME

Net::Argus::HostPairs - Extract host pair data from an Argus data file

=head1 SYNOPSIS

    use Net::Argus::HostPairs;

    $hp = Argus::HostPairs::new->();
    $hp->argus_files("argus_data_file");
    $hp->hostset1("www.mydomain.com");    
    $hp->portset1("http,https,8001,8080); 
    $hp->hostset2("123.45.0.0/16");       
    $hp->accumulate();                    
    $hp->print_pairs();
    ...

=head1 DESCRIPTION

Net::Argus::HostPairs is a module for processing Argus data files using ra
and printing out pairs of hosts that match specified host and port
descriptions, along with associated byte and packet counts, and other data.
Pairs are specified by setting (optional) values for 'hostset1' and an
associated 'portset1', 'hostset2' and an associated 'portset2', and by
setting an optional Argus filter to further narrow down candidate hosts. If
a hostset or portset is not specified, any host/port will match. Pairs are
selected when they match all specified host and port sets, in any order (ie
no attempt is made to use flow direction or src/dst information in the Argus
data file).

Net::Argus::HostPairs only processes IP data, and all calls to ra are done
with a filter specifying ip.  Because there are a wide variety of possible
data gathering modes and fields, the routines for gathering, sorting, and
printing data are constructed based on which fields are needed, and then
evaled.

=head1 MODES

There are three modes, each offering different levels of detail.

=over 4

=item Summary mode

In summary mode, hostset 2 data is aggregated over all matching hostset 2
hosts per hostset1 host, and one line is printed per matching hostset1 host.

=item Hostpair mode

Data is aggregated for each distinct pair of hosts, and one line of data
is printed out per pair.

=item Portpair mode

Hostpair mode data is printed out, along with a breakdown for each pair
of ports involved in transaction.

=back

=head1 USAGE

See the hostpairs application for an example of how to use this module, and
for a more complete description of how it works.

=head1 BUGS

Probably lots. 

=head1 AUTHOR

Clauss Strauch (cbs@cs.cmu.edu)

=head1 COPYRIGHT

Copyright (c) 2001 Carnegie Mellon University
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

Carnegie Mellon requests users of this software to return to

Software_Distribution@CS.CMU.EDU or

Software Distribution Coordinator
School of Computer Science
Carnegie Mellon University
Pittsburgh PA 15213-3890

any improvements or extensions that they make and grant Carnegie Mellon
the rights to redistribute these changes.

=cut
