
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * rapath - print derivable path information from argus data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rapath.h>

#include <math.h>


#define RA_HASHTABLESIZE	65536

int RaFlowModelFile = 0;
int RaAllocHashTableHeaders = 0;
int RaAllocArgusRecordStore = 0;
int RaAllocArgusRecord      = 0;

struct timeval RaStarTime, RaLastTime;

#define RA_SVCPASSED	0x0100
#define RA_SVCFAILED	0x0200
#define RA_SVCTEST	(RA_SVCFAILED|RA_SVCPASSED)


struct ArgusServiceRecord {
   u_int status;
   struct ArgusRecord *argus;
};
 
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
struct ArgusRecordStore *RaThisArgusStore = NULL;

int RaInitialized = 0;


void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaWriteOut = 0;
   
      if ((RaModelerQueue = RaNewQueue(65536)) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      RaStarTime.tv_sec  = 0;
      RaStarTime.tv_usec = 0;
      RaLastTime.tv_sec  = 0;
      RaLastTime.tv_usec = 0;

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }
      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{

   if (!RaParseCompleting) {
      RaParseCompleting++;
      RaProcessQueue (RaModelerQueue, ARGUS_STOP);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void RaProcessQueue(struct RaQueueStruct *, unsigned char);

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rapath Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options]\n", ArgusProgramName);

   fprintf (stderr, "options: -D <level>     specify debug level\n");
   fprintf (stderr, "         -n             don't convert numbers to names.\n");
   fprintf (stderr, "         -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "         -r <filelist>  read argus data <filelist>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "         -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                            mm/dd[/yy]\n");
   fprintf (stderr, "         -T <secs>      attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   exit(1);
}



void RaPrintArgusPath (struct ArgusRecordStore *);
struct ArgusRecordStore *RaFindRevArgusRecord(struct ArgusRecord *);

struct ArgusServiceRecord ArgusThisSrv;


void RaProcessRecord (struct ArgusRecord *argus)
{  
   if (((argus->argus_far.status & ARGUS_ICMPUNREACH_MAPPED) ||
        (argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)) &&
                         (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS)) {

     argus->argus_far.flow.ip_flow.ip_id = 0;

      switch (argus->argus_far.flow.ip_flow.ip_p) {
         case IPPROTO_UDP:
         case IPPROTO_TCP:
            argus->argus_far.flow.ip_flow.sport = 0xFFFF;
            argus->argus_far.flow.ip_flow.dport = 0xFFFF;
            break;

         case IPPROTO_ICMP:
            break;

         default:
            break;
      }

      ArgusThisSrv.argus = argus;
      RaProcessSrvRecord (&ArgusThisSrv);
   }
}


void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;

   struct ArgusRecordStore *store, *qstr;
   int RaTimeout = -1;

   if ((store = RaFindArgusRecord(argus)) != NULL) {
      RaThisArgusStore = store;

      if (RaCheckTimeout (store))
         RaPrintArgusPath (store);

      RaUpdateArgusStore(argus, store);

   } else {
      if ((argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)) {
         if ((store = RaNewArgusStore(argus, ARGUS_START)) != NULL) {
            if (store->queue && ((qstr = RaNewArgusStore(argus, ARGUS_STATUS)) != NULL)) {
               RaAddToQueue(store->queue, &qstr->qhdr);
               qstr->farhdrstatus = ArgusIndexRecord (qstr->argus, qstr->farhdrs);
               qstr->status |= RA_MODIFIED;
            }

            RaThisArgusStore = store;
            store->farhdrstatus = ArgusIndexRecord (store->argus, store->farhdrs);
            store->ArgusTimeout = RaTimeout;
            store->status |= RA_MODIFIED;
            store->status |= srv->status & RA_SVCTEST;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
}



void
process_tcp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_icmp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_udp (struct ArgusRecord *argus)
{

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   RaProcessRecord (argus);
}


void
process_ip (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_arp (struct ArgusRecord *argus)
{
}


void
process_non_ip (struct ArgusRecord *argus)
{
}



int RaThisHash = 0;
struct ArgusFlow *RaThisFlow;
int RaThisFlowRev = 0;

char *RaPrintTimeStats (struct ArgusRecord *, char *);

char *
RaPrintTimeStats (struct ArgusRecord *argus, char *str)
{
   char *retn = NULL;
   struct ArgusAGRStruct *agr;

   ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);

   if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
      retn = str;
      sprintf (retn, "%7d +/- %-6d  max %7d  min %7d   n = %d", agr->act.mean, agr->act.stdev,
                                                  agr->act.max, agr->act.min, agr->count);
   }

   return (retn);
}

int RaCompareArgusStore (const void *, const void *);
void RaPackQueue (struct RaQueueStruct *);
void RaSortQueue (struct RaQueueStruct *);

void
RaPrintArgusPath (struct ArgusRecordStore *store)
{
   char buf[MAXSTRLEN], *ptr = buf, *str = NULL;
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecord *argus = store->argus;
   unsigned int srchost, dsthost, intnode, ttl;
   char srcAddrString[64], dstAddrString[64], intAddrString[64];
   char statsbuf[256], *stats = statsbuf;
   char date[128];
   int len, pad, margin;

   str = get_ip_string (store->argus);
   printf ("%s\n", str);
   len = strlen(str);
   
   print_date (store->argus, date);

   pad = len - (strlen(date) + (3 * hfield) + (cflag ? ((4 * 10) + 3) : 0) + (Iflag ? 7 : 0) + (gflag ? 9 : 0));
 
   if (pad < hfield) {
      pad += 2 * hfield;
      hfield = pad / 3;
      pad = hfield;
   }

   if (store->queue) {
      RaSortQueue(store->queue);
      while ((obj = (struct ArgusRecordStore *) RaPopQueue(store->queue)) != NULL) {
         struct ArgusICMPObject *icmp =
                   (struct ArgusICMPObject *) obj->farhdrs[ARGUS_ICMP_DSR_INDEX];

         argus = obj->argus;

         if (argus && (obj->status & RA_MODIFIED)) {
            if (obj->act.n > 0) {
               obj->agr.act.mean = obj->act.sumtime/obj->act.n;
               obj->agr.act.stdev = sqrt (obj->act.sumsqrd/obj->act.n - pow (obj->act.sumtime/obj->act.n, 2.0));
            }
            if (obj->idle.n > 0) {
               obj->agr.idle.mean = obj->idle.sumtime/obj->idle.n;
               obj->agr.idle.stdev = sqrt (obj->idle.sumsqrd/obj->idle.n - pow (obj->idle.sumtime/obj->idle.n, 2.0));
            }
      
            obj->agr.type   = ARGUS_AGR_DSR;
            obj->agr.length = sizeof(obj->agr);
            bcopy ((char *) argus, buf, argus->ahdr.length);
            bcopy ((char *)&obj->agr,&buf[argus->ahdr.length], obj->agr.length);
      
            argus = (struct ArgusRecord *) ptr;
            argus->ahdr.length += obj->agr.length;
         }

         srchost = argus->argus_far.flow.ip_flow.ip_src;
         dsthost = argus->argus_far.flow.ip_flow.ip_dst;

         intnode = icmp->osrcaddr;
   
         ttl = argus->argus_far.attr_ip.sttl;
   
         sprintf(srcAddrString, "%s", ipaddr_string(&srchost));
         sprintf(dstAddrString, "%s", ipaddr_string(&dsthost));
         sprintf(intAddrString, "%s", ipaddr_string(&intnode));

         stats = RaPrintTimeStats (argus, statsbuf);

         if (idflag)
            printf ("                ");

         margin = pad + ((2 * hfield) - (strlen(srcAddrString) + strlen(dstAddrString)));

         if (dsthost != intnode) {
            if (nflag)
               printf ("Path  %15.15s  ->  %15.15s    Intermediate Node:  %15.15s  ",
                               srcAddrString, dstAddrString, intAddrString);
            else
               printf ("Path  %s  ->  %s    Intermediate Node:  %*.*s  ",
                               srcAddrString, dstAddrString,
                               margin, margin, intAddrString);
         } else {
            if (nflag)
               printf ("Path  %15.15s  ->  %15.15s     Terminating Node:  %15.15s  ",
                               srcAddrString, dstAddrString, intAddrString);
            else
               printf ("Path  %s  ->  %s     Terminating Node:  %*.*s  ",
                               srcAddrString, dstAddrString,
                               margin, margin, intAddrString);
         }
         printf ("  Distance: %2d  %s\n", ttl, stats);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaPrintArgusPath (0x%x) returning\n", store);
#endif
}


struct ArgusRecordStore *
RaFindArgusRecord(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   unsigned short hash = 0, *ptr;
   int i, len;

   RaThisFlow = &argus->argus_far.flow;
   ptr = (unsigned short *) RaThisFlow;

   for (i = 0, len = (sizeof(*RaThisFlow)) / sizeof(unsigned short); i < len; i++)
      hash += *ptr++;

   RaThisHash = hash;

   if ((hashEntry = RaFindHashObject ()) != NULL)
      retn = hashEntry->storeobj;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaFindArgusRecord: returning 0x%x\n", retn);
#endif

   return (retn);
}


int RaTotalRecords = 0;



struct ArgusRecordStore *
RaNewArgusStore(struct ArgusRecord *argus, int state)
{
   struct ArgusRecordStore *retn;
   struct ArgusRecord *argusbuf;
   int length = argus->ahdr.length;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordStore *) ArgusCalloc (1, sizeof (*retn))) != NULL) {
      RaAllocArgusRecordStore++;
      if ((argusbuf = (struct ArgusRecord *) ArgusCalloc (1, length)) != NULL) {
         bcopy((char *) argus, (char *) argusbuf, length);
         retn->argus = argusbuf;
         RaTotalRecords++;

         if (state == ARGUS_START) {
            if ((retn->rahtblhdr = RaAddHashEntry (retn)) != NULL) {
               retn->queue = RaNewQueue(64);
               RaAddToQueue(RaModelerQueue, &retn->qhdr);

            } else {
               ArgusFree (retn);
               retn = NULL;
               ArgusFree (argusbuf);
               return (retn);;
            }
         }
         retn->agr.laststartime = argus->argus_far.time.last;
         lasttime = argus->argus_far.time.last;

         retn->qhdr.lasttime = lasttime;
         retn->qhdr.logtime  = lasttime;
         retn->agr.lasttime  = lasttime;

         retn->agr.act.min  = 0x7FFFFFFF;
         retn->agr.idle.min = 0x7FFFFFFF;
         retn->agr.status  |= ARGUS_AGR_USECACTTIME;
         retn->agr.status  |= ARGUS_AGR_USECIDLETIME;
         retn->agr.count  = 1;

      } else {
         ArgusFree (retn);
         retn = NULL;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewArgusStore: added 0x%x\n", retn);
#endif

   return (retn);
}

void
RaRemoveArgusRecord(struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaRemoveArgusRecord: done\n");
#endif
}


void
RaUpdateArgusStore(struct ArgusRecord *argus, struct ArgusRecordStore *store)
{
   int i, index, start, end, found = 0;
   struct ArgusRecordStore **array, *str;
   struct ArgusICMPObject *argicmp = (struct ArgusICMPObject *) ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];

   store->status |= RA_MODIFIED;
   store->qhdr.lasttime = ArgusGlobalTime;

   if (store->queue) {
      array = (struct ArgusRecordStore **) store->queue->array;
      start = store->queue->head;
      end   = store->queue->tail;

      for (i = 0, index = start; i < store->queue->count; i++, index++) {
         if ((start + i) > store->queue->size)
           index = 0;

         if ((str = array[index]) != NULL) {
            struct ArgusICMPObject *stricmp = (struct ArgusICMPObject *) str->farhdrs[ARGUS_ICMP_DSR_INDEX];

            if ((argus->argus_far.attr_ip.sttl == str->argus->argus_far.attr_ip.sttl)) {
               if ((stricmp && argicmp) && (stricmp->osrcaddr == argicmp->osrcaddr)) {
                  RaMergeArgusRecord(argus, str);
                  found++;
                  break;
               }
            }
         }
      }

      if (!(found) && ((argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)  ||
                      ((argus->argus_far.status & ARGUS_ICMPUNREACH_MAPPED)  &&
                       (store->queue->count > 0)))) {
         if (store->queue && ((str = RaNewArgusStore(argus, ARGUS_STATUS)) != NULL)) {
            RaAddToQueue (store->queue, &str->qhdr);
            str->farhdrstatus = ArgusIndexRecord (str->argus, str->farhdrs);
            str->status |= RA_MODIFIED;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaUpdateArgusStore: \n");
#endif
}

void
RaTimeoutArgusStore (struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus;

   if (store && ((argus = store->argus) != NULL)) {
      if (store->status & RA_MODIFIED)
         RaPrintArgusPath (store);

      if (store->rahtblhdr)
         RaRemoveHashEntry (store->rahtblhdr);

      RaRemoveArgusRecord(argus);
      ArgusFree (argus);
      ArgusFree (store);
      RaAllocArgusRecordStore--;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaTimeoutArgusStore: 0x%x\n", store);
#endif
}


int
RaCheckTimeout (struct ArgusRecordStore *store)
{
   int retn = 0;
   struct ArgusRecord *argus = store->argus;
   unsigned int lapseTime;
   struct timeval *tvp = &ArgusGlobalTime;

   if (argus) {
      if (store->ArgusTimeout > 0) {
         lapseTime = store->qhdr.lasttime.tv_sec + store->ArgusTimeout;
         if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
             (tvp->tv_usec > store->qhdr.lasttime.tv_usec)))
            retn++;
   
         lapseTime = store->qhdr.logtime.tv_sec + store->ArgusTimeout;
         if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
             (tvp->tv_usec > store->qhdr.logtime.tv_usec)))
      
            RaPrintArgusPath (store);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}



#include <stdio.h>
#include <errno.h>



struct RaHashTableHeader *
RaFindHashObject ()
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *head = NULL, *target;

   if ((target = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
      head = target;
      do {
         if (!(bcmp ((char *) RaThisFlow, (char *) &target->flow, sizeof(*RaThisFlow)))) {
               retn = target;
               break;
            } else
               target = target->nxt;
      } while (target != head);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaFindHashObject: returning 0x%x\n", retn);
#endif

   return (retn);
}



struct RaHashTableHeader *
RaAddHashEntry (struct ArgusRecordStore *store)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *start = NULL;

   if ((retn = (struct RaHashTableHeader *) ArgusCalloc (1, sizeof (struct RaHashTableHeader))) != NULL) {
      RaAllocHashTableHeaders++;
      bcopy((char *) RaThisFlow, (char *)&retn->flow, sizeof (*RaThisFlow));
      retn->storeobj = store;
      retn->hash = RaThisHash;
      
      if ((start = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      RaHashTable.array[RaThisHash % RaHashTable.size] = retn;
      RaHashTable.count++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddHashEntry (0x%x) returning 0x%x\n", store, retn);
#endif

   return (retn);
}

void
RaRemoveHashEntry (struct RaHashTableHeader *rahtblhdr)
{
   unsigned short hash = rahtblhdr->hash;

   rahtblhdr->prv->nxt = rahtblhdr->nxt;
   rahtblhdr->nxt->prv = rahtblhdr->prv;

   if (rahtblhdr == RaHashTable.array[hash % RaHashTable.size]) {
      if (rahtblhdr == rahtblhdr->nxt)
         RaHashTable.array[hash % RaHashTable.size] = NULL;
      else
         RaHashTable.array[hash % RaHashTable.size] = rahtblhdr->nxt;

      RaHashTable.count--;
   }

   ArgusFree (rahtblhdr);
   RaAllocHashTableHeaders--;
 
#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaRemoveHashObject (0x%x) returning\n", rahtblhdr);
#endif
}


#define RA_MAXQSCAN  2048


int
RaCompareArgusStore (const void *void1, const void *void2)
{
   int retn = 0;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;

   if ((store1 && store1->argus) && ((store2 && store2->argus))) {
      struct ArgusRecord *argus1 = store1->argus;
      struct ArgusRecord *argus2 = store2->argus;

      retn = (argus1->argus_far.attr_ip.sttl - argus2->argus_far.attr_ip.sttl);
   }

   return (retn);
}


void
RaPackQueue (struct RaQueueStruct *queue)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *obj;

   if (!((queue->head == 0) && (queue->count == queue->tail))) {
      if ((newq = RaNewQueue (queue->count)) != NULL) {
         while (queue->count)
            if ((obj = RaPopQueue(queue)) != NULL)
               RaAddToQueue(newq, obj);

         ArgusFree(queue->array);

         queue->head  = newq->head;
         queue->tail  = newq->tail;
         queue->count = newq->count;
         queue->size  = newq->count;
         queue->array = newq->array;
      }
   } else
      queue->size = queue->count;
}

void
RaSortQueue (struct RaQueueStruct *queue)
{
   int i;

   RaPackQueue (queue);
   qsort ((char *) queue->array, queue->size, sizeof (void *), RaCompareArgusStore);

   for (i = 0; i < queue->count; i++)
      queue->array[i]->index = i;
}
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while (queue->count) {
            if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
               RaTimeoutArgusStore(obj);
         }
         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}



struct RaQueueStruct *
RaNewQueue (int size)
{
   struct RaQueueStruct *retn =  NULL;

   if ((retn = (struct RaQueueStruct *) ArgusCalloc (1, sizeof (struct RaQueueStruct))) != NULL) {
      retn->head = 0;
      retn->tail = 0;
      retn->count = 0;
      if ((retn->array = (struct ArgusQueueHeader **) ArgusCalloc (size, sizeof(void *))) != NULL)
         retn->size = size;
      else {
         ArgusFree(retn);   
         retn = NULL;   
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewQueue () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *obj = NULL;

   while ((obj = RaPopQueue(queue)))
      ArgusFree(obj);

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaDeleteQueue (0x%x) returning\n", queue);
#endif
}


int
RaGetQueueCount(struct RaQueueStruct *queue)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaGetQueueCount (0x%x) returning %d\n", queue, queue->count);
#endif

   return (queue->count);
}

void
RaPushQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int start = queue->head;

   if (obj && (queue->count < queue->size)) {
      if (--start < 0)
         start = queue->size;
   
      if (queue->array[start] == NULL) {
         queue->count++;
         obj->index = start;
         queue->array[start] = obj;
         queue->head = start;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPushQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}

#define RA_MAXQUEUESIZE         1048576

void
RaAddToQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *qhdr, **oarray;

   if (obj) {
      if (!((queue->count + 1) < (queue->size))) {
         if ((queue->size << 1) <= RA_MAXQUEUESIZE) {
            if ((newq = RaNewQueue (queue->size << 1)) != NULL) {
               while (queue->count)
                  if ((qhdr = RaPopQueue(queue)) != NULL)
                     RaAddToQueue(newq, qhdr);

               oarray = queue->array;

               queue->head  = newq->head;
               queue->tail  = newq->tail;
               queue->count = newq->count;
               queue->size  = newq->size;
               queue->array = newq->array;

               newq->array = oarray;
               newq->count = 0;
               RaDeleteQueue (newq);
            }
         }
   #ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusAddToQueue: expanded queue size to %d\n", queue->size);
   #endif
      }

      queue->count++;

      obj->index = queue->tail;
      queue->array[queue->tail] = obj;

      while (queue->array[queue->tail] != NULL) {
         queue->tail++;
         if (queue->tail == queue->size)
            queue->tail = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddToQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}



struct ArgusQueueHeader *
RaPopQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *retn = NULL;
   struct ArgusQueueHeader *obj = NULL;

   if (queue && queue->count) {
      if ((obj = queue->array[queue->head]) != NULL)
         retn = RaRemoveFromQueue(queue, obj);
      else
         ArgusLog (LOG_ERR, "RaPopQueue(0x%x) internal queue error count %d head NULL\n", queue, queue->count);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x) returning 0x%x\n", queue, retn);
#endif
   
   return(retn);
}


struct ArgusQueueHeader *
RaRemoveFromQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct ArgusQueueHeader *retn = NULL;

   if ((queue != NULL) && (obj != NULL)) {
      int index = obj->index;

      if (queue->count && (queue->array[index] == obj)) {
         queue->count--;
         queue->array[index] = NULL;
         obj->index = 0;
         retn = obj;

         if (queue->count) {
            if (queue->head == index) {
               while (queue->array[queue->head] == NULL) {
                  queue->head++;
                  if (queue->head == queue->size)
                     queue->head = 0;
               }
            }

         } else {
            queue->head = 0;
            queue->tail = 0;
         }

      } else
         ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) obj not in queue\n", queue, obj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x, 0x%x) returning 0x%x\n", queue, obj, obj);
#endif

   return (retn);
}


struct ArgusListStruct *
ArgusNewList ()
{
   struct ArgusListStruct *retn = NULL;
 
   if ((retn = (struct ArgusListStruct *) ArgusCalloc (1, sizeof (struct ArgusListStruct))) != NULL) {
      retn->start = NULL;
      retn->count = 0;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusNewList () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
ArgusDeleteList (struct ArgusListStruct *list)
{
   if (list) {
      while (list->start)
         ArgusPopFrontList(list);

      ArgusFree (list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusDeleteList (0x%x) returning\n", list);
#endif
}

int
ArgusListEmpty (struct ArgusListStruct *list)
{
   return (list->start == NULL);
}

int
ArgusGetListCount(struct ArgusListStruct *list)
{
   return (list->count);
}

void
ArgusPushFrontList(struct ArgusListStruct *list, void *obj)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if (list && obj) {
      if ((lobj = (struct ArgusListObjectStruct *) ArgusCalloc (1, sizeof(*lobj))) != NULL) {
         lobj->obj = obj;
   
         if (list->start) {
            lobj->nxt = list->start;
            lobj->prv = list->start->prv;
            lobj->nxt->prv = lobj;
            lobj->prv->nxt = lobj;
         } else {
            lobj->prv = lobj;
            lobj->nxt = lobj;
         }
   
         list->start = lobj;
         list->count++;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusPushFrontList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void
ArgusPushBackList(struct ArgusListStruct *list, void *obj)
{
   ArgusPushFrontList(list, obj);
   list->start = list->start->nxt;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusPushBackList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void *
ArgusFrontList(struct ArgusListStruct *list)
{
   void *retn = NULL;

   if (list->start)
      retn = list->start->obj;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusFrontList (0x%x) returning 0x%x\n", list, retn);
#endif

   return (retn);
}


void
ArgusPopBackList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      list->start = list->start->prv;
      ArgusPopFrontList(list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusPopBackList (0x%x) returning\n", list);
#endif
}

void
ArgusPopFrontList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      if (--list->count > 0) {
         if (lobj->prv)
            lobj->prv->nxt = lobj->nxt;
 
         if (lobj->nxt)
            lobj->nxt->prv = lobj->prv;
 
        list->start = lobj->nxt;
 
      } else
         list->start = NULL;
 
      ArgusFree(lobj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusPopFrontList (0x%x) returning\n", list);
#endif
}


int RaFlowMajorModified = 0;

struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
long long RaGetActiveDuration (struct ArgusRecord *);
long long RaGetuSecDuration (struct ArgusRecord *);

struct ArgusRecord *
RaCopyArgusRecord (struct ArgusRecord *argus)
{
   int length = 0;
   struct ArgusRecord *retn = NULL;

   if (argus && ((length = argus->ahdr.length) > 0))
      if ((retn = (struct ArgusRecord *) ArgusCalloc(1, length)) != NULL)
         bcopy ((char *) argus, (char *) retn, length);

   return (retn);
}


long long
RaGetActiveDuration (struct ArgusRecord *argus)
{
   return (RaGetuSecDuration (argus));
}

long long
RaGetuSecDuration (struct ArgusRecord *argus)
{
   long long retn = 0;
   int sec, usec;

   sec  = argus->argus_far.time.last.tv_sec  - argus->argus_far.time.start.tv_sec;
   usec = argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec;

   if (usec < 0) {
      sec--; usec += 1000000;
   }

   retn = (sec * 1000000) + usec;

   return (retn);
}


void
RaMergeArgusRecord(struct ArgusRecord *a1, struct ArgusRecordStore *store)
{
   u_char buf[MAXSTRLEN];
   struct ArgusRecord *a2 = NULL, *tmp = (struct ArgusRecord *)buf;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   struct ArgusFarStruct *far1 = NULL;
   struct ArgusFarStruct *far2 = NULL;
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;
   unsigned int i, status = 0, TimeMultiplier;
   long long duration;

   if ((a2 = store->argus) != NULL) {
      bzero (buf, sizeof(buf));
      bcopy ((char *) a2, buf, a2->ahdr.length);

      a1DSRStatus = ArgusIndexRecord (a1,  a1farhdr);
      a2DSRStatus = ArgusIndexRecord (tmp, a2farhdr);

      if (RaFlowMajorModified) {
         a1DSRStatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
         store->farhdrstatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
      }

      status = a1DSRStatus & a2DSRStatus;

      far1 = (struct ArgusFarStruct *) a1farhdr[ARGUS_FAR_DSR_INDEX];
      far2 = (struct ArgusFarStruct *) a2farhdr[ARGUS_FAR_DSR_INDEX];

/*
      if (!(RaThisFlowRev)) {
         if (((far2->src.count + far1->src.count) < far2->src.count) ||
             ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
 
      } else {
         if (((far2->src.count + far1->dst.count) < far2->src.count) ||
             ((far2->dst.count + far1->src.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
      }
*/

      for (i = 0; i < 32; i++) {
         if (status & 0x01) {
            switch (i) {
               case ARGUS_FAR_DSR_INDEX: {
                  u_int diffstartime = 0;
                  int diffusec = 0;

                  far1 = (struct ArgusFarStruct *) a1farhdr[i];
                  far2 = (struct ArgusFarStruct *) a2farhdr[i];

                  if (!(RaThisFlowRev)) {
/*
                     if (((far2->src.count + far1->src.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);
*/
                     far2->src.count    += far1->src.count;
                     far2->src.bytes    += far1->src.bytes;
                     far2->dst.count    += far1->dst.count;
                     far2->dst.bytes    += far1->dst.bytes;
                     far2->src.appbytes += far1->src.appbytes;
                     far2->dst.appbytes += far1->dst.appbytes;

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                     }

                  } else {
/*
                     if (((far2->src.count + far1->dst.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->src.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);
*/

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                     }

                     far2->src.count    += far1->dst.count;
                     far2->src.bytes    += far1->dst.bytes;
                     far2->dst.count    += far1->src.count;
                     far2->dst.bytes    += far1->src.bytes;
                     far2->src.appbytes += far1->dst.appbytes;
                     far2->dst.appbytes += far1->src.appbytes;
                  }


                  diffstartime  = (far1->time.start.tv_sec  - store->agr.laststartime.tv_sec);
                  if ((diffusec = (far1->time.start.tv_usec - store->agr.laststartime.tv_usec)) < 0) {
                     diffstartime--;
                     diffusec += 1000000;
                  }

                  store->agr.laststartime = far1->time.start;

                  if (diffstartime > 0) {
                     if (diffstartime > 2147) {
                        if (store->agr.status & ARGUS_AGR_USECIDLETIME) {
                           if (store->agr.idle.min != 0x7FFFFFFF)
                              store->agr.idle.min /= 1000;
                           store->agr.idle.max /= 1000;
                           store->idle.sumtime /= 1000;
                           store->idle.sumsqrd /= 1000000.0;
                           store->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                           store->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                        }
                     }

                     if (store->agr.status & ARGUS_AGR_USECIDLETIME)
                        TimeMultiplier = 1000000;
                     else
                        TimeMultiplier = 1000;

                     diffstartime *= TimeMultiplier;
                     diffstartime += (diffusec * 1000) / (TimeMultiplier / 1000);

                     if ((store->agr.idle.min > diffstartime) && (diffstartime > 0))
                        store->agr.idle.min = diffstartime;

                     if ((store->agr.idle.max < diffstartime) && (diffstartime > 0))
                        store->agr.idle.max = diffstartime;

                     store->idle.sumtime += diffstartime;
                     store->idle.sumsqrd += pow(diffstartime, 2.0);
                     store->idle.n++;
                  }

                  duration = RaGetuSecDuration(a1);
                  if (duration > 0x6FFFFFFF) {
                     if (store->agr.status & ARGUS_AGR_USECIDLETIME) {
                         if (store->agr.idle.min != 0x7FFFFFFF)
                            store->agr.act.min /= 1000;
                         store->agr.act.max /= 1000;
                         store->act.sumtime /= 1000;
                         store->act.sumsqrd /= 1000000.0;
                         store->agr.status &= ~ARGUS_AGR_USECACTTIME;
                         store->agr.status |=  ARGUS_AGR_MSECACTTIME;
                     }
                  }

                  if (store->agr.status & ARGUS_AGR_USECACTTIME)
                     TimeMultiplier = 1;
                  else
                     TimeMultiplier = 1000;

                  duration /= TimeMultiplier;

                  if ((store->agr.act.min > duration) && (duration > 0))
                     store->agr.act.min = duration;

                  if ((store->agr.act.max < duration) && (duration > 0))
                     store->agr.act.max = duration;

                  store->act.sumtime += duration;
                  store->act.sumsqrd += pow(duration, 2.0);
                  store->act.n++;

                  store->agr.count++;

                  if (far1->ArgusTransRefNum != far2->ArgusTransRefNum)
                     far2->ArgusTransRefNum = 0;

                  if ((far2->time.start.tv_sec  > far1->time.start.tv_sec) ||
                     ((far2->time.start.tv_sec == far1->time.start.tv_sec) &&
                     (far2->time.start.tv_usec  > far1->time.start.tv_usec)))

                     far2->time.start = far1->time.start;

                  if ((far2->time.last.tv_sec  < far1->time.last.tv_sec) ||
                     ((far2->time.last.tv_sec == far1->time.last.tv_sec) &&
                     (far2->time.last.tv_usec  < far1->time.last.tv_usec)))

                     far2->time.last = far1->time.last;

                  if (RaThisArgusStore->qhdr.lasttime.tv_sec != far2->time.last.tv_sec) {
                     RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     store->agr.lasttime = far2->time.last;
                  } else
                     if (RaThisArgusStore->qhdr.lasttime.tv_usec < far2->time.last.tv_usec) {
                        RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     }

                  break;
               }

               case ARGUS_TCP_DSR_INDEX: {
                  struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
                  struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

                  tcp2->state |= tcp1->state;

                  if (!(tcp2->synAckuSecs))
                     tcp2->synAckuSecs = tcp1->synAckuSecs;
                  else
                     if (tcp2->synAckuSecs > tcp1->synAckuSecs)
                        if (tcp1->synAckuSecs)
                           tcp2->synAckuSecs = tcp1->synAckuSecs;

                  if (!(tcp2->ackDatauSecs))
                     tcp2->ackDatauSecs = tcp1->ackDatauSecs;
                  else
                     if (tcp2->ackDatauSecs > tcp1->ackDatauSecs)
                        if (tcp1->ackDatauSecs)
                           tcp2->ackDatauSecs = tcp1->ackDatauSecs;

                  tcp2->src.seqbase   = 0;
                  tcp2->dst.seqbase   = 0;

                  tcp2->src.ackbytes += tcp1->src.ackbytes;
                  tcp2->dst.ackbytes += tcp1->dst.ackbytes;
                  tcp2->src.bytes    += tcp1->src.bytes;
                  tcp2->dst.bytes    += tcp1->dst.bytes;
                  tcp2->src.rpkts    += tcp1->src.rpkts;
                  tcp2->dst.rpkts    += tcp1->dst.rpkts;        

                  if (tcp2->src.win > tcp1->src.win);
                     tcp2->src.win = tcp1->src.win;
                  if (tcp2->dst.win > tcp1->dst.win);
                     tcp2->dst.win = tcp1->dst.win;        

                  tcp2->src.flags    |= tcp1->src.flags;
                  tcp2->dst.flags    |= tcp1->dst.flags;        

                  break;
               }

               case ARGUS_RTP_DSR_INDEX: {
                  struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
                  struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];
/*
                  if ((((rtp2->sdrop + rtp1->sdrop) < rtp2->sdrop) || ((rtp2->ddrop + rtp1->ddrop) < rtp2->ddrop)) ||
                      (((rtp2->ssdev + rtp1->ssdev) < rtp2->ssdev) || ((rtp2->dsdev + rtp1->dsdev) < rtp2->dsdev))) {
                     RaSendArgusRecord(&store);
                     rtp2->sdrop = rtp1->sdrop;
                     rtp2->ddrop = rtp1->ddrop;
                     rtp2->ssdev = rtp1->ssdev;
                     rtp2->dsdev = rtp1->dsdev;

                  } else {
*/
                     rtp2->sdrop += rtp1->sdrop;
                     rtp2->ddrop += rtp1->ddrop;
                     rtp2->ssdev += rtp1->ssdev;
                     rtp2->dsdev += rtp1->dsdev;
/*
                  }
*/
                  break;
               }

               case ARGUS_ICMP_DSR_INDEX:
               case ARGUS_IGMP_DSR_INDEX:
               case ARGUS_ARP_DSR_INDEX:
               case ARGUS_FRG_DSR_INDEX:
                  break;

               case ARGUS_VLAN_DSR_INDEX: {
                  struct ArgusVlanStruct *vlan1 = (struct ArgusVlanStruct *) a1farhdr[i];
                  struct ArgusVlanStruct *vlan2 = (struct ArgusVlanStruct *) a2farhdr[i];

                  if ((vlan2->status & ARGUS_SRC_VLAN) && (vlan1->status & ARGUS_SRC_VLAN))
                     if (vlan2->sid != vlan1->sid)
                        vlan2->status |= ARGUS_SRC_CHANGED;

                  if ((vlan2->status & ARGUS_DST_VLAN) && (vlan1->status & ARGUS_DST_VLAN))
                     if (vlan2->did != vlan1->did)
                        vlan2->status |= ARGUS_DST_CHANGED;

                  vlan2->status |= vlan1->status;
                  vlan2->sid = vlan1->sid;
                  vlan2->did = vlan1->did;
                  break;
               }

               case ARGUS_MPLS_DSR_INDEX: {
                  struct ArgusMplsStruct *mpls1 = (struct ArgusMplsStruct *) a1farhdr[i];
                  struct ArgusMplsStruct *mpls2 = (struct ArgusMplsStruct *) a2farhdr[i];

                  if ((mpls2->status & ARGUS_SRC_MPLS) && (mpls1->status & ARGUS_SRC_MPLS))
                     if (mpls2->slabel != mpls1->slabel)
                        mpls2->status |= ARGUS_SRC_CHANGED;

                  if ((mpls2->status & ARGUS_DST_MPLS) && (mpls1->status & ARGUS_DST_MPLS))
                     if (mpls2->dlabel != mpls1->dlabel)
                        mpls2->status |= ARGUS_DST_CHANGED;

                  mpls2->status |= mpls1->status;
                  mpls2->slabel = mpls1->slabel;
                  mpls2->dlabel = mpls1->dlabel;
                  break;
               }

               case ARGUS_ESP_DSR_INDEX: {
                  struct ArgusESPStruct *esp1 = (struct ArgusESPStruct *) a1farhdr[i];
                  struct ArgusESPStruct *esp2 = (struct ArgusESPStruct *) a2farhdr[i];

                  if (esp1 && esp2) {
                     if (esp1->src.spi != esp2->src.spi)
                        esp2->src.spi = 0;

                     if (esp1->dst.spi != esp2->dst.spi)
                        esp2->dst.spi = 0;

                     esp2->src.lastseq = esp1->src.lastseq;
                     esp2->dst.lastseq = esp1->dst.lastseq;
                     esp2->src.lostseq = esp1->src.lostseq;
                     esp2->dst.lostseq = esp1->dst.lostseq;
                  }
                  break;
               }
    
               case ARGUS_TIME_DSR_INDEX: {
                  struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
                  struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];
                  long long sumtime = 0, sumtime1 = 0, sumtime2 = 0;
                  double sumsqrd = 0.0, sumsqrd1 = 0.0, sumsqrd2 = 0.0;

                  if (time1 && time2) {
                     if ((time2->src.act.n += time1->src.act.n) > 0) {
                        sumtime1 = (time1->src.act.mean * time1->src.act.n);
                        sumtime2 = (time2->src.act.mean * time2->src.act.n);
//                      if ((sumtime = sumtime1 + sumtime2) < sumtime1)
//                         goto doneRaMergeArgus;
  
                        time2->src.act.mean  = sumtime/time2->src.act.n;
                        if (time1->src.act.n)
                           sumsqrd1 = (time1->src.act.n * pow(time1->src.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.act.n * pow(time2->src.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.act.n;
//                      if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
//                         goto doneRaMergeArgus;
  
                        time2->src.act.stdev = sqrt (sumsqrd/time2->src.act.n - pow (sumtime/time2->src.act.n, 2.0));
                        if (time2->src.act.max < time1->src.act.max)
                           time2->src.act.max = time1->src.act.max; 
                        if (time2->src.act.min > time1->src.act.min)
                           time2->src.act.min = time1->src.act.min; 
                     }

                     if ((time2->dst.act.n += time1->dst.act.n) > 0) {
                        sumtime1 = (time1->dst.act.mean * time1->dst.act.n);
                        sumtime2 = (time2->dst.act.mean * time2->dst.act.n);
//                      if ((sumtime = sumtime1 + sumtime2) < sumtime1)
//                         goto doneRaMergeArgus;
  
                        time2->dst.act.mean  = sumtime/time2->dst.act.n;
                        if (time1->dst.act.n)
                           sumsqrd1 = (time1->dst.act.n * pow(time1->dst.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.act.n * pow(time2->dst.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.act.n;
//                      if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
//                         goto doneRaMergeArgus;
  
                        time2->dst.act.stdev = sqrt (sumsqrd/time2->dst.act.n - pow (sumtime/time2->dst.act.n, 2.0));
                        if (time2->dst.act.max < time1->dst.act.max)
                           time2->dst.act.max = time1->dst.act.max; 
                        if (time2->dst.act.min > time1->dst.act.min)
                           time2->dst.act.min = time1->dst.act.min; 
                     }

                     if ((time2->src.idle.n += time1->src.idle.n) > 0) {
                        sumtime1 = (time1->src.idle.mean * time1->src.idle.n);
                        sumtime2 = (time2->src.idle.mean * time2->src.idle.n);
//                      if ((sumtime = sumtime1 + sumtime2) < sumtime1)
//                         goto doneRaMergeArgus;
  
                        time2->src.idle.mean  = sumtime/time2->src.idle.n;
                        if (time1->src.idle.n)
                           sumsqrd1 = (time1->src.idle.n * pow(time1->src.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.idle.n * pow(time2->src.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.idle.n;
//                      if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
//                         goto doneRaMergeArgus;
  
                        time2->src.idle.stdev = sqrt (sumsqrd/time2->src.idle.n - pow (sumtime/time2->src.idle.n, 2.0));
                        if (time2->src.idle.max < time1->src.idle.max)
                           time2->src.idle.max = time1->src.idle.max; 
                        if (time2->src.idle.min > time1->src.idle.min)
                           time2->src.idle.min = time1->src.idle.min; 
                     }

                     if ((time2->dst.idle.n += time1->dst.idle.n) > 0) {
                        sumtime1 = (time1->dst.idle.mean * time1->dst.idle.n);
                        sumtime2 = (time2->dst.idle.mean * time2->dst.idle.n);
//                      if ((sumtime = sumtime1 + sumtime2) < sumtime1)
//                         goto doneRaMergeArgus;
  
                        time2->dst.idle.mean  = sumtime/time2->dst.idle.n;
                        if (time1->dst.idle.n)
                           sumsqrd1 = (time1->dst.idle.n * pow(time1->dst.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.idle.n * pow(time2->dst.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.idle.n;
//                      if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
//                         goto doneRaMergeArgus;
  
                        time2->dst.idle.stdev = sqrt (sumsqrd/time2->dst.idle.n - pow (sumtime/time2->dst.idle.n, 2.0));
                        if (time2->dst.idle.max < time1->dst.idle.max)
                           time2->dst.idle.max = time1->dst.idle.max; 
                        if (time2->dst.idle.min > time1->dst.idle.min)
                           time2->dst.idle.min = time1->dst.idle.min; 
                     }
                  }

                  break;
               }

               case ARGUS_SRCUSRDATA_DSR_INDEX:
               case ARGUS_DSTUSRDATA_DSR_INDEX:
                  break;

               case ARGUS_AGR_DSR_INDEX: {
                  struct ArgusAGRStruct *agr1 = (struct ArgusAGRStruct *) a1farhdr[i];
                  unsigned int sumtime;
                  double sumsqrd;

                  if ((store->agr.lasttime.tv_sec < agr1->lasttime.tv_sec) ||
                      ((store->agr.lasttime.tv_sec == agr1->lasttime.tv_sec) &&
                       (store->agr.lasttime.tv_usec < agr1->lasttime.tv_usec))) {

                     RaThisArgusStore->qhdr.lasttime = agr1->lasttime;
                     store->agr.lasttime = agr1->lasttime;
                  }

                  if (store->agr.act.min > agr1->act.min) store->agr.act.min = agr1->act.min;
                  if (store->agr.act.max < agr1->act.max) store->agr.act.max = agr1->act.max;

                  sumtime = agr1->act.mean * agr1->act.n;
                  sumsqrd = (agr1->act.n * pow(agr1->act.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->act.n;

                  store->act.sumtime += agr1->act.mean * agr1->act.n;
                  store->act.sumsqrd += sumsqrd;
                  store->act.n += agr1->act.n;

                  sumtime = agr1->idle.mean * agr1->idle.n;
                  sumsqrd = (agr1->idle.n * pow(agr1->idle.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->idle.n;

                  store->idle.sumtime += agr1->idle.mean * agr1->idle.n;
                  store->idle.sumsqrd += sumsqrd;
                  store->idle.n += agr1->idle.n;

                  store->agr.count += agr1->count;
                  break;
               }
            }
         }
         status >>= 1;
      }

      store->status |= RA_MODIFIED;
      store->argus->ahdr.status |= ARGUS_MERGED;
      bcopy (buf, (char *) a2, ((struct ArgusRecord *)buf)->ahdr.length);
#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: merged\n");
#endif
      return;

   } else {
      store->argus = RaCopyArgusRecord(a1);
#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: swapped");
#endif
      return;
   }
/*
doneRaMergeArgus:
   RaSendArgusRecord(&store);
   ArgusFree(store->argus);
   store->argus = RaCopyArgusRecord(a1);

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaMergeArgusRecord: dumped and swapped");
#endif
*/
}
