
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * ragator - argus record aggregator  with flow model manipulation.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <ragator.h>
#include <math.h>

void RaProcessRecord (struct ArgusRecord *argus) { }
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
int RaInitialized = 0;
int RaTimeout     = 0x1FFFFFFF;
int RaIdleTimeout = 0;

/*
   Format is "bins[L]:range"
   where range is  value-value
   value is %d[ums]
*/

int RaHistoTimeParse (char *);
int *RaHistoTimeValues;

int
RaHistoTimeParse (char *str) 
{
   int RaHistoStart = 0;
   int RaHistoEnd = 0;
   int RaHistoBins = 0;
   int RaHistoTimeLog = 0;
   double RaHistoBinSize;
   int retn = 0, i;
   char *ptr, *vptr, tmpbuf[64], *tmp = tmpbuf;

   bzero (tmpbuf, 64);
   strcpy (tmpbuf, str);

   if ((ptr = strchr (tmp, ':')) != NULL) {
      *ptr++ = '\0';
      vptr = ++ptr;
      if (strchr (tmp, 'L'))
         RaHistoTimeLog++;

      if (isdigit(*tmp))
         if ((RaHistoBins = atoi(tmp)) < 0)
            return (retn);

      if ((ptr = strchr (vptr, '-')) != NULL) {
         *ptr++ = '\0';
         if ((RaHistoStart = atoi(vptr)) < 0)
            return (retn);
         if ((RaHistoEnd = atoi(ptr)) < 0)
            return (retn);

         retn++;
         RaHistoTimeSeries = RaHistoBins;
         RaHistoBinSize = ((RaHistoEnd - RaHistoStart) * 1.0) / RaHistoBins * 1.0;
         if ((RaHistoTimeValues = (int *) ArgusCalloc (RaHistoBins + 1, sizeof(int))) != NULL) {
            for (i = 0; i < (RaHistoBins - 1); i++) {
               if (RaHistoTimeLog)
                  RaHistoTimeValues[i] = RaHistoBinSize * (i + 1);
               else
                  RaHistoTimeValues[i] = RaHistoBinSize * (i + 1);
            }

            RaHistoTimeValues[RaHistoBins] = 0x7FFFFFFF;

         } else
            ArgusLog (LOG_ERR, "RaHistoTimeParse: ArgusCalloc %s\n", strerror(errno));
      }
   }

   return (retn);
}

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;
   
      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));
   
      if (ArgusFlowModelFile)
         if ((RaFlowModelFile = RaReadFlowModelFile(NULL)) < 0)
            exit(0);
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (Hflag) {
         RaPrintTransactions++;
         if (!(RaHistoTimeParse (Hstr)))
            usage();
      }

      if (Bflag) {
         if ((RaHoldingBuffer = (struct ArgusHoldingBuffer *) ArgusCalloc (1, sizeof(*RaHoldingBuffer))) != NULL) {
            if ((RaHoldingBuffer->array = (struct ArgusListStruct **) ArgusCalloc (Bflag, sizeof(void *))) != NULL) {
               RaHoldingBuffer->size = Bflag;
            } else {
               ArgusFree (RaHoldingBuffer);
               RaHoldingBuffer = NULL;
               exit (-1);
            }
         } else
            exit (-1);
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;
      if (RaModelerQueue)
         RaProcessQueue (RaModelerQueue, ARGUS_STOP);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragator Version %s\n", version);
   fprintf (stderr, "usage:  %s [-f flowfile]\n", ArgusProgramName);
   fprintf (stderr, "usage:  %s [-f flowfile] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -f <flowfile>  read flow model from <flowfile>.\n");
   fprintf (stderr, "ra-options: -a             print record summaries on termination.\n");
   fprintf (stderr, "            -A             print application bytes.\n");
   fprintf (stderr, "            -b             dump packet-matching code.\n");
   fprintf (stderr, "            -c             print packet and byte counts.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -g             print record time duration.\n");
   fprintf (stderr, "            -G             print both start and last time values.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -I             print transaction state and option indicators.\n");
   fprintf (stderr, "            -l             print last time values [default is start time].\n");
   fprintf (stderr, "            -m             print MAC addresses.\n");
   fprintf (stderr, "            -n             don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R             print out response data when availabile.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                            mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z             print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   int i;

   if (xflag && (srv->status == RA_SVCFAILED))
      return;

   RaThisActiveDuration = RaGetActiveDuration(argus);

   if (Hflag) {
      for (i = 0; i < RaHistoTimeSeries; i++)
         if (RaThisActiveDuration < RaHistoTimeValues[i]) {
            RaThisActiveIndex = i;
            break;
         }
   } else
      RaThisActiveIndex = 0;

   bcopy(flow, RaArgusFlow, sizeof(struct ArgusFlow));

   if ((rap = RaFlowModelOverRides(argus)) != NULL) {
      RaModifyFlow(rap, argus);
      RaTimeout = rap->ArgusTimeout;
      RaIdleTimeout = rap->ArgusIdleTimeout;
   } else {
      RaModifyDefaultFlow(argus);
   }

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL)
      store = RaFindRevArgusRecord(&RaHashTable, argus);

   if (store) {
      RaThisArgusStore = store;

      RaCheckTimeout (store, argus);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }

      if (!(store->data[RaThisActiveIndex])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, store->data[RaThisActiveIndex]);

      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL)
            RaAddToQueue(RaModelerQueue, &store->qhdr);

         store->ArgusTimeout     = RaTimeout;
         store->ArgusIdleTimeout = RaIdleTimeout;

         RaThisArgusStore = store;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

   RaIdleTimeout = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaProcessSrvRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   RaIdleTimeout = 7200;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
   RaIdleTimeout = 0;
}


void
process_icmp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_udp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessSrvRecord (&ArgusThisSrv);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  256
#define RA_MAXQSIZE  250
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);

         break;

      default:
         while (queue->count > RA_MAXQSIZE) {
            obj = (struct ArgusRecordStore *) RaRemoveFromQueue(RaModelerQueue, RaModelerQueue->start->prv);
            RaTimeoutArgusStore(obj);
         }

         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusAGRStruct *agr = NULL;
   int i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.mean = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.mean = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               if ((Hflag || ArgusFlowModelFile) && (data->agr.count > 1)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else {
                  argus->ahdr.status &= ~(ARGUS_MERGED);
               }
            }
      
            if (wfile) {

#ifdef _LITTLE_ENDIAN
               ArgusHtoN(argus);
#endif

               ArgusWriteNewLogfile (wfile, argus);
      
#ifdef _LITTLE_ENDIAN
               ArgusNtoH(argus);
#endif

            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {
                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:              
                              printf ("%s", get_tcp_string (argus));
                              break;
            
                           case IPPROTO_ICMP:              
                              printf ("%s", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s", get_nonip_string (argus));
                        break;
                  }

                  if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
                     printf ("%s\n", RaGetUserDataString(argus));
                  else
                     printf ("\n");
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.mean = 0;
                        time->src.act.stdev = 0;
                        time->src.act.max = 0;
                        time->src.act.min = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.mean = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.max = 0;
                        time->src.idle.min = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.mean = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.max = 0;
                        time->dst.act.min = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.mean = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.max = 0;
                        time->dst.idle.min = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.min = 0x7FFFFFFF;   agr->act.mean = 0;
                        agr->act.stdev = 0;  agr->act.max = 0;
                        agr->idle.n = 0;
                        agr->idle.min = 0x7FFFFFFF; agr->idle.mean = 0;
                        agr->idle.stdev = 0; agr->idle.max = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.mean = 0;
            data->agr.act.stdev = 0;
            data->agr.act.max = 0;
            data->agr.act.min = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.mean = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.max = 0;
            data->agr.idle.min = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

