/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: session_time.cc,v 1.21 2005/02/03 13:41:29 gvdo Exp $
*/

#include <iostream>
#include <cmath>
#include <unistd.h>

#include <ardour/timestamps.h>

#include <pbd/error.h>

#include <ardour/ardour.h>
#include <ardour/configuration.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/tempo.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

/* BBT TIME*/

void
Session::bbt_time (jack_nframes_t when, BBT_Time& bbt)
{
	_tempo_map->bbt_time (when, bbt);
}

/* SMPTE TIME */

int
Session::set_smpte_type (float fps, bool drop_frames)
{
	smpte_frames_per_second = fps;
	smpte_drop_frames = drop_frames;
	if ((int) smpte_frames_per_second == smpte_frames_per_second) {
		_frames_per_smpte_frame =  _current_frame_rate / (int) smpte_frames_per_second;
	} else {
		_frames_per_smpte_frame = (jack_nframes_t) ceil ((double) _current_frame_rate / (double) smpte_frames_per_second);
	}
	last_smpte_valid = false;
	// smpte type bits are the middle two in the upper nibble
	switch ((int) ceil (fps)) {
	case 24:
		mtc_smpte_bits = 0;
		break;

	case 25:
		mtc_smpte_bits = 0x20;
		break;

	case 30:
	default:
		if (drop_frames) {
			mtc_smpte_bits = 0x40;
		} else {
			mtc_smpte_bits =  0x60;
		}
		break;
	};

	SMPTETypeChanged (); /* EMIT SIGNAL */

	set_dirty();

	return 0;
}

void
Session::set_smpte_offset (jack_nframes_t off)
{
	_smpte_offset = off;
	last_smpte_valid = false;
	 SMPTEOffsetChanged (); /* EMIT SIGNAL */
}

void
Session::smpte_time (jack_nframes_t when, SMPTE_Time& smpte)
{
	if (last_smpte_valid && when == last_smpte_when) {
		smpte = last_smpte;
		return;
	}

	jack_nframes_t offset_sample;
	jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
	long fps_int;
                                                                                                                   
        fps_int = (int)rint(smpte_frames_per_second);
                                                                                                                   
        if (when < _smpte_offset) {
                offset_sample = (_smpte_offset - when);
                smpte.negative = true;
        } else {
                offset_sample =  when - _smpte_offset;
                smpte.negative = false;
        }

	left = offset_sample;
        hrs = left / (frame_rate() * 60 * 60);
        left -= hrs * frame_rate() * 60 * 60;
        mins = left / (frame_rate() * 60);
        left -= mins * frame_rate() * 60;
        secs = left / frame_rate();
        left -= secs * frame_rate();
	/* if we're using an integral smpte frame rate */
	if (fps_int == smpte_frames_per_second) {
		frames = (left / (frame_rate() / fps_int));
	} else {

	/* we're now using the NTSC frame rate.
	From AES11-2003 section 4.5 video referencing (on page 6);
		48 kHz   NTSC 5 frames = 8008 samples,
		44.1 kHz NTSC 100 frames = 147147 samples
	..so NTSC has a frame rate of 29.97002997 exactly.
	anyway, because of this, some NTSC frames will have a different number of samples in them to
	others. this is a real bummer as far as the SMPTE ruler is concerned.	*/

		if (frame_rate() % 48000 == 0) {
			frames = (left * 5) / 8008;
		} else if (frame_rate() % 44100 == 0) {
			frames = (left * 100) / 147147;
		} else {
			frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
		}
	} 
	if (smpte_drop_frames) {

		long extra_frames = 0;
		long i;
		long total_minutes = 0;

		total_minutes = mins + (hrs * 60);
		extra_frames = (2 * total_minutes) - (2 * (total_minutes / 10));
		for (i = 1; i <= extra_frames; i++ ) {
			frames += 1;
			if (frames > (fps_int - 1)) {
                        	secs += frames / fps_int;
                        	frames -= (fps_int * (frames / fps_int));
                	}
	                if (secs > 59) {
	                        mins += 1;
	                        secs -= 60;
				if (!(mins % 10) == 0) {
					extra_frames += 2; // dropped frames from our extra frames
				}
	                }
	                if (mins > 59) {
	                        hrs += 1;
	                        mins -= 60;
	                }
		}		
	}
		
			
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;

	last_smpte_when = when;
	last_smpte = smpte;
	last_smpte_valid = true;
}

void
Session::smpte_time_subframes (jack_nframes_t when, SMPTE_Time& smpte)
{
	if (last_smpte_valid && when == last_smpte_when) {
                smpte = last_smpte;
                return;
        }
                                                                                                                   
        jack_nframes_t offset_sample;
        jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
	long subframes;
	long fps_int;
                                                                                                                   
        fps_int = (int)rint(smpte_frames_per_second);
                                                                                                                   
        if (when < _smpte_offset) {
                offset_sample = (_smpte_offset - when);
                smpte.negative = true;
        } else {
                offset_sample =  when - _smpte_offset;
                smpte.negative = false;
        }
                                                                                                                   
        left = offset_sample;
        hrs = left / (frame_rate() * 60 * 60);
        left -= hrs * frame_rate() * 60 * 60;
        mins = left / (frame_rate() * 60);
        left -= mins * frame_rate() * 60;
        secs = left / frame_rate();
        left -= secs * frame_rate();
         /* if we're using an integral smpte frame rate */
        if (fps_int == smpte_frames_per_second) {
                frames = (left / (frame_rate() / fps_int));
		left -= frames * (frame_rate() / fps_int); // half sample error @ 44100 24 fps
		subframes = left / (frame_rate() / (fps_int * 80));
        } else {

                /* NTSC */                                                                                                   
                if (frame_rate() % 48000 == 0) {
                        frames = (left * 5) / 8008;
			left -= (frames * 8008) / 5;
                } else if (frame_rate() % 44100 == 0) {
                        frames = (left * 100) / 147147;
			left -= (frames * 147147) / 100;
                } else {
                        frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
			left -= (jack_nframes_t) floor (frames * frame_rate() / smpte_frames_per_second);
                }
		 /* eighty bits in a SMPTE frame */
        	subframes = (int) floor (left / (frame_rate() / (smpte_frames_per_second * 80)));
        }
	if (smpte_drop_frames) {
                                                                                                                   
                long extra_frames = 0;
                long i;
                long total_minutes = 0;
                                                                                                                   
                total_minutes = mins + (hrs * 60);
                extra_frames = (2 * total_minutes) - (2 * (total_minutes / 10));
                for (i = 1; i <= extra_frames; i++ ) {
                        frames += 1;
                        if (frames > (fps_int - 1)) {
                                secs += frames / fps_int;
                                frames -= (fps_int * (frames / fps_int));
                        }
                        if (secs > 59) {
                                mins += 1;
                                secs -= 60;
                                if (!(mins % 10) == 0) {
                                        extra_frames += 2; //add drop frames to our extra frames
                                }
                        }
                        if (mins > 59) {
                                hrs += 1;
                                mins -= 60;
                        }
                }
        }
	
        smpte.subframes = subframes;
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;
                                                                                                                   
        last_smpte_when = when;
        last_smpte = smpte;
        last_smpte_valid = true;

}

void
Session::smpte_duration (jack_nframes_t when, SMPTE_Time& smpte) const
{
        jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
        long subframes;
	long fps_int;
                                                                                                                   
        fps_int = (int)rint(smpte_frames_per_second);

	left = when;
        hrs = left / (frame_rate() * 60 * 60);
        left -= hrs * frame_rate() * 60 * 60;
        mins = left / (frame_rate() * 60);
        left -= mins * frame_rate() * 60;
        secs = left / frame_rate();
        left -= secs * frame_rate();
	 /* if we're using an integral smpte frame rate */
        if (fps_int == smpte_frames_per_second) {
                frames = (left / (frame_rate() / fps_int));
		left -= frames * (frame_rate() / fps_int);
		subframes = left / (frame_rate() / (fps_int * 80));
        } else {
                                                                                                                   
        /* Never Twice the Same Colour */
                                                                                                                   
                if (frame_rate() % 48000 == 0) {
                        frames = (left * 5) / 8008;
                        left -= (frames * 8008) / 5;
                } else if (frame_rate() % 44100 == 0) {
                        frames = (left * 100) / 147147;
                        left -= (frames * 147147) / 100;
                } else {
                        frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
                        left -= (jack_nframes_t) floor (frames * frame_rate() / smpte_frames_per_second);
                }
                 /* eighty bits in a SMPTE frame */
                subframes = (int) floor (left / (frame_rate() / (smpte_frames_per_second * 80)));
        }
	if (smpte_drop_frames) {
                                                                                                                   
                long extra_frames = 0;
                long i;
                long total_minutes = 0;
                                                                                                                   
                total_minutes = mins + (hrs * 60);
                extra_frames = (2 * total_minutes) - (2 * (total_minutes / 10));
                for (i = 1; i <= extra_frames; i++ ) {
                        frames += 1;
                        if (frames > (fps_int - 1)) {
                                secs += frames / fps_int;
                                frames -= (fps_int * (frames / fps_int));
                        }
                        if (secs > 59) {
                                mins += 1;
                                secs -= 60;
                                if (!(mins % 10) == 0) {
                                        extra_frames += 2; //add drop frames to our extra frames
                                }
                        }
                        if (mins > 59) {
                                hrs += 1;
                                mins -= 60;
                        }
                }
        }
                                                                                                                   
        smpte.subframes = subframes;
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;

/*  i've left the original here.. i can't really test the new one.
it was only accurate to frames. maybe it was intentional ..nm

	double sample;
	double secs;
	double computed_secs;
	double approx_secs;
	long hrs;
	long mins;
	
	sample = when;

	approx_secs = (sample / frame_rate());
	secs = approx_secs;
	hrs =  (long) floor (secs / 3600.0);
	secs -= (hrs * 3600.0);
	mins = (long) floor (secs / 60.0);
	secs -= (mins * 60.0);

	computed_secs = (hrs * 3600.0) + (mins * 60.0) + floor (secs);

	smpte.frames = (long) ((approx_secs - computed_secs) * smpte_frames_per_second);
	smpte.seconds = (long) secs;
	smpte.minutes = mins;
	smpte.hours = hrs;

*/
}

void
Session::smpte_duration_string (char* buf, jack_nframes_t when) const
{
	SMPTE_Time smpte;

	smpte_duration (when, smpte);
	snprintf (buf, sizeof (buf), "%02ld:%02ld:%02ld:%02ld", smpte.hours, smpte.minutes, smpte.seconds, smpte.frames);
}

void
Session::smpte_time (SMPTE_Time &t)

{
	smpte_time (_transport_frame, t);
}

int
Session::jack_sync_callback (jack_transport_state_t state,
			     jack_position_t* pos)
{
	bool slave = synced_to_jack();

	switch (state) {
	case JackTransportStopped:
		if (slave && _transport_frame != pos->frame && post_transport_work == 0) {
		 	request_locate (pos->frame, false);
			// cerr << "SYNC: stopped, locate to " << pos->frame << " from " << _transport_frame << endl;
			return false;
		} else {
			return true;
		}
		
	case JackTransportStarting:
		// cerr << "SYNC: starting @ " << pos->frame << " a@ " << _transport_frame << " our work = " <<  post_transport_work << " pos matches ? " << (_transport_frame == pos->frame) << endl;
		if (slave) {
			return _transport_frame == pos->frame && post_transport_work == 0;
		} else {
			return true;
		}
		break;

	case JackTransportRolling:
		// cerr << "SYNC: rolling slave = " << slave << endl;
		if (slave) {
			start_transport ();
		}
		break;

	default:
		error << compose (_("Unknown JACK transport state %1 in sync callback"), state)
		      << endmsg;
	} 

	return true;
}

void
Session::jack_timebase_callback (jack_transport_state_t state,
				 jack_nframes_t nframes,
				 jack_position_t* pos,
				 int new_position)
{
	BBT_Time bbt;

	/* frame info */

	pos->frame = _transport_frame;
	pos->valid = JackPositionTimecode;

	/* BBT info */
	
	if (_tempo_map) {

		TempoMap::Metric metric (_tempo_map->metric_at (_transport_frame));
		_tempo_map->bbt_time_with_metric (_transport_frame, bbt, metric);
		
		pos->bar = bbt.bars;
		pos->beat = bbt.beats;
		pos->tick = bbt.ticks;

		// XXX still need to set bar_start_tick

		pos->beats_per_bar = metric.meter().beats_per_bar();
		pos->beat_type = metric.meter().note_divisor();
		pos->ticks_per_beat = Meter::ticks_per_beat;
		pos->beats_per_minute = metric.tempo().beats_per_minute();

		pos->valid = jack_position_bits_t (pos->valid | JackPositionBBT);
	}

#if 0
	/* SMPTE info */

	t.smpte_offset = _smpte_offset;
	t.smpte_frame_rate = smpte_frames_per_second;

	if (_transport_speed) {

		if (auto_loop) {

			Location* location = _locations.auto_loop_location();

			if (location) {

				t.transport_state = JackTransportLooping;
				t.loop_start = location->start();
				t.loop_end = location->end();
				t.valid = jack_transport_bits_t (t.valid | JackTransportLoop);

			} else {

				t.loop_start = 0;
				t.loop_end = 0;
				t.transport_state = JackTransportRolling;

			}

		} else {

			t.loop_start = 0;
			t.loop_end = 0;
			t.transport_state = JackTransportRolling;

		}

	} 

#endif		
}

jack_nframes_t
Session::convert_to_frames_at (jack_nframes_t position, AnyTime& any)
{
	double secs;
	
	switch (any.type) {
	case AnyTime::BBT:
		return _tempo_map->frame_time ( any.bbt);
		break;

	case AnyTime::SMPTE:
		/* XXX need to handle negative values */
		secs = any.smpte.hours * 60 * 60;
		secs += any.smpte.minutes * 60;
		secs += any.smpte.seconds;
		secs += any.smpte.frames / smpte_frames_per_second;
		return _smpte_offset + (jack_nframes_t) floor (secs * frame_rate());
		break;

	case AnyTime::Seconds:
		return (jack_nframes_t) floor (any.seconds * frame_rate());
		break;

	case AnyTime::Frames:
		return any.frames;
		break;
	}

	return any.frames;
}
