#!/bin/sh
# ftp-push.sh
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec with-ftp --version
                    ;;


      --help|-h)
		printf "update a remote copy of a local directory\\n"
		printf "usage: ftp-push [options] src dest\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf "Update a recursive copy of SRC at DEST, using \"with-ftp\"\\n"
		printf "to access DEST.\\n"
		printf "\\n"
		printf "DEST may be a local directory or an \"ftp:\" url.\\n"
		printf "\\n"
		printf "Files in DEST that are not in SRC are removed.  Files missing\\n"
		printf "are sent.  Files whose md5 checksum has changed are replaced.\\n"
		printf "\\n"
		printf "\"ftp-push\" creates directories at DEST \".md5-index\" containing\\n"
		printf "md5 the files stored there.  Consequently, local files and directories\\n"
		printf "having the name \".md5-index\" are ignored.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 
while test $# -ne 0 ; do

  case "$1" in 

    --)			shift
			break
			;;

    -*)			printf "ftp-push: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
  printf "usage: ftp-push [options] src dest\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

src="$1"
shift

dest="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$src"
src="`pwd`"

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHFTP_SITE" != "$dest" ; then
  exec with-ftp "$dest" ftp-push "$src" "$dest"
fi


################################################################
# Some Handy Functions
# 

remote_rmrf()
{
  if wftp-cd "$1" 2> /dev/null ; then
    for remote_dead in `wftp-ls | grep -v -E -e "^(\\.|\\.\\.)\$` ; do
      remote_rmrf "$remote_dead"
    done
    wftp-cd ".."
  else
    wftp-delete "$1" 2> /dev/null || true
  fi
}

update_this_directory()
{
  here="`pwd`"
  there="`wftp-pwd`"

  # remove extra files from there
  # 

  for remote_file in `wftp-ls | grep -v -E -e "^(\\.|\\.\\.)\$"` ; do
    if test "$remote_file" = ".md5-index" ; then
      if wftp-cd .md5-index 2> /dev/null ; then
        wftp-cd ..
      else
        remote_rmrf .md5-index
      fi
    elif test ! -e "$remote_file" ; then
      wftp-delete ".md5-index/$remote_file" 2> /dev/null || true
      remote_rmrf "$remote_file"
    fi
  done


  # make sure there is a remote .md5-index directory
  # 
  if wftp-cd .md5-index 2> /dev/null ; then
    wftp-cd ..
  else
    wftp-mkdir .md5-index
  fi

  # copy local files
  # 
  for local_file in `ls -a | grep -v -E -e "^(\\.md5-index|\\.|\\.\\.)\$"` ; do
    if test -d "$local_file" ; then
      cd "$local_file"
      if ! wftp-cd "$local_file" 2> /dev/null ; then
        remote_rmrf "$local_file"
	wftp-mkdir "$local_file"
	wftp-cd "$local_file"
      fi
      ( update_this_directory )
      cd "$here"
      wftp-cd "$there"
    else
      set +e
      remote_md5="`wftp-get \".md5-index/$local_file\" 2> /dev/null`"
      set -e
      local_md5="`md5 \"$local_file\"`"
      if test "$remote_md5" = "$local_md5" ; then
        printf "U %s\\n" "`pwd`/$local_file"
      else
        printf "M %s" "`pwd`/$local_file"
	wftp-delete ".md5-index/$local_file" 2> /dev/null || true
	cat "$local_file" | wftp-put "$local_file"
	printf "%s\\n" "$local_md5" | wftp-put ".md5-index/$local_file"
	printf "\\n"
      fi
    fi
  done
}


################################################################
# Do It
# 

cd "$src"
wftp-home

update_this_directory


# tag: Tom Lord Tue Jan 15 05:45:40 2002 (with-ftp-lib/ftp-push.sh)
#
