# 
# register-archive.sh - record the location of an archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "record the location of an archive\\n"
		printf "usage: register-archive [options] archive location\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -v --verbose                  print a reassuring message\\n"
		printf "\\n"
		printf " -d --delete                   remove a registered archive\\n"
		printf " -f --force                    force removals and fail without\\n"
		printf "                               complaint\\n"
		printf "\\n"
		printf "Record the location of ARCHIVE.\\n"
		printf "\\n"
		printf "With -d, remove the registration of a previously registered\\n"
		printf "archive.  When accompanied by -f, override permissions on\\n"
		printf "the registration file and don't complain if the archive is\\n"
		printf "not registered.\\n"
		printf "\\n"
		printf "A location should be either a directory name or an ftp URL.\\n"
		printf "\\n"
		printf "Archive locations are stored in ~/.arch-params/=locations.\\n"
		printf "\\n"
		printf "You must register the location of a remote archive before you\\n"
		printf "access it.  It is not strictly necessary to register the locations\\n"
		printf "of local archives (you can always specify their location using\\n"
		printf "command line arguments and/or environment variables), but registering\\n"
		printf "local archive locations is recommend (for simplicity).\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

verbose=0
delete=0
force=0

while test $# -ne 0 ; do

  case "$1" in 

    -v|--verbose)	shift
    			verbose=1
			;;

    -d|--delete)	shift
    			delete=1
			;;

    -f|--force)		shift
    			force=1
			;;

    -*)			printf "register-archive: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $delete -eq 0 -a $# -ne 2 ; then
  printf "usage: register-archive [options] archive location\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
else
  archive="$1"
  location="$2"
fi

if test $delete -ne 0 -a $# -ne 1 ; then
  printf "usage: register-archive [options] -d archive\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
else
  archive="$1"
fi


################################################################
# Sanity Check
# 

larch valid-archive-name --errname register-archive -- "$archive"

if test $delete -eq 0 ; then
  larch valid-archive-location --errname register-archive -- "$location"
fi

################################################################
# Do it:
# 

mkdir -p ~/.arch-params/=locations

if test $delete -eq 0 ; then 

  cd  ~/.arch-params/=locations

  rm -f "$archive"
  touch "$archive"
  chmod go-rw "$archive"
  printf "%s\\n" "$location" >> "$archive"

  if test $verbose -ne 0 ; then
    printf "registered archive %s\\n" "$archive"
    printf "  location: %s\\n" "$location"
  fi

else

  cd  ~/.arch-params/=locations

  if test $force -eq 0 -a ! -e "$archive" ; then
    printf "register-archive: archive not registered (%s)\\n" "$archive" 1>&2
    exit 1
  fi

  if test $force -eq 0 ; then
    rm "$archive"
  else
    rm -f "$archive"
  fi
fi

