# 
# copy-from-cache.sh - try to copy a revision from the cache
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "try to copy a revision from a cache\\n"
		printf "usage: copy-from-cache [options] cachedir revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Look for a cached copy of REVISION in the project tree\\n"
		printf "which are immediate subdirectories of CACHEDIR.  If one is found,\\n"
		printf "copy it to the current directory and exit with status 0.  Otherwise,\\n"
		printf "exit with status 1.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

quiet=--quiet
report=--report
verbose=

while test $# -ne 0 ; do

  case "$1" in 

    --silent)	shift
    		quiet=
		report=
		verbose=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		;;

    --debug)	shift
    		larch heading "copy-from-cache: debugging output enabled\\n"
    		set -x
		;;


    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    --)			shift
			break
			;;

    -*)			printf "copy-from-cache: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
  printf "usage: copy-from-cache [options] cachedir revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

cachedir="$1"
shift

revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

larch valid-package-name -l -e copy-from-cache -- "$revision"

archive=`larch my-default-archive -R "$archroot" -A "$archive"`
category=`larch parse-package-name -b $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name -v $revision`
lvl=`larch parse-package-name -l $revision`

here="`pwd`"
cd "$cachedir"
cachedir="`pwd`"

################################################################
# Greetings
# 
#  This is an internal command.

if test ! -z "$verbose" ; then 
  ARCH__OUTLINE_DEPTH="$ARCH__OUTLINE_DEPTH*"
  larch heading "copy-from-cache\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "copy-from-cache start time: %s\\n" "`date`"
  larch heading --sub "seeking revision: %s\\n" "$branch--$vsn--$lvl"
  larch heading --sub "to copy to: %s\\n" "$here"
  larch heading --sub "cache directory: %s\\n" "$cachedir"
fi

################################################################
# Look for Cached Copies
# 


for wd in [=a-zA-Z+]* ; do
  for state in locked unlocked ; do
    if test -d "$wd/{arch}/++pristine-trees/$state/$category/$branch/$branch--$vsn/$archive/$branch--$vsn--$lvl" ; then
  
      if test ! -z "$report" ; then
        larch heading --sub "copying revision from local cache: %s\\n" $branch--$vsn--$lvl
        larch heading --sub --sub "from project tree: %s\\n" "$wd"
      fi
  
      cd "$wd/{arch}/++pristine-trees/$state/$category/$branch/$branch--$vsn/$archive/$branch--$vsn--$lvl"
      
      find . | copy-file-list -- - . "$here"
  
      if test ! -z "$verbose" ; then
        larch heading --sub "copy-from-cache finish time: %s\\n" "`date`"
      fi
  
      exit 0
  
    fi
  done
done

if test ! -z "$verbose" ; then
  larch heading --sub "no cached copy found\\n"
  larch heading --sub "copy-from-cache finish time: %s\\n" "`date`"
fi

exit 1

