# file-history.sh: report the history of all files
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "report the history of all files\\n"
                printf "usage: file-history [options] [[archive]/version]"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " -r --reverse                  reverse the listing\\n"
                printf "\\n"
		printf "Print a list of all files in a given version, along with an indication\\n"
		printf "of what patches modified, added, removed, or deleted them.\\n"
                printf "\\n"
                printf "The output format contains a section for each file:\\n"
                printf "\\n"
                printf "	TAG\\n"
                printf "		PATCH	[ADMR]	[NAME]\\n"
                printf "		PATCH	[ADMR]	[NAME]\\n"
                printf "		...\\n"
                printf "\\n"
                printf "The PATCH field is a patch level where the file was affected\\n"
                printf "in some way.  The next field is some non-empty string of letters\\n"
                printf "from [ADMR] (added, deleted, modified, renamed).\\n"
                printf "\\n"
                printf "If the file was added or renamed, the new name is included in\\n"
                printf "the third field which is otherwise empty.\\n"
                printf "\\n"
                printf "The indented fields are sorted patch level.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

reverse=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-find: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-find: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -*)                 printf "file-history: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: file-history [options] [archive]/version\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  versionspec="$1"
  shift
fi

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library -e file-history`"

larch valid-package-name -e file-history --vsn "$versionspec"
archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$versionspec\"`"
version="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --non-arch \"$versionspec\"`"

libdir="`larch library-find \"$archive/$version--base-0\"`"
cd "$libdir"
cd ..
libdir="`pwd`"


################################################################
# Do It
# 

larch library-revisions $reverse --full "$archive/$version" \
| awk -f "$ARCH_SUBCMD_ROOT/library/file-history.awk"

# tag: Tom Lord Sun Jan 20 01:11:00 2002 (library/file-history.sh)
#
