// vscreen_widget.h  -*-c++-*-
//
// "widgets" are sort of what they sound like -- entities that get drawn on the
// screen in some stacked order and can grab keyboard input.  By default, the
// widget currently on the top of the stack has keyboard "focus".  (overriding
// this may be an option eventually but isn't right now)
// (the widget with keyboard focus gets to determine cursor information)
//
//  A note on memory management: to avoid problems with references to widgets
// remaining on the stack, widget destructions are queued for later handling.
// handle_pending_deletes should be called when it's safe to do so (ie, in the
// main loop)
//
//  I could maybe override the delete operator (??), but that would be
// confusing; instead, I ask that widget deletion go through the "destroy"
// routine.  I don't think it's possible to enforce this via the compiler, but
// please do; Bad Things may happen if you don't..

#ifndef VSCREEN_WIDGET_H
#define VSCREEN_WIDGET_H

#include <list>
#include <sigc++/signal_system.h>

#include "curses++.h"

class vs_container;

struct size
{
  int w, h;
  size(int _w, int _h):w(_w), h(_h) {}
};

struct point
{
  int x, y;
  point(int _x, int _y):x(_x), y(_y) {}
};

struct rect
{
  int x, y, w, h;
  rect(int _x, int _y, int _w, int _h):x(_x), y(_y), w(_w), h(_h) {}
  void set_size(const size &s) {w=s.w; h=s.h;}
  size get_size() {return size(w, h);}
};

class keybindings;

class vscreen_widget:public SigC::Object
{
  friend class vs_container;

  // FIXME: trim this list :P
  friend bool vscreen_poll();
  friend void vscreen_mainloop(int);
  friend void vscreen_redraw();
  friend void vscreen_settoplevel(vscreen_widget *);
  friend void vscreen_suspend();
  friend void vscreen_resume();
  friend void vscreen_updatecursornow();
  friend void vscreen_handleresize();

  // Used to store info on externally/explicitly set bindings.
  struct binding_connection
  {
    std::string keyname;

    keybindings *bindings;

    SigC::Slot0<void> slot;

    binding_connection():bindings(NULL) {}
    binding_connection(const std::string &_keyname, keybindings *_bindings, const SigC::Slot0<void> &_slot)
      :keyname(_keyname), bindings(_bindings), slot(_slot) {}
  };

  // Bindings set via connect_key() and connect_key_post()
  std::list<binding_connection> auxillary_bindings, auxillary_post_bindings;

  cwindow win;

  int timeout_value;

  vs_container *owner;

  // Needed for recreating the window when the vscreen's window gets switched.
  // This stores the CURRENT size of the widget.
  rect geom;

  // Whether the widget is visible (distinct from whether it has a window;
  // answers the question "should this widget have a window?")
  bool visible;

  // The background to paint before drawing.
  int bgattr;

  // Tracks whether or not we have the focus.
  bool isfocussed;

  // Used to set the owner-window without setting the owner.  Used only
  // to handle the toplevel widget (which has a window but no owner)
  // Like alloc_size
  void set_owner_window(cwindow _win, int x, int y, int w, int h);

  // Used to update the "focussed" state
  void set_isfocussed(bool _isfocussed);
protected:
  cwindow get_win() {return win;}

  virtual void paint()=0;

  // Returns true if it consumed the character (otherwise, it'll be processed
  // further)
  virtual bool handle_char(chtype ch);
public:
  static void handle_pending_deletes();

  // show() and hide() do the expected.  show_all() makes a container show
  // all of its "children".  (err..it doesn't make sense for menubars to show
  // their menus, but aside from that..)
  void show();
  virtual void show_all();
  void hide();
  void toggle_visible()
  {
    if(visible)
      hide();
    else
      show();
  }
  void set_visible(bool _visible)
  {
    if(visible!=_visible)
      {
	if(_visible)
	  show();
	else
	  hide();
      }
  }

  vscreen_widget();

  virtual ~vscreen_widget();

  // This should be called when an arbitrary widget is to have a character
  // sent to it.
  bool dispatch_char(chtype);

  // This should be called when an arbitrary widget is to have a mouse event
  // sent to it.  Override it to change mousing behavior.
  virtual void dispatch_mouse(short id, int x, int y, int z, mmask_t bstate);

  virtual size size_request()=0;
  // Returns the desired widget size.

  virtual bool focus_me();
  // Returns true if this widet wants the keyboard focus (used in, eg, dialog
  // boxes)

  bool get_isfocussed() {return isfocussed;}

  void alloc_size(int x, int y, int w, int h);
  // Sets the geometry of this widget
  void set_owner(vs_container *w);
  // Makes this widget a child of the given widget (incidentally will delete
  // any allocated size; setting the owner to NULL hides the widget for now)

  void display();

  int timeout(int msecs);

  // Override with care!  The main reason to do so is if you want to use the
  // destroy() routine to signal something.  Doing so is, uh, a hack.  A
  // nasty hack.  In fact, this is mainly done to fix a bug in the
  // download_list for aptitude which I don't have the motivation to fix
  // properly yet.
  virtual void destroy();

  vs_container *get_owner() {return owner;}

  virtual bool get_cursorvisible()=0;
  virtual point get_cursorloc()=0;

  int get_startx() {return geom.x;}
  int get_starty() {return geom.y;}
  int get_width() {return geom.w;}
  int get_height() {return geom.h;}

  bool get_visible() {return visible;}

  // Should NOT be overridden -- that was a thinko
  void sync() {if(win) {win.touch(); win.noutrefresh();}}

  // from the original vscreen
  int scroll(int n=1) {return win?win.scroll(n):0;}

  int addch(chtype ch) {return win?win.addch(ch):0;}
  int mvaddch(int y, int x, chtype ch) {return win?win.mvaddch(y,x,ch):0;}
  int addstr(const char *str) {return win?win.addstr(str):0;}
  int addnstr(const char *str, int n) {return win?win.addnstr(str, n):0;}
  int mvaddstr(int y, int x, const char *str) {return win?win.mvaddstr(y, x, str):0;}
  int mvaddnstr(int y, int x, const char *str, int n) {return win?win.mvaddnstr(y, x, str, n):0;}

  int attroff(int attrs) {return win?win.attroff(attrs):0;}
  int attron(int attrs) {return win?win.attron(attrs):0;}
  int attrset(int attrs) {return win?win.attrset(attrs):0;}

  void bkgdset(const chtype ch) {if(win) win.bkgdset(ch);}
  int bkgd(const chtype ch) {return win?win.bkgd(ch):0;}
  chtype getbkgd() {return win?win.getbkgd():0;}

  int border(chtype ls, chtype rs, chtype ts, chtype bs, chtype tl, chtype tr, chtype bl, chtype br)
    {return win?win.border(ls,rs,ts,bs,tl,tr,bl,br):0;}
  int box(chtype verch, chtype horch) {return win?win.box(verch,horch):0;}
  int hline(chtype ch, int n) {return win?win.hline(ch,n):0;}
  int vline(chtype ch, int n) {return win?win.vline(ch,n):0;}
  int mvhline(int y, int x, chtype ch, int n) {return win?win.mvhline(y, x, ch, n):0;}
  int mvvline(int y, int x, chtype ch, int n) {return win?win.mvvline(y,x,ch,n):0;}

  int delch() {return win?win.delch():0;}
  int mvdelch(int y, int x) {return win?win.mvdelch(y, x):0;}

  int deleteln() {return win?win.deleteln():0;}
  int insdelln(int n) {return win?win.insdelln(n):0;}
  int insertln() {return win?win.insertln():0;}

  int echochar(chtype ch) {return win?win.echochar(ch):0;}

  int move(int y, int x) {return win?win.move(y,x):0;}
  void getyx(int &y, int &x) {if(win) win.getyx(y,x); else y=x=0;}
  void getbegyx(int &y, int &x) {if(win) win.getbegyx(y,x); else y=x=0;}
  void getmaxyx(int &y, int &x) {if(win) win.getmaxyx(y,x); else y=x=0;}
  int getmaxy() {return win?win.getmaxy():0;}
  int getmaxx() {return win?win.getmaxx():0;}

  void show_string_as_progbar(int x, int y, std::string s, int attr1, int attr2, int size1, int totalsize) {if(win) win.show_string_as_progbar(x, y, s, attr1, attr2, size1, totalsize);}

  void display_header(std::string s, unsigned int attr) {if(win) win.display_header(s, attr);}
  void display_status(std::string s, unsigned int attr) {if(win) win.display_status(s, attr);}

  int erase() {return win?win.erase():0;}
  int clear() {return win?win.clear():0;}
  int clrtobot() {return win?win.clrtobot():0;}
  int clrtoeol() {return win?win.clrtoeol():0;}

  // FIXME: we should preserve these settings ourselves and restore them on
  // set_win().  ?
  int keypad(bool bf) {return win?win.keypad(bf):0;}
  int meta(bool bf) {return win?win.meta(bf):0;}

  bool enclose(int y, int x)
  {
    if(win)
      return y>=geom.y && y<geom.y+geom.h && x>=geom.x && x<geom.x+geom.w;
    else
      return false;
  }

  void set_bg(int _bgattr);
  int get_bg() {return bgattr;}

  typedef std::list<binding_connection>::iterator key_connection;
  // This can be used to connect to a pseudo-signal for keypresses.
  // Most useful for stuff like setting up hotkeys and keyboard accelerators..
  key_connection connect_key(const std::string &key,
			     keybindings *bindings,
			     const SigC::Slot0<void> &slot);
  // Same, but the key is tested for after all other possibilities are
  // exhausted.
  key_connection connect_key_post(const std::string &key,
				  keybindings *bindings,
				  const SigC::Slot0<void> &slot);

  // The opposite..
  void disconnect_key(key_connection c);
  // Eww, do I really need two of these?
  void disconnect_key_post(key_connection c);

  // Signals:
  //
  // I use signals for events that an external object (eg,
  // a container) might want to act on.  For instance, when an object is
  // hidden, its parent might want to rearrange its children.
  //
  // In contrast, virtual methods are used for specific behaviors of this
  // class: for instance, displaying the widget itself.

  SigC::Signal0<void> shown_sig;
  // Emitted when the object is shown.  (not to be confused with the obsolete
  // show_sig, which was a request by the object to be shown)

  SigC::Signal0<void> hidden_sig;
  // similarly

  SigC::Signal0<void> destroyed;
  // Sent before a widget is destroyed.
  // A widget is always hidden before being destroyed

  SigC::Signal0<void> do_layout;
  // Sent when the widget's layout needs to be recalculated and child windows
  // need to be re-updated (mainly when the size is altered)
  // This should not be called directly by the user.  Use vscreen_queuelayout()
  // instead.

  SigC::Signal0<void> focussed;
  SigC::Signal0<void> unfocussed;
  // Sent when we gain or lose the keyboard focus.
};

#endif

