// download_bar.cc
//
//  Copyright 1999,2000 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#include "aptitude.h"

#include "download_bar.h"
#include "ui.h"

#include <vscreen/config/keybindings.h>
#include <vscreen/config/colors.h>
#include <vscreen/vs_minibuf_win.h>
#include <vscreen/vs_util.h>

#include <apt-pkg/strutl.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/acquire-worker.h>

#include <signal.h>

#define CRITICAL_ENTER	\
  vscreen_acquirelock();\
  sigset_t ___signals,___oldset;\
  sigemptyset(&___signals);	\
  sigaddset(&___signals, SIGWINCH);\
  sigprocmask(SIG_BLOCK, &___signals, &___oldset);\
  assert(!sigismember(&___oldset, SIGWINCH));

#define CRITICAL_EXIT	\
  sigprocmask(SIG_SETMASK, &___oldset, &___signals);\
  assert(sigismember(&___signals, SIGWINCH));\
  vscreen_releaselock();

bool download_status_bar::MediaChange(string media, string drive)
{
  CRITICAL_ENTER

  char buf[512];
  snprintf(buf, 512,
	   _("Please insert the following disc into the drive \"%s\":\n%s"),
	   drive.c_str(), media.c_str());

  vscreen_widget *w=vs_dialog_ok(buf, SigC::slot(vscreen_exitmain),
				 get_color("MediaChange"));
  w->show_all();
  popup_widget(w);

  CRITICAL_EXIT

  vscreen_mainloop();  // Eeeeeek!  Recursive mainloop!  I'm afraid..

  return true;
}

void download_status_bar::IMSHit(pkgAcquire::ItemDesc &itm)
{
  CRITICAL_ENTER

  last_msg=_("Hit ")+itm.Description;
  last_switchtime=time(0);

  vscreen_update();
  vscreen_tryupdate();

  CRITICAL_EXIT
}

void download_status_bar::Fetch(pkgAcquire::ItemDesc &itm)
{
  CRITICAL_ENTER

  last_msg=_("Downloading ")+itm.ShortDesc;
  last_switchtime=time(0);

  vscreen_update();
  vscreen_tryupdate();

  CRITICAL_EXIT
}

void download_status_bar::Done(pkgAcquire::ItemDesc &itm)
{
  CRITICAL_ENTER

  last_msg=_("Got ")+itm.Description;
  last_switchtime=time(0);

  vscreen_update();
  vscreen_tryupdate();

  CRITICAL_EXIT
}

void download_status_bar::Fail(pkgAcquire::ItemDesc &itm)
{
  CRITICAL_ENTER

  last_msg=itm.Description+": "+itm.Owner->ErrorText;
  last_switchtime=time(0);

  vscreen_update();
  vscreen_tryupdate();

  CRITICAL_EXIT
}

bool download_status_bar::Pulse(pkgAcquire *Owner)
{
  pkgAcquireStatus::Pulse(Owner);

  if(difftime(time(0), last_switchtime)>1)
    {
      CRITICAL_ENTER

      pkgAcquire::Worker *test=Owner->WorkersBegin();
      bool found=false;
      while(test)
	{
	  if(test==last_worker)
	    {
	      found=true;
	      break;
	    }
	  test=Owner->WorkerStep(test);
	}

      if(!found)
	last_worker=Owner->WorkersBegin();
      else if(last_worker!=NULL)
	last_worker=Owner->WorkerStep(last_worker);

      if(last_worker==NULL)
	last_worker=Owner->WorkersBegin();

      while(last_worker!=NULL && !(last_worker->CurrentItem || !last_worker->Status.empty()))
	last_worker=Owner->WorkerStep(last_worker);

      if(last_worker==NULL)
	last_msg=_("Downloading...");
      else
	{
	  if(!last_worker->CurrentItem)
	    last_msg=last_worker->Status;
	  else
	    {
	      last_msg=last_worker->CurrentItem->ShortDesc;
	      if(last_worker->CurrentItem->Owner->Mode)
		last_msg+=string("[")+last_worker->CurrentItem->Owner->Mode+"]";

	      if(last_worker->TotalSize>0)
		last_msg+=": "+SizeToStr(last_worker->CurrentSize)+"B/"+SizeToStr(last_worker->TotalSize)+"B";
	      else if(last_worker->CurrentSize>0)
		last_msg+=": "+SizeToStr(last_worker->CurrentSize)+"B";
	    }
	}

      vscreen_poll();

      vscreen_update();
      vscreen_tryupdate();

      CRITICAL_EXIT
    }

  return !cancelled;
}

void download_status_bar::paint()
{
  int width=getmaxx();
  string todisp=last_msg,totalprogress;

  int barsize=0;
  if((TotalBytes+TotalItems)>0)
    {
      char progress_string[50]; // More'n enough chars.
      unsigned long ETA=(unsigned long)((TotalBytes-CurrentBytes)/CurrentCPS);
      // Straight from acqprogress.cc, that is..
      
      barsize=int((double(width*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems));
      if(barsize>width)
	barsize=width;

      snprintf(progress_string, 50, " [ %i%% ] (%sB/s %s )", int(double((100.0*(CurrentBytes+CurrentItems)))/(TotalBytes+TotalItems)), SizeToStr(CurrentCPS).c_str(), TimeToStr(ETA).c_str());
      totalprogress=progress_string;
    }

  if(totalprogress.size()>(unsigned) width)
    todisp="";
  else if(todisp.size()+totalprogress.size()>(unsigned) width)
    todisp=last_msg.substr(0, width-totalprogress.size());
  else
    while(todisp.size()<(width-totalprogress.size()))
      todisp+=" ";

  todisp+=totalprogress;

  show_string_as_progbar(0,
			 0,
			 todisp,
			 get_color("Progress"),
			 get_color("ScreenStatusColor"),
			 barsize,
			 width);
}

void download_status_bar::Start()
{
  pkgAcquireStatus::Start();
}

void download_status_bar::Stop()
{
  CRITICAL_ENTER

  pkgAcquireStatus::Stop();

  destroy();

  CRITICAL_EXIT
}

bool download_status_bar::handle_char(chtype ch)
{
  if(global_bindings.key_matches(ch, "ExitScreen"))
    {
      cancelled=true;
      return true;
    }
  else
    return false;
}

size download_status_bar::size_request()
{
  return size(0,0);
}
