/* 
   Copyright  1998, 1999 Enbridge Pipelines Inc. 
   Copyright  1999 Dave Carrigan
   All rights reserved.

   This module is free software; you can redistribute it and/or modify
   it under the same terms as Apache itself. This module is
   distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. The copyright holder of this
   module can not be held liable for any general, special, incidental
   or consequential damages arising out of the use of the module.

   $Id: auth_ldap_cache.c,v 1.2 1999/08/11 03:17:03 dave Exp $

*/

#include "auth_ldap.h"
extern module MODULE_VAR_EXPORT auth_ldap_module;

/* Stolen from glib */
static const int primes[] =
{
  11,
  19,
  37,
  73,
  109,
  163,
  251,
  367,
  557,
  823,
  1237,
  1861,
  2777,
  4177,
  6247,
  9371,
  14057,
  21089,
  31627,
  47431,
  71143,
  106721,
  160073,
  240101,
  360163,
  540217,
  810343,
  1215497,
  1823231,
  2734867,
  4102283,
  6153409,
  9230113,
  13845163,
  0
};

ldapcache *
auth_ldap_new_cache(int maxentries)
{
  int i;
  ldapcache *ret;

  if (maxentries == 0) 
    return NULL;

  ret = (ldapcache *)malloc(sizeof(ldapcache));
  ret->maxentries = maxentries;
  ret->size = maxentries / 3;
  if (ret->size < 64) ret->size = 64;
  for (i = 0; primes[i] && primes[i] < ret->size; ++i) ;
  ret->size = primes[i]? primes[i] : primes[i-1];
  ret->nodelist = (cache_node **)calloc(ret->size, sizeof(cache_node *));
  for (i=0; i < ret->size; ++i)
    ret->nodelist[i] = NULL;
  return ret;
}

/*
 * Taken from Glib g_str_hash function
*/
unsigned long
auth_ldap_hashdn(const char *dn)
{
  const char *p;
  unsigned long h=0, g;

  for(p = dn; *p != '\0'; ++p) {
    h = ( h << 4 ) + *p;
    if ( ( g = h & 0xf0000000 ) ) {
      h = h ^ (g >> 24);
      h = h ^ g;
    }
  }

  return h;
}

/*
 * Attempts to bind to the LDAP server using the credentials provided. 
 * If there's a cache, will first check the cache and return the cached result
 * (if the entry is not stale). If an actual ldap operation is required and a
 * cache exists, will update the cache.
 *
 * The mutex for the connection should already be acquired.
 */

int
auth_ldap_authbind(const char *pw, request_rec *r)
{
  int result;
  time_t curtime;
  cache_node *node;
  auth_ldap_config_rec *sec;
  auth_ldap_server_conf *conf;
  int failures = 0;

  conf = (auth_ldap_server_conf *)ap_get_module_config(r->server->module_config,
						       &auth_ldap_module);
  sec = (auth_ldap_config_rec *)
    ap_get_module_config(r->per_dir_config, &auth_ldap_module);

  time(&curtime);
  node = NULL;
  if (sec->ldc->ldapopcache != NULL) {
    /*
     * Cache handling. Entries are cached by DN and contain the last password
     * used to bind. If the bind was successful, the timestamp of the
     * bind is stored; otherwise 0 is stored. If the password this time doesn't 
     * match the password stored in the cache, then a bind is required.
     */ 
    unsigned long hash;

    hash = auth_ldap_hashdn(sec->dn) % sec->ldc->ldapopcache->size;
    for (node = sec->ldc->ldapopcache->nodelist[hash]; node != NULL; node = node->next)
      if (strcmp(node->dn, sec->dn) == 0) 
	break;
    
    /* 
     * At this point, node points to the cached DN, or null if the DN
     * isn't in the cache. 
     */
    if (node) {
      compare_node *n;
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		    "{%d} Found entry matching %s in cache", 
		    (int)getpid(), sec->dn);
      if (strcmp(node->pw, pw) != 0 || 
	  (conf->opcache_ttl > 0 && curtime - node->bindtime >= conf->opcache_ttl)) {
	if (strcmp(node->pw, pw) != 0) {
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
			"{%d} Deleting cached entry for %s because passwd "
			"doesn't match cache", 
			(int)getpid(), sec->dn);
	} else {
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
			"{%d} Deleting stale cached entry for %s (%d > %d)", 
			(int)getpid(), sec->dn, (int)(curtime - node->bindtime), 
			(int)(conf->opcache_ttl));
	}
	free(node->pw);
	node->pw = strdup(pw);
	node->bindtime = 0;
	n = node->compares;
	while (n != NULL) {
	  compare_node *d = n;
	  n = n->next;
	  free(d->attrib);
	  free(d->value);
	  free(d);
	}
	node->compares = NULL;
      } else {			/* Passwords match, entry isn't stale, return! */
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		      "{%d} Found cached entry for %s", (int)getpid(), sec->dn);
	return node->bindtime? OK : AUTH_REQUIRED;
      }
    }
    
    /*
     * At this point, if node is NULL, it's because this DN isn't in the cache.
     * If it's not NULL, it's because the node's password and the current password
     * don't match, or else the entry is stale. 
     */
    if (node == NULL) {
      cache_node *p;
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		    "{%d} Inserting %s into bind cache", (int)getpid(), sec->dn);
      node = (cache_node *)malloc(sizeof(cache_node));
      node->dn = strdup(sec->dn);
      node->pw = strdup(pw);
      node->bindtime = 0;
      node->compares = NULL;
      node->next = NULL;
      for (p = sec->ldc->ldapopcache->nodelist[hash]; 
	   p && p->next != NULL; p = p->next) ;
      if (p)
	p->next = node;
      else
	sec->ldc->ldapopcache->nodelist[hash] = node;
    }
  }

  /*
   * If we get here, it's either because there's no cache, or there wasn't
   * a positive match in the cache. If there is a cache, node will be non-null
   * and will be pointing to the node for this DN in the cache.
   * Now, do the auth. If node is non-null, put the auth results into node.
   */
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Validating uncached user %s via bind", (int)getpid(), sec->dn);

 start_over:
  if (failures++ > 10) {
    auth_ldap_log_reason(r, "Too many failures connecting to LDAP server");
    return 0;
  }
  if (!auth_ldap_connect_to_server(r)) {
    return AUTH_REQUIRED;
  }

  sec->needbind = 1;
  /* 
   * Attempt to bind with the retrieved dn and the password. If the bind
   * fails, it means that the password is wrong (the dn obviously
   * exists, since we just retrieved it)
   */
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} LDAP OP: simple bind", (int)getpid());
  if ((result = 
       ldap_simple_bind_s(sec->ldc->ldap, sec->dn, const_cast(pw))) == LDAP_SERVER_DOWN) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Server is down; reconnecting and starting over", (int)getpid());
    auth_ldap_free_connection(r, 1);
    goto start_over;
  }
  
  if (result != LDAP_SUCCESS) {
    auth_ldap_log_reason(r, "User bind as %s failed: LDAP error: %s; URI %s",
			 sec->dn, ldap_err2string(result), r->uri);
    ap_note_basic_auth_failure(r);
    return AUTH_REQUIRED;
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		"{%d} authenticate: accepting", (int)getpid());

  /* Success! */
  if (node)
    node->bindtime = curtime;
  return OK;
}

/*
 * Does an ldap_compare operation. Optionally caches compares to avoid the
 * operation. 
 */
int
auth_ldap_compare(const char *dn, const char *attrib, const char *value, request_rec *r)
{
  int result;
  compare_node *node = NULL;
  time_t curtime;
  auth_ldap_config_rec *sec;
  auth_ldap_server_conf *conf;
  int failures = 0;

  conf = (auth_ldap_server_conf *)ap_get_module_config(r->server->module_config,
						    &auth_ldap_module);
  sec = (auth_ldap_config_rec *)
    ap_get_module_config(r->per_dir_config, &auth_ldap_module);

  time(&curtime);

  if (ap_acquire_mutex(sec->ldc->mtx) != MULTI_OK)
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "Could not acquire connection mutex. Expect deadlocks.");
    

  if (sec->ldc->ldapopcache != NULL && conf->cache_compares) {
    unsigned long hash = auth_ldap_hashdn(dn) % sec->ldc->ldapopcache->size;
    cache_node *p;

    /* Find the cached DN */
    for (p = sec->ldc->ldapopcache->nodelist[hash]; p != NULL; p = p->next)
      if (strcmp(p->dn, dn) == 0) break;

    if (p) {
      compare_node *q;
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "{%d} found entry for %s in cache", (int)getpid(), dn);
      for (q = NULL, node = p->compares; node != NULL; q = node, node = node->next)
	if (strcmp(node->attrib, attrib) == 0 &&
	    strcmp(node->value, value) == 0)
	  break;

      if (node == NULL) {
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		      "{%d} adding new compare %s=%s for %s to cache", 
		      (int)getpid(), attrib, value, dn);
	node = (compare_node *)malloc(sizeof(compare_node));
	node->attrib = strdup(attrib);
	node->value = strdup(value);
	node->comparetime = 0;
	node->next = NULL;
	if (q)
	  q->next = node;
	else
	  p->compares = node;
      } else {
	if (conf->opcache_ttl > 0 && curtime - node->comparetime >= conf->opcache_ttl) {
	  node->comparetime = 0;
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
			"{%d} redoing expired cached compare %s=%s for %s", 
			(int)getpid(), attrib, value, dn);
	} else {
	  int ret = node->comparetime;
	  if (!ap_release_mutex(sec->ldc->mtx))
	    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
			  "Could not release connection mutex. Expect deadlocks.");
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
			"{%d} found cached compare %s=%s for %s", 
			(int)getpid(), attrib, value, dn);
	  
	  return ret;
	}
      }
    } else {
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "{%d} no entry in cache for compare %s=%s in %s", (int)getpid(), 
		    attrib, value, dn);
    }
  }

  /*
   * At this point, node points to a new compare node to be filled in with
   * a call to ldap_compare, or else it's NULL
   */

 start_over:
  if (failures++ > 10) {
    auth_ldap_log_reason(r, "Too many failures connecting to LDAP server");
    return 0;
  }
  if (!auth_ldap_connect_to_server(r)) {
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");
    return 0;
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Doing LDAP compare of uncached %s=%s in entry %s", (int)getpid(),
		attrib, value, dn);

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} LDAP OP: compare", (int)getpid());
  if ((result = ldap_compare_s(sec->ldc->ldap, const_cast(dn), 
			       const_cast(attrib), const_cast(value))) == 
      LDAP_SERVER_DOWN) { 
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Server is down; reconnecting and starting over", (int)getpid());
    auth_ldap_free_connection(r, 1);
    goto start_over;
  }

  if (node) {
    int ret;
    ret = node->comparetime = (result == LDAP_COMPARE_TRUE)? curtime : 0;
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");
    return ret;
  } else {
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");
    return (result == LDAP_COMPARE_TRUE)? curtime : 0;
  }
}
