/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <sqlext.h>
#include "odbcResultRow.hpp"
#include "odbcStatement.hpp"
#include "odbcWrap.hpp"

using namespace std;

ODBCStatement::ODBCStatement( SQLHDBC connection, const std::string &query )
  throw (Error):
  statement( NULL )
{
  try {
    if ( !query.empty() ) {
      odbcWrap( SQLAllocHandle( SQL_HANDLE_STMT, connection, &statement ),
                SQL_HANDLE_STMT, statement );

      assert( statement );
      assert( sizeof( char ) == sizeof( SQLCHAR ) );
#ifndef NDEBUG
      cerr << query << endl;
#endif
      odbcWrap( SQLExecDirect( statement, (SQLCHAR *)query.c_str(), SQL_NTS ),
                SQL_HANDLE_STMT, statement );
    };
  } catch ( Error &e ) {
    if ( statement )
      SQLFreeHandle( SQL_HANDLE_STMT, statement );
    ERRORMACRO( false, Error, ,
                "Error in query \"" << query << "\": " << e.what() );
  }
}

ODBCStatement::~ODBCStatement(void)
{
  if ( resultRow )
    resultRow->invalidate();
  if( statement )
    SQLFreeHandle( SQL_HANDLE_STMT, statement );
}

int ODBCStatement::getNumCols(void) throw (Error)
{
  SQLSMALLINT retVal;
  if ( statement )
    odbcWrap( SQLNumResultCols( statement, &retVal ),
              SQL_HANDLE_STMT, statement );
  else
    retVal = 0;
  return retVal;
}

std::string ODBCStatement::getColAttr( int col ) throw (Error)
{
  SQLCHAR retVal[4096];
  ERRORMACRO( col >= 0 && col < getNumCols(), Error, ,
              "Column number of meta-column must be between 0 and "
              << ( getNumCols() - 1 ) << " but was " << col << "." );
  SQLColAttribute( statement, col + 1, SQL_DESC_LABEL, retVal,
                   sizeof(retVal), NULL, NULL );
  assert( sizeof( SQLCHAR ) == sizeof( char ) );
  return (char *)retVal;
}

ResultRowPtr &ODBCStatement::fetchRow(void) throw (Error)
{
  if ( resultRow ) {
    ERRORMACRO( resultRow.use_count() <= 1, Error, ,
                "Can not fetch new row, "
                "because previous row was not released yet." );
    assert( resultRow.use_count() == 1 );
  };
  if ( SQL_SUCCEEDED( SQLFetch( statement ) ) )
    resultRow = ResultRowPtr( new ODBCResultRow( statement ) );
  else
    resultRow.reset();
  return resultRow;
}
