%{
/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <assert.h>
#include <klocale.h>
#include <stdlib.h>
#include <string.h>

/* The code being executed by the lexer is a mess and a real hack indeed.
   However this is only reflecting the bad design of the Mealmaster
   file-format.
   Alas! If only people would now better, before they start developing
   file-formats. */

/* Functions for keeping track of read-position. */
extern long position;
extern int recipeStartPos;
extern int recipeStartLine;
extern int recipeEndPos;
extern int recipeEndLine;

/* Use external function for reading in data. */
extern int readData( char *buffer, int max_size );
#define YY_INPUT(buffer,result,max_size ) \
{ \
   assert( YY_NULL == 0 ); \
   result = readData( buffer, max_size ); \
}

/* Keep track of read-position. */
#define YY_USER_ACTION { position += yyleng; }
#define userinput(c) ( position++, yyinput() )
#define userunput(c) \
{ \
   position--; \
   unput(c); \
}

int currentBuffer = 0;
int continuationBuffer = 3;
int ingredientOpen[ 4 ];

extern void writeData( int index, const char *buffer, int size );
extern void writeUnit( int index, const char *buffer, int size );
extern void moveData( int index );

void writeData( int index, const char *buffer )
{
  writeData( index, buffer, strlen( buffer ) );
}

void writeDouble( int index, double number )
{
  char buffer[40];
  /* Do NOT use exponential representation, because XSL cannot handle it. */
  sprintf( buffer, "%f", number );
  writeData( index, buffer, strlen( buffer ) );
}

void writeNumber( int index, int number )
{
  char buffer[40];
  sprintf( buffer, "%d", number );
  writeData( index, buffer, strlen( buffer ) );
}

void openIngredient( int index )
{
  assert( index >= 0 && index <= 1 );
  assert( ingredientOpen[ index ] == 0 || ingredientOpen[ index ] == 1 );
  if ( ingredientOpen[ index ] == 0 ) {
    writeData( index + 1, "      <ingredient>\n" );
    ingredientOpen[ index ] = 1;
  };
}

void startPreparation( int index )
{
  assert( index >= 0 );
  assert( index <= 3 );
  if ( index <= 1 ) {
    if ( ingredientOpen[ index ] == 1 ) {
      writeData( index + 1, "</name>\n        <prep>" );
      ingredientOpen[ index ] = 2;
    };
  } else if ( index == 2 ) {
    currentBuffer = 4;
    continuationBuffer = 4;
  };
}

void closeIngredient( int index )
{
  assert( index >= 0 && index <= 1 );
  if ( ingredientOpen[ index ] == 1 ) {
    writeData( index + 1, "</name>\n      </ingredient>\n" );
    ingredientOpen[ index ] = 0;
  };
  if ( ingredientOpen[ index ] == 2 ) {
    writeData( index + 1, "</prep>\n      </ingredient>\n" );
    ingredientOpen[ index ] = 0;
  };
}

void moveDataX(void) {
  moveData( 1 );
  if ( ingredientOpen[ 0 ] == 1 ) {
    moveData( 3 );
    if ( continuationBuffer == 4 ) {
      startPreparation( 0 ); moveData( 1 );
      moveData( 4 );
    };
    closeIngredient( 0 ); moveData( 1 );
  } else if ( ingredientOpen[ 0 ] == 2 ) {
    moveData( 3 );
    moveData( 4 );
    closeIngredient( 0 );
    moveData( 1 );
  };
  closeIngredient( 1 );
  moveData( 2 );
  continuationBuffer = 3;
  currentBuffer = 0;
};

/* Variables for collecting numbers to compute amount from. */
typedef enum { none, natural, fraction, composite, floating } AmountType;
AmountType amountType;
double amountDouble;
int amountLength;
int amountNumbers[3];

/* Buffer for storing the current line, when trying to interpret it as
   ingredient. The buffer can be pushed back on the input-buffer of flex. */
int bufferLength = 0;
char buffer[12] = "           ";
void unputBuffer(void);

/* Variables for storing unicode-sequences and checking against overlong
   sequences. */
char unicodeSequence[7] = "      ";
int unicodeLength = 0;
int unicode;

/* Variables for delaying section titles. */
int titleSize = 0;
char titleText[300];
int sectionOpen = 1;
int paragraphOpen = 1;
int textOpen = 1;
int writeTitle( const char *text, int len )
{
  int retVal;
  if ( titleSize + len <= (signed)sizeof( titleText ) ) {
    memcpy( &titleText[ titleSize ], text, len );
    titleSize += len;
    retVal = 1;
  } else
    retVal = 0;
  return retVal;
}

/* Variable for delaying space-characters in ingredient-text.
   Trailing spaces are ignored. */
int ingredientSpaces[ 4 ];
int ingredientColumn;
void flushIngredientSpaces(void)
{
  assert( currentBuffer >= 1 && currentBuffer <= 4 );
  for ( int i=0; i<ingredientSpaces[ currentBuffer - 1 ]; i++ )
    writeData( currentBuffer, " ", 1 );
  ingredientSpaces[ currentBuffer - 1 ] = 0;
}
void addIngredientColumn(void);

/* Variables for delaying error messages. */
extern int lineNo;
extern int errorLine;
extern const char *errorMessage;
void reportError( const char *_errorMessage );
void ignoreRecipe( const char *_errorMessage );

void closeParagraph(void)
{
  if ( paragraphOpen ) {
    writeData( 0, "</par>\n" );
    paragraphOpen = 0;
  };
}

void closeText(void)
{
  closeParagraph();
  if ( textOpen ) {
    writeData( 0, "      </text>\n" );
    textOpen = 0;
  };
}

void closeSection(void)
{
  closeText();
  if ( sectionOpen ) {
    writeData( 0, "    </section>\n" );
    sectionOpen = 0;
  };
};

void closeRecipe(void);

void openSection(void)
{
  if ( !sectionOpen ) {
    writeData( 0, "    <section>\n" );
    if ( titleSize > 0 ) {
      writeData( 0, "      <title>" );
      writeData( 0, titleText, titleSize );
      writeData( 0, "</title>\n" );
      titleSize = 0;
    };
    sectionOpen = 1;
  };
};

void openParagraph(void)
{
  openSection();
  if ( !paragraphOpen ) {
    writeData( 0, "        <par>" );
    paragraphOpen = 1;
  };
}

void openText(void)
{
  openParagraph();
  if ( !textOpen ) {
    writeData( 0, "      <text>\n" );
    textOpen = 1;
  };
}

void proceedInstructions(void);

/* Options
   noyywrap: Prevent parser from searching for subsequent input sources.
   stack:    This lexer needs a stack.
   never-interactive: Do not call 'isatty' on yyin (yyin is NULL here!).
   nostdin: Set yyin and yyout to zero. */

%}

%option  noyywrap
%option stack
%option never-interactive
%option nostdinit

%x header header2 section section2 line blindline title categories categories2
%x servings servings2 ingredients ingredients2 ingredients3 skipws unit
%x instructions unicode0 printUnicode unicode2 unicode3 unicode3Check
%x unicode4 unicode4Check unicode5 unicode5Check unicode6 unicode6Check
%x unicodeError ignoreline ignoreUnicode errorState ingredientsStart
%x ingredient28 ingredient28End ingredient99 ingredientEnd
%x prepEnd titleUnicode

UNIT "x "|"sm"|"md"|"lg"|"cn"|"pk"|"pn"|"dr"|"ds"|"ct"|"bn"|"sl"|"ea"|"t "|"ts"|"T "|"tb"|"fl"|"c "|"pt"|"qt"|"ga"|"oz"|"lb"|"ml"|"cb"|"cl"|"dl"|"l "|"mg"|"cg"|"dg"|"g "|"kg"|"  "

%%

<*><<EOF>> {
#ifndef NDEBUG
  fprintf( stderr, "Encountered end-of-file.\n" );
#endif
  int retVal;
  switch ( YY_START ) {
  case INITIAL:
    retVal = YY_NULL;
    break;
  default:
    reportError( "Unexpected end of file." );
    closeRecipe();
    BEGIN( INITIAL );
    retVal = +2;
  };
  assert( YY_NULL == 0 );
  return retVal;
}

<INITIAL>MMMMM|----- {
  /* Seems to be somewhat out of sync. Don't now, how to solve this. */
  recipeStartPos = position - yyleng;
  recipeStartLine = lineNo;
  BEGIN( header );
}

<INITIAL>[^\n] yy_push_state( ignoreline );
<INITIAL>\n lineNo++;

<errorState>(MMMMM|-----)-*\r?\n {
  closeRecipe();
  lineNo++;
  return +2;
}
<errorState>\n lineNo++;
<errorState>[^\n] yy_push_state( ignoreline );

<skipws>[\ \t]/\ 
<skipws>[\ \t]/[^\ ] yy_pop_state();
<skipws>[^\ \t] {
  userunput( *yytext );
  yy_pop_state();
}

<header>[Mm][Ee][Aa][Ll]-[Mm][Aa][Ss][Tt][Ee][Rr] BEGIN( header2 );
<header>[ -~]
<header>\r?\n {
  lineNo++;
  BEGIN( INITIAL );
};
<header>[^ -~] {
  userunput( *yytext );
  yy_push_state( unicode0 );
}

<header2>-*\r?\n {
  lineNo++;
  paragraphOpen = 0;
  textOpen = 0;
  titleSize = 0;
  currentBuffer = 0;
  BEGIN( title );
  writeData( 0, "<recipe>\n  <title>" );
}
<header2>[ -~]
<header2>[^\n -~] {
  userunput( *yytext );
  yy_push_state( unicode0 );
}

<section>-+/[^-] {
  closeSection();
  if ( titleSize > 0 ) {
    /* Empty section has to be put in between. */
    openSection();
    closeSection();
  };
  BEGIN( section2 );
}
<section>\r?\n {
   lineNo++;
   ignoreRecipe( "Error reading section." );
}
<section>. ignoreRecipe( "Error reading section." );

<section2>-*\r?\n {
  lineNo++;
  yy_pop_state();
}
<section2>-  {
  if ( !writeTitle( "-", 1 ) )
    ignoreRecipe( "Section title to long." );
}
<section2>& {
  if ( !writeTitle( "&amp;", 5 ) )
    ignoreRecipe( "Section title to long." );
}
<section2>< {
  const char *ltText = "&lt;";
  const int ltLen = 4;
  assert( (signed)strlen( ltText ) == ltLen );
  if ( !writeTitle( "&lt;", 4 ) )
    ignoreRecipe( "Section title to long." );
}
<section2>> {
  if ( !writeTitle( "&gt;", 4 ) )
    ignoreRecipe( "Section title to long." );
}
<section2>[ -%'-,.-;=?-~]+ {
  if ( !writeTitle( yytext, yyleng ) )
    ignoreRecipe( "Section-title to long." );
}
<section2>\t {
  if ( !writeTitle( "   ", 3 ) )
    ignoreRecipe( "Section-title to long." );
}
<section2>[^\n\t -~] {
  userunput( *yytext );
  yy_push_state( titleUnicode );
  yy_push_state( unicode0 );
}

<blindline,ignoreline>[\t -~]+
<blindline,ignoreline>\r?\n {
  lineNo++;
  yy_pop_state();
}
<blindline>[^\t\n -~] {
  userunput( *yytext );
  yy_push_state( unicode0 );
}
<ignoreline>[^\t\n -~] {
  userunput( *yytext );
  yy_push_state( ignoreUnicode );
  yy_push_state( unicode0 );
}

<line,ingredient99,categories2>& writeData( currentBuffer, "&amp;", 5 );
<line,ingredient99,categories2>< writeData( currentBuffer, "&lt;", 4 );
<line,ingredient99,categories2>> writeData( currentBuffer, "&gt;", 4 );
<line,ingredient99,categories2>\t writeData( currentBuffer, "   ", 3 );

<ingredientsStart>. {
  userunput( *yytext );
  writeData( 0, "</unit>\n  </servings>\n  <ingredients>\n" );
  sectionOpen = 0;
  titleSize = 0;
  currentBuffer = 1;
  ingredientOpen[0] = 0;
  ingredientOpen[1] = 0;
  ingredientOpen[2] = 1;
  ingredientOpen[3] = 2;
  ingredientSpaces[2] = 0;
  ingredientSpaces[3] = 0;
  continuationBuffer = 3;
  yy_pop_state();
}

<line>[ -%'-;=?-~]+ writeData( currentBuffer, yytext, yyleng );
<ingredient99>[ -%'-:=?-~]+ writeData( currentBuffer, yytext, yyleng );
<ingredient99>; {
  startPreparation( currentBuffer - 1 );
}
<line,ingredient99>\r?\n {
   lineNo++;
   yy_pop_state();
}
<line,ingredient99>[^\n\t -~] {
  userunput( *yytext );
  yy_push_state( printUnicode );
  yy_push_state( unicode0 );
}

<ingredient28>\  {
  addIngredientColumn();
  assert( currentBuffer>= 1 && currentBuffer <= 4 );
  ingredientSpaces[ currentBuffer - 1 ]++;
}
<ingredient28>& {
  addIngredientColumn();
  flushIngredientSpaces();
  writeData( currentBuffer, "&amp;", 5 );
}
<ingredient28>< {
  addIngredientColumn();
  flushIngredientSpaces();
  writeData( currentBuffer, "&lt;", 4 );
}
<ingredient28>> {
  addIngredientColumn();
  flushIngredientSpaces();
  writeData( currentBuffer, "&gt;", 4 );
}

<ingredient28>[!-%'-:=?-~] {
  addIngredientColumn();
  flushIngredientSpaces();
  writeData( currentBuffer, yytext, yyleng );
}
<ingredient28>; {
  addIngredientColumn();
  if ( ingredientOpen[ currentBuffer - 1 ] == 1 ) {
    startPreparation( currentBuffer - 1 );
  } else {
    flushIngredientSpaces();
    writeData( currentBuffer, yytext, yyleng );
  };
}
<ingredient28,ingredient28End>\r?\n {
  currentBuffer = 1;
  lineNo++;
  yy_pop_state();
}
<ingredient28>[^\n -~] {
  addIngredientColumn();
  userunput( *yytext );
  yy_push_state( printUnicode );
  yy_push_state( unicode0 );
}

<ingredient28End>\ \  {
  /* Two-column format. */
  currentBuffer = 2;
  yy_pop_state();
}
<ingredient28End>[^\ ] {
  /* Overlong ingredient-line (standard-violation by B&S-software). */
  userunput( *yytext );
  BEGIN( ingredient99 );
}
<ingredient28End>\ /[^\ ] {
  /* Overlong ingredient-line (standard-violation by B&S-software). */
  userunput( *yytext );
  BEGIN( ingredient99 );
}

<title>[\ \t]
<title>\r?\n lineNo++;
<title>"Title: " {
  BEGIN( categories );
  yy_push_state( line );
}
<title>[^\n\t ] ignoreRecipe( "Expecting \"Title: \"." );

<categories>[\ \t]
<categories>\r?\n lineNo++;
<categories>"Categories: " {
  BEGIN( categories2 );
  writeData( 0, "</title>\n  <categories>\n    <category>" );
}
<categories>[^\n\t ] ignoreRecipe( "Expecting \"Categories: \"." );

<categories2>[ -%'-+\--;=?-~]+ writeData( 0, yytext, yyleng );
<categories2>\ *,\ * writeData( 0, "</category>\n    <category>" );

<categories2>\r?\n {
  lineNo++;
  writeData( 0, "</category>\n  </categories>\n  <servings>\n    <amount>" );
  BEGIN( servings );
}
<categories2>[^\n\t -~] {
  userunput( *yytext );
  yy_push_state( printUnicode );
  yy_push_state( unicode0 );
}

<servings>[\ \t]
<servings>\r?\n lineNo++;
<servings>"Yield: "|"Servings: " BEGIN( servings2 );
<servings>[^\n\t ] {
  ignoreRecipe( "Expecting \"Yield: \" or \"Servings: \"." );
}

<servings2>[0-9]+ {
  BEGIN( ingredients );
  yy_push_state( ingredientsStart );
  yy_push_state( line );
  yy_push_state( skipws );
  writeData( 0, yytext, yyleng );
  writeData( 0, "</amount>\n    <unit>" );
}
<servings2>. ignoreRecipe( "Expecting number." );

<ingredients>(MMMMM|-----)-*\r?\n {
  proceedInstructions();
  closeRecipe();
  lineNo++;
  return 1;
};
<ingredients>(MMMMM|-----)/[^\r\n] {
  moveDataX();
  yy_push_state( section );
}
<ingredients>\ {0,6}[0-9]+\.[0-9]*/[\ \/] {
   /* Floating-point number (f.e. "0.8" ) */
   if ( yyleng <= 7 ) {
      amountType = floating;
      amountDouble = atof( yytext );
      memcpy( buffer, yytext, yyleng );
      bufferLength = yyleng;
      BEGIN( unit );
      for ( int i=0; i<=7 - amountLength; i++ ) {
         char c = userinput();
         buffer[ bufferLength++ ] = c;
         if ( c != ' ' ) {
           proceedInstructions();
           break;
         };
      };
   } else {
      position -= yyleng;
      REJECT;
   };
}
<ingredients>\ {0,6}[0-9]+/[\ \/] {
   /* Natural number or beginning of fraction or composite number
      (f.e. "250", "1/3" or "1 1/2") */
   if ( yyleng <= 7 ) {
      amountNumbers[0] = atoi( yytext );
      amountLength = yyleng;
      memcpy( buffer, yytext, yyleng );
      bufferLength = yyleng;
      BEGIN( ingredients2 );
   } else {
      position -= yyleng;
      REJECT;
   };
}
<ingredients>\ *\r?\n {
  currentBuffer = 0;
  lineNo++;
}
<ingredients>\ {8} {
   /* No amount specified. */
   amountType = none;
   assert( yyleng == 8 );
   for ( int i=0; i<8; i++ )
      buffer[ i ] = ' ';
   bufferLength = 8;
   BEGIN( unit );
}
<ingredients>[^\n] {
   /* Interpretation as ingredient failed. */
   bufferLength = 0;
   userunput( *yytext );
   proceedInstructions();
}

<ingredients2>\/ {
   /* Fraction bar of a fraction (f.e. "13/20"). */
   buffer[ bufferLength++ ] = '/';
   if ( yyleng <= 7 - amountLength ) {
      assert( yyleng == 1 );
      amountLength++;
      BEGIN( ingredients3 );
      amountType = fraction;
   } else
     proceedInstructions();
}
<ingredients2>\ *[0-9]+/\/ {
   /* Second number of a composite number (f.e. "1 1/2"). */
   if ( yyleng + 1 <= 7 - amountLength ) {
      amountNumbers[1] = atoi( yytext );
      amountLength += yyleng + 1;
      for ( int i=0; i<yyleng; i++ )
         buffer[ bufferLength++ ] = yytext[i];
      char c = userinput();
      assert( c == '/' );
      buffer[ bufferLength++ ] = c;
      BEGIN( ingredients3 );
      amountType = composite;
   } else {
      position -= yyleng;
      REJECT;
   };
}
<ingredients2>\  {
   assert( yyleng == 1 );
   buffer[ bufferLength++ ] = ' ';
   if ( 1 <= 7 - amountLength ) {
      /* Store whitespace characters. */
      amountLength++;
   } else {
      /* Interpretation as natural number succeeded. */
      assert( amountLength == 7 );
      BEGIN( unit );
      amountType = natural;
   };
}
<ingredients2>[^ \/] {
   /* Interpretation as ingredient-line failed. */
   userunput( *yytext );
   proceedInstructions();
};

<ingredients3>[0-9]+/\  {
   /* Finishing interpretation of amount-text. */
   for ( int i=0; i<yyleng; i++ )
      buffer[ bufferLength++ ] = yytext[i];
   if ( yyleng <= 7 - amountLength ) {
      int denominator = atoi( yytext );
      switch ( amountType ) {
      case fraction:
         amountNumbers[1] = denominator;
         break;
      case composite:
         amountNumbers[2] = denominator;
         break;
      default:
         assert(false);
      };
      amountLength += yyleng;
      BEGIN( unit );
      for ( int i=0; i<=7 - amountLength; i++ ) {
         char c = userinput();
         buffer[ bufferLength++ ] = c;
         if ( c != ' ' ) {
           printf( "Encountered %c\n", c );
           proceedInstructions();
           break;
         };
      };
   } else
     proceedInstructions();
}
<ingredients3>. {
   /* Interpretation as ingredient-line failed. */
   userunput( *yytext );
   proceedInstructions();
}

<unit>{UNIT}/\  {
   openSection();
   bufferLength = 0;
   if ( currentBuffer == 0 )
     currentBuffer = 1;
   closeIngredient( currentBuffer - 1 );
   openIngredient( currentBuffer - 1 );
   if ( amountType != none ) {
     writeData( currentBuffer, "        <amount>" );
     switch ( amountType ) {
     case natural:
       writeData( currentBuffer, "<fraction><nominator>" );
       writeNumber( currentBuffer, amountNumbers[0] );
       writeData( currentBuffer, "</nominator><denominator>" );
       writeNumber( currentBuffer, 1 );
       writeData( currentBuffer, "</denominator></fraction>" );
       break;
     case fraction:
       writeData( currentBuffer, "<fraction><nominator>" );
       writeNumber( currentBuffer, amountNumbers[0] );
       writeData( currentBuffer, "</nominator><denominator>" );
       writeNumber( currentBuffer, amountNumbers[1] );
       writeData( currentBuffer, "</denominator></fraction>" );
       break;
     case composite:
       writeData( currentBuffer, "<fraction><nominator>" );
       writeNumber( currentBuffer, amountNumbers[0] * amountNumbers[2] +
                                 amountNumbers[1] );
       writeData( currentBuffer, "</nominator><denominator>" );
       writeNumber( currentBuffer, amountNumbers[2] );
       writeData( currentBuffer, "</denominator></fraction>" );
       break;
     case floating:
       writeData( currentBuffer, "<float>" );
       writeDouble( currentBuffer, amountDouble );
       writeData( currentBuffer, "</float>" );
       break;
     default:
       assert( false );
     };
     writeData( currentBuffer, "</amount>\n" );
    };
   if ( yytext[0] != ' ' ) {
     writeData( currentBuffer, "        <unit>" );
     writeUnit( currentBuffer, yytext, yytext[1] == ' ' ? 1 : 2 );
     writeData( currentBuffer, "</unit>\n" );
   };
   writeData( currentBuffer, "        <name>" );
   /* Ignore space-character. */
   char c = userinput();
   assert( c == ' ' );

   assert( currentBuffer >= 1 && currentBuffer <= 2 );
   ingredientSpaces[ currentBuffer - 1 ] = 0;

   ingredientColumn = 0;
   BEGIN( ingredientEnd );
   yy_push_state( ingredient28 );
}
<unit>. {
  /* Interpretation as ingredient-line failed. */
  userunput( *yytext );
  proceedInstructions();
}

<ingredientEnd>\ {11}\- {
  /* Continuation of last line. */
  if ( ingredientOpen[ currentBuffer - 1 ] > 0 ) {
    flushIngredientSpaces();
    ingredientColumn = 1;
    yy_push_state( ingredient28 );
  } else if ( currentBuffer == 2 ) {
    currentBuffer = continuationBuffer;
    ingredientColumn = 1;
    yy_push_state( ingredient28 );
  } else
    ignoreRecipe( "Found stray continuation of ingredient-line." );
}
<ingredientEnd>. {
  userunput( *yytext );
  BEGIN( ingredients );
}

<instructions>(MMMMM|-----)-*/[\r\n] {
  closeRecipe();
  return 1;
};
<instructions>(MMMMM|-----)/[^\n\r] {
  closeSection();
  yy_push_state( section );
}
<instructions>[\ \t]*\r?\n {
  closeParagraph();
  lineNo++;
}
<instructions>[^\n:] {
  if ( !paragraphOpen )
    openParagraph();
  else
    writeData( 0, " " );
  userunput( *yytext );
  yy_push_state( line );
  yy_push_state( skipws );
}
<instructions>: {
  closeParagraph();
  openParagraph();
  yy_push_state( line );
}

<unicode0>[\300-\337]/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  if ( ( unicode & 0x1E ) == 0 )
    BEGIN(unicodeError);
  else
    BEGIN(unicode2);
}
<unicode0>\340/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode3Check);
}
<unicode0>[\341-\357]/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode3);
}
<unicode0>\360/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode4Check);
}
<unicode0>[\361-\367]/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode4);
}
<unicode0>\370/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode5Check);
}
<unicode0>[\371-\373]/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode5);
}
<unicode0>\374/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode6Check);
}
<unicode0>\375/[\200-\277] {
  unicodeLength = 0;
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = *(unsigned char *)yytext & 0x1F;
  BEGIN(unicode6);
}

<unicode0>[\300-\337\340-\357\360-\367\370-\373\374-\375]/[^\200-\277] {
  yy_pop_state();
  switch ( YY_START ) {
  case ignoreUnicode:
    yy_pop_state();
    break;
  case printUnicode:
  case titleUnicode:
    yy_pop_state();
  default:
    ignoreRecipe( "Erroneous character." );
  };
}

<unicode0>[^\300-\337\340-\357\360-\367\370-\373\374-\375] {
  yy_pop_state();
  switch ( YY_START ) {
  case ignoreUnicode:
    yy_pop_state();
    break;
  case printUnicode:
  case titleUnicode:
    yy_pop_state();
  default:
    ignoreRecipe( "Erroneous character." );
  };
}

<unicode2>[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicodeSequence[ unicodeLength ] = '\0';
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  yy_pop_state();
  if ( ( unicode >= 0xD800 && unicode <= 0xDFFF ) ||
       ( unicode == 0xFFFE ) || ( unicode == 0xFFFF ) ) {
    switch ( YY_START ) {
    case ignoreUnicode:
      yy_pop_state();
      break;
    case printUnicode:
    case titleUnicode:
      yy_pop_state();
    default:
      ignoreRecipe( "Forbidden UTF-8 character." );
    };
  } else {
    switch ( YY_START ) {
    case ignoreUnicode:
      yy_pop_state();
      break;
    case printUnicode:
      yy_pop_state();
      writeData( currentBuffer, unicodeSequence, unicodeLength );
      break;
    case titleUnicode:
      yy_pop_state();
      if ( titleSize + unicodeLength <= (signed)sizeof(titleText) ) {
        memcpy( &titleText[ titleSize ], unicodeSequence, unicodeLength );
        titleSize += unicodeLength;
      } else
        ignoreRecipe( "Section-title to long." );
    };
  };
}
<unicode3>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  BEGIN(unicode2);
}
<unicode3Check>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  if ( ( unicode & 0x20 ) == 0 )
    BEGIN(unicodeError);
  else
    BEGIN(unicode2);
}
<unicode4>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  BEGIN(unicode3);
}
<unicode4Check>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  if ( ( unicode & 0x30 ) == 0 )
    BEGIN(unicodeError);
  else
    BEGIN(unicode3);
}
<unicode5>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  BEGIN(unicode4);
}
<unicode5Check>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  if ( ( unicode & 0x38 ) == 0 )
    BEGIN(unicodeError);
  else
    BEGIN(unicode4);
}
<unicode6>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  BEGIN(unicode5);
}
<unicode6Check>[\200-\277]/[\200-\277] {
  assert( unicodeLength + 1 < (signed)sizeof( unicodeSequence ) );
  unicodeSequence[ unicodeLength++ ] = *yytext;
  unicode = ( unicode << 6 ) | ( *(unsigned char *)yytext & 0x3F );
  if ( ( unicode & 0x3C ) == 0 )
    BEGIN(unicodeError);
  else
    BEGIN(unicode5);
}

<unicode3,unicode3Check,unicode4,unicode4Check,unicode5,unicode5Check,unicode6,unicode6Check>[\200-\277]/[^\200-\277] {
  yy_pop_state();
  switch ( YY_START ) {
  case ignoreUnicode:
    yy_pop_state();
    break;
  case printUnicode:
  case titleUnicode:
    yy_pop_state();
  default:
    ignoreRecipe( "Erroneous Unicode-character." );
  };
}

<unicodeError>[\200-\277]/[\200-\277] 

<unicodeError>[\200-\277]/[^\200-\277] {
  yy_pop_state();
  switch ( YY_START ) {
  case ignoreUnicode:
    yy_pop_state();
    break;
  case printUnicode:
  case titleUnicode:
    yy_pop_state();
  default:
    ignoreRecipe( "Erroneous character." );
  };
}

<*>. {
#ifndef NDEBUG
   fprintf( stderr, "Problem in state %d with character %c (0x%x) while "
            "parsing line %d\n", YY_START, *yytext, (int)*yytext, lineNo );
#endif
   assert( false );
}

%%

void unputBuffer(void)
{
  for ( int i=bufferLength-1; i>=0; i-- )
    userunput( buffer[ i ] );
}

void reportError( const char *_errorMessage )
{
  errorMessage = _errorMessage;
  errorLine = lineNo;
  while( yy_start_stack_ptr > 0 )
    yy_pop_state();
  BEGIN(errorState);
}

void ignoreRecipe( const char *_errorMessage )
{
  reportError( _errorMessage );
  yy_push_state( ignoreline );
}

void proceedInstructions(void)
{
  moveDataX();
  unputBuffer();
  closeSection();
  writeData( 0, "   </ingredients>\n   <instructions>\n" );
  BEGIN( instructions );
}

void closeRecipe(void)
{
  recipeEndPos = position;
  recipeEndLine = lineNo;
#ifndef NDEBUG
  while( yy_start_stack_ptr > 0 ) {
    yy_pop_state();
    fprintf( stderr, "Error: state %d still was on the stack.\n", YY_START );
  };
#endif
  closeSection();
  writeData( 0, "  </instructions>\n</recipe>\n" );
  BEGIN( INITIAL );
}

void addIngredientColumn(void)
{
  ingredientColumn++;
  if ( ingredientColumn >= 28 )
    BEGIN( ingredient28End );
}
