// ---------------------------------------------------------------------------
// - Integer.hpp                                                             -
// - standard object library - integer class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_INTEGER_HPP
#define  ALEPH_INTEGER_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

namespace aleph {

  /// The Integer class is the object version of the native integer
  /// operation for the aleph subsystem. All operations are made with at least
  /// a 64 bits representation (i.e long long). Standard constructors and
  /// operators are provided. The Integer class is also a literal class, so
  /// the toLiteral and repr method are implemented virtually.
  /// @author amaury darsch

  class Integer : public Literal {
  private:
    /// the integer representation
    t_long d_value;

  public:
    /// create a new default integer
    Integer (void);

    /// create a new integer from a long long integer
    /// @param value the value to create
    Integer (const t_long value);

    /// create a new integer from a string
    /// @param value the value to convert
    Integer (const String& value);

    /// copy constructor for this integer
    /// @param that the integer class to copy
    Integer (const Integer& that);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this integer
    String toliteral (void) const;

    /// @return a string representation of this integer
    String tostring (void) const;

    /// @return a clone of this integer
    Object* clone (void) const;

    /// @return the integer serial code
    t_byte serialid (void) const;

    /// serialize this integer to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize an integer from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// assign an integer with a native value
    /// @param value the value to assign
    Integer& operator = (const long value);

    /// Assign an integer with a native value
    /// @param value the value to assign
    Integer& operator = (const Integer& value);

    /// compare this integer with a native value
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const long value) const;

    /// compare this integer with a native value
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const long value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are equals
    bool operator == (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are not equals
    bool operator != (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are less
    bool operator < (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are less or equal
    bool operator <= (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are greater
    bool operator > (const Integer& value) const;

    /// compare two integers
    /// @param value the value to compare
    /// @return true if they are greater or equal
    bool operator >= (const Integer& value) const;

    /// compute the opposite of the integer
    /// @param x the integer to oppose
    /// @return a new integer opposite of the argument
    friend Integer operator - (const Integer& x);
    
    /// add two integers together
    /// @param x the first argument to add
    /// @param y the second argument to add
    /// @return a new integer sum of the previous one
    friend Integer operator + (const Integer& x, const Integer& y);

    /// substract two integers together
    /// @param x the first argument to substract
    /// @param y the second argument to substract
    /// @return a new integer difference of the previous one
    friend Integer operator - (const Integer& x, const Integer& y);

    /// multiply two integers together
    /// @param x the first argument to multiply
    /// @param y the second argument to multiply
    /// @return a new integer product of the previous one
    friend Integer operator * (const Integer& x, const Integer& y);

    /// divide two integers together
    /// @param x the numerator
    /// @param y the denumerator
    /// @return the division of the arguments  
    friend Integer operator / (const Integer& x, const Integer& y);

    /// get the modulo between two integers
    /// @param x the integer value
    /// @param y the modulus
    /// @return the division of the arguments  
    friend Integer operator % (const Integer& x, const Integer& y);

    /// add an integer to this one
    /// @param x the argument to add
    /// @return this added integer
    Integer& operator += (const Integer& x);

    /// substract an integer to this one
    /// @param x the argument to substract
    /// @return this added integer
    Integer& operator -= (const Integer& x);

    /// multiply an integer with this one
    /// @param x the argument to multiply
    /// @return this multiplied integer
    Integer& operator *= (const Integer& x);

    /// divide an integer with this one
    /// @param x the argument to divide
    /// @return this multiplied integer
    Integer& operator /= (const Integer& x);

    /// @return the absolute value of this number
    Integer abs (void) const;

    /// @return a native integer representation
    t_long tointeger (void) const;

    /// convert a string to an integer
    /// @param value the string to convert
    /// @return a native integer representation
    static t_long tointeger (const String& value);

    /// @return a random number
    static t_long random (void);

    /// evaluate an object to an integer value
    /// @param robj the current runnable
    /// @param nset the current nameset
    /// @param object the object to evaluate
    static t_long evalto (Runnable* robj, Nameset* nset, Object* object);

    /// generate a new integer
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this integer with another object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// set an object to this integer
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// apply this integer with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the real class a friend
    friend class Real;
  public:
    // the memory allocation
    void* operator new    (const t_size size);
    void  operator delete (void* handle);
  };
}

#endif
