// ---------------------------------------------------------------------------
// - Nameset.cpp                                                            -
// - aleph engine - nameset class implementation                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Symbol.hpp"
#include "Nameset.hpp"
#include "Exception.hpp"

namespace aleph {

  // the parent nameset as a quark
  static const long QUARK_PARENT = String::intern ("..");

  // create a default nameset

  Nameset::Nameset (void) {
    p_parent = nilp;
  }

  // destroy this nameset

  Nameset::~Nameset (void) {
    Object::dref (p_parent);
  }

  // set the parent nameset 

  void Nameset::setparent (Nameset* nset) {
    if (p_parent != nilp) {
      remove (QUARK_PARENT);
      Object::dref (p_parent);
    }
    Object::iref (nset);
    p_parent = nset;
    if (nset != nilp) symcst (QUARK_PARENT, nset);
  }

  // add a new object by name

  void Nameset::bind (const String& name, Object* object) {
    bind (name.toquark (), object);
  }

  // return true if the name exists in this nameset
  bool Nameset::exists (const String& name) const {
    return exists (name.toquark ());
  }

  // return an object by name but do not evaluate

  Object* Nameset::find (const String& name) {
    return find (name.toquark ());
  }

  // remove an object by name

  void Nameset::remove (const String& name) {
    remove (name.toquark ());
  }

  // create a new const symbol by quark

  void Nameset::symcst (const long quark, Object* object) {
    Symbol* sym = new Symbol (quark, object);
    sym->setconst (true);
    bind (quark, sym);
  }

  // create a new const symbol by name

  void Nameset::symcst (const String& name, Object* object) {
    Symbol* sym = new Symbol (name, object);
    sym->setconst (true);
    bind (name, sym);
  }

  // create a new symbol by quark

  void Nameset::symdef (const long quark, Object* object) {
    Symbol* sym = new Symbol (quark, object);
    bind (quark, sym);
  }

  // create a new symbol by name

  void Nameset::symdef (const String& name, Object* object) {
    Symbol* sym = new Symbol (name, object);
    bind (name, sym);
  }
}
