// ---------------------------------------------------------------------------
// - Main.cpp                                                                -
// - the aleph file librarian main program                                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "System.hpp"
#include "Axlopts.hpp"
#include "Librarian.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"
#include "OutputTerm.hpp"

// this procedure does a librarian report
static void axl_report (const aleph::Librarian& axl, bool info, bool list) {
  using namespace aleph;
  if (info == false) return;
  OutputTerm tout;
  if (list == true) {
    Strvec flist = axl.getlist ();
    long len     = flist.length ();
    for (long i = 0; i < len; i++) tout.writeln (flist.get (i));
  } else {
    axl.format (tout);
  }
}

// this procedure extract a file from the librarian
static void axl_extract (const aleph::Librarian& axl, 
			 const aleph::String& name) {
  using namespace aleph;
  InputMapped* im = axl.extract (name);
  if (im == nilp) return;
  OutputFile os (name);
  while (im->iseof () == false) os.write (im->read ());
}

// main interpreter 

int main (int argc, char** argv) {
  using namespace aleph;

  // create a new option instance
  Axlopts* options = new Axlopts (argc,argv);

  // get the librarian name
  String lname = options->getlname ();

  // get the path vector
  const Strvec& paths = options->getpath ();
  long  plen = paths.length ();

  // get the librarian options
  bool cflg = options->getcflg ();
  bool xflg = options->getxflg ();
  bool info = options->getinfo ();
  bool list = options->getlist ();

  try {
    if (cflg == true) {
      // create the librarian
      Librarian axl;
      for (long i = 0; i < plen; i++) axl.add (paths.get (i));
      axl.write (lname);
      // check for report
      axl_report (axl, info, list);
    }
    if (xflg == true) {
      Librarian axl (lname);
      // check if we extract eveything
      if (plen == 0) {
	Strvec flist = axl.getlist  ();
	long   flen  = flist.length ();
	for (long i = 0; i < flen; i++) {
	  String name  = flist.get(i);
	  axl_extract (axl, name);
	}
      } else {
	for (long i = 0; i < plen; i++) {
	  String name = paths.get (i);
	  axl_extract (axl, name);
	}
      }
      // check for report
      axl_report (axl, info, list);
    } else {
       Librarian axl (lname);
       // check for report
       axl_report (axl, info, list);
    }    
  } catch (const Exception& e) {
    OutputTerm terr (OutputTerm::ERROR);
    terr << "error: " << e.getval () << " with librarian " << lname << eolc;
    System::exit (1);
  } catch (...) {
    OutputTerm terr;
    terr << "fatal: internal error\n";
    System::exit (1);
  }
  
  // clean eveything and return
  delete options;
  return 0;
}
