// ---------------------------------------------------------------------------
// - System.hpp                                                              -
// - standard object library - system function class definition              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_SYSTEM_HPP
#define  ALEPH_SYSTEM_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The System class is a collection of static convenient functions. There
  /// purposes is to provide a uniform interface for os dependant features.
  /// @author amaury darsch

  class System {
  public:
    /// register the progran name
    /// @param name the name to register
    static void setpgm (const String& name);

    /// @return the program name
    static String getpgm (void);

    /// @return the host name
    static String hostname (void);

    /// @return the user name
    static String username (void);

    /// @return the aleph url
    static String geturl (void);

    /// @return the os name
    static String osname (void);

    /// @return the os type
    static String ostype (void);
    
    /// @return the major version of the aleph system
    static String major (void);    
 
   /// @return the minor version of the aleph system
    static String minor (void);    

    /// @return the patch version of the aleph system
    static String patch (void);    

    /// @return the full version of the aleph system
    static String version (void);

   /// @return the option separator
    static char optsep (void);

    /// @return an environment variable value
    static String getenv (const String& name);

    /// exit unconditionnaly with a status code
    /// @param status the exit status code
    static void exit (int status);

    /// @return the process id
    static long getpid (void);

    /// pause for a certain time 
    /// @param time the time to pause in milliseconds
    static void sleep (const long time);

    /// swap a word betwen host and network order
    /// @param value the value to swap
    static t_word wswap (const t_word value);

    /// swap a quad betwen host and network order
    /// @param value the value to swap
    static t_quad qswap (const t_quad value);

    /// swap an octa betwen host and network order
    /// @param value the value to swap
    static t_octa oswap (const t_octa value);

    /// remove a file name extension
    /// @param name the file name
    /// @return a file name without its extension
    static String rmext (const String& name);

    /// concatenate two names to form a new os path name
    /// @param dir the directory name
    /// @param name the name to concatenate
    /// @return the new path name
    static String join (const String& dir, const String& name);

    /// extract the file name from a path
    /// @param path the path to process
    static String xname (const String& path);
 
   /// extract the file name extension
    /// @param name the name to process
    static String xext (const String& name);

    /// @return true if the name corresponds to a regular file name
    static bool isfile (const String& name);

    /// @return true if the name corresponds to a directory name
    static bool isdir (const String& name);

    /// @remove a file by name
    static bool rmfile (const String& name);

    /// @remove a directory by name
    static bool rmdir (const String& name);
  };
}

#endif
