// ---------------------------------------------------------------------------
// - List.cpp                                                                -
// - standard object library - doubly linked list class implementation       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "List.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {

  // the supported list quarks
  static const long QUARK_GET    = String::intern ("get");
  static const long QUARK_GETIT  = String::intern ("get-iterator");
  static const long QUARK_LENGTH = String::intern ("length");
  static const long QUARK_APPEND = String::intern ("append");
  static const long QUARK_INSERT = String::intern ("insert");

  // the private list structure
  struct s_list {
    // the object to store
    Object* p_object;
    // the previous element
    s_list* p_prev;
    // the next element;
    s_list* p_next;
    // simple constructor
    s_list (void) {
      p_object = nilp;
      p_prev   = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_list (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };

  // -------------------------------------------------------------------------
  // - list class section                                                    -
  // -------------------------------------------------------------------------

  // create an empty list

  List::List (void) {
    p_root = nilp;
    p_last = nilp;
  }

  // copy constructor for this list

  List::List (const List& that) {
    p_root = nilp;
    p_last = nilp;
    s_list* node = that.p_root;
    while (node != nilp) {
      append (node->p_object);
      node = node->p_next;
    }
  }
    
  // destroy this list

  List::~List (void) {
    delete p_root;
  }

  // return the class name
  
  String List::repr (void) const {
    return "List";
  }

  // assign a list to this one

  List& List::operator = (const List& that) {
    if (this == &that) return *this;
    delete p_root;
    p_root = nilp;
    p_last = nilp;
    s_list* node = that.p_root;
    while (node != nilp) {
      append (node->p_object);
      node = node->p_next;
    }
    return *this;
  }

  // make this list a shared object

  void List::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    s_list* node = p_root;
    while (node != nilp) {
      if (node->p_object != nilp) node->p_object->mksho ();
      node = node->p_next;
    }
  }

  // insert an object at the beginning of the list

  void List::insert (Object* object) {
    wrlock ();
    try {
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();
      // insert in the list
      s_list* node   = new s_list;
      node->p_object = Object::iref (object);
      node->p_next   = p_root;
      if (p_root == nilp) {
	p_root = node;
	p_last = node;
	unlock ();
	return;
      }
      p_root->p_prev = node;
      p_root = node;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // append an object to the end of the list

  void List::append (Object* object) {
    wrlock ();
    try {
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();
      // append in the list
      s_list* node   = new s_list;
      node->p_object = Object::iref (object);
      if (p_root == nilp) {
	p_root = node;
	p_last = node;
	unlock ();
	return;
      }
      p_last->p_next = node;
      node->p_prev = p_last;
      p_last = node;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the number of elements in the list

  long List::length (void) const {
    rdlock ();
    try {
      s_list* node = p_root;
      long result  = 0;
      while (node != nilp) {
	result++;
	node = node->p_next;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return an object by index

  Object* List::get (const long index) const {
    rdlock ();
    try {
      long count   = 0;
      s_list* node = p_root;
      if (index < 0) {
	unlock ();
	throw Exception ("index-error", "invalid negative index in list get");
      }
      // loop in the list
      while (node != nilp) {
	if (count == index) {
	  Object* result = node->p_object;
	  unlock ();
	  return result;
	}
	count++;
	node = node->p_next;
      }
      unlock ();
      throw Exception ("index-error", "invalid index in list get method");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a new iterator for this list
  
  Iterator* List::makeit (void) {
    return new Listit (this);
  }

  // create a new list in a generic way

  Object* List::mknew (Vector* argv) {
    long len = 0;
    if ((argv == nilp) || ((len = argv->length ()) == 0)) return new List;
    // build the list
    List* result = new List;
    for (long i = 0; i < len; i++)
      result->append (argv->get (i));
    return result;
  }

  // apply a list with a set of arguments by quark

  Object* List::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH) return new Integer (length ());
      if (quark == QUARK_GETIT)  return makeit ();
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_APPEND) {
	Object* result = argv->get (0);
	append (result);
	robj->post (result);
	return result;
      }

      if (quark == QUARK_INSERT) {
	Object* result = argv->get (0);
	insert (result);
	robj->post (result);
	return result;
      }

      if (quark == QUARK_GET) {
	rdlock ();
	try {
	  long val = argv->getint (0);
	  Object* result = get (val);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - list iterator class section                                           -
  // -------------------------------------------------------------------------

  // create a new list iterator

  Listit::Listit (List* lst) {
    p_list = lst;
    Object::iref (lst);
    p_node = nilp;
    begin ();
  }

  // destroy this list iterator

  Listit::~Listit (void) {
    Object::dref (p_list);
  }

  // return the class name

  String Listit::repr (void) const {
    return "Listit";
  }

  // make this list iterator a shared object

  void Listit::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_list != nilp) p_list->mksho ();
  }

  // reset the iterator to the begining

  void Listit::begin (void) {
    p_node = p_list->p_root;
  }

  // reset the iterator to the end

  void Listit::end (void) {
    p_node = p_list->p_last;
  }

  // go to the next object

  void Listit::next (void) {
    if (p_node == nilp) return;
    p_node = p_node->p_next;
  }

  // go to the previous object
  void Listit::prev (void) {
    if (p_node == nilp) return;
    if (p_node->p_prev == nilp) return;
    p_node = p_node->p_prev;
  }

  // get the object at the current position

  Object* Listit::getobj (void) {
    if (p_node == nilp) return nilp;
    return p_node->p_object;
  }

  // return true if the iterator is at the end
  
  bool Listit::isend (void) {
    if (p_node == nilp) return true;
    return false;
  }
}
