// ---------------------------------------------------------------------------
// - Input.hpp                                                               -
// - standard object library - input stream base class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_INPUT_HPP
#define  ALEPH_INPUT_HPP

#ifndef  ALEPH_BUFFER_HPP
#include "Buffer.hpp"
#endif

namespace aleph {

  /// The input stream class is a base class which provides a read method for
  /// character in a generic way. Various classes are derived from it, like 
  /// the InputTerm or InputFile. The class implements a buffer which provides
  /// a generic pushback method. When the read method is invoked, the character
  /// is placed in the buffer.
  /// @author amaury darsch

  class Input : public virtual Object {
  protected:
    /// the local buffer
    mutable Buffer d_buffer;

  public:
    /// @return the stream descriptor
    virtual int getsid (void) const;

    /// @return the next available character
    virtual char read (void) =0;

    /// @return a buffer of character
    /// @param size the number of character to read
    virtual Buffer* read (const long size);

    /// @return the next available line
    virtual String readln (void);

    /// @return true if we are at the eof
    virtual bool iseof (void) const =0;

    /// check if we can read a character
    /// @param tout the timeout value
    virtual bool valid (const long tout) const =0;

    /// push back a character in the input stream.
    /// @param value the character to push back
    virtual void pushback (const char value);

    /// pushback a string on this input stream
    virtual void pushback (const String& value);

    /// pushback a character buffer by size
    /// @param buffer the buffer to push back
    /// @param size   the buffer size
    virtual void pushback (const char* buffer, const long size);

    /// @return the size of the input buffer
    virtual long buflen (void) const;

    /// apply this input with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments  to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
