// ---------------------------------------------------------------------------
// - HtmlPage.hpp                                                            -
// - aleph:www library - html page class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_HTMLPAGE_HPP
#define  ALEPH_HTMLPAGE_HPP

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

#ifndef  ALEPH_BUFFER_HPP
#include "Buffer.hpp"
#endif

#ifndef  ALEPH_COOKIE_HPP
#include "Cookie.hpp"
#endif

namespace aleph {

  /// The HtmlPage is a generic class used to produce dynamic html pages.
  /// The class is constructed with two buffers. The head buffer stores
  /// the html head content. The body buffer stores the html body content.
  /// When the page is ready, th "writehead", "writebody" or "writepage"
  /// methods can be used to write on the specified output stream. Various
  /// methods are provided to query the content length or to retrieve the
  /// page content as a buffer or string.
  /// @author amaury darsch

  class HtmlPage : public Object {
  private:
    /// the http header
    Buffer d_http;
    /// the head buffer
    Buffer d_head;
    /// the body buffer
    Buffer d_body;

  public:
    /// create an empty html page
    HtmlPage (void);

    /// @return the class name
    String repr (void) const;

    /// add the page title
    /// @param title the title to add
    void addtitle (const String& title);

    /// add the page author
    /// @param author the author to add
    void addauthor (const String& author);

    /// add a page style link
    /// @param style the style to set
    void addstyle (const String& style);

    /// add a meta markup by name and content
    /// @param name the meta name to add
    /// @param content the meta content to add
    void addmeta (const String& name, const String& content);

    /// add a cookie to the http headedr
    /// @param cookie the cookie to add
    void addcookie (Cookie* cookie);

    /// add some content to the http header
    /// @param content the content to add
    void addhttp (const String& content);

    /// add some content to the head
    /// @param content the content to add
    void addhead (const String& content);

    /// add some content to the body
    /// @param content the content to add
    void addbody (const String& content);

    /// write the html http header to the output stream
    /// @param os the output stream
    void writehttp (Output& os) const;

    /// write the html page head to the output stream
    /// @param os the output stream
    void writehead (Output& os) const;

    /// write the html page body to the output stream
    /// @param os the output stream
    void writebody (Output& os) const;

    /// write the html page to the output stream
    /// @param os the output stream
    void writepage (Output& os) const;

    /// write the page as a result of a cgi request
    void writecgi (Output& os) const;

    /// @return a buffer containing the html page
    Buffer* getbuffer (void) const;

    /// generate a new html page object
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this html page with a set of arguments and a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);

  private:
    // make the copy constructor private
    HtmlPage (const HtmlPage&);
    // make the assignment operator private
    HtmlPage& operator = (const HtmlPage&);
  };
}

#endif
