// ---------------------------------------------------------------------------
// - Class.hpp                                                               -
// - aleph engine - class class definition                                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_CLASS_HPP
#define  ALEPH_CLASS_HPP

#ifndef  ALEPH_LOCALSET_HPP
#include "Localset.hpp"
#endif

#ifndef  ALEPH_QARRAY_HPP
#include "Qarray.hpp"
#endif

namespace aleph {

  /// The Class class is the class used to model aleph class object. The aleph
  /// model is based on a single inheritance mechanism. A class binds only
  /// a local set - which somehow is equivalent to c++ static data member.
  /// When an instance of that class is created, it is the instance initial
  /// method which is responsible to define the local data members.
  /// @author amaury darsch

  class Class : public Object {
  private:
    /// the class local set
    Localset* p_cset;
    /// the list of data members
    Qarray d_mdata;

  public:
    /// create a new class 
    Class (void);

    /// destroy this class class
    ~Class (void);

    /// @return the class name
    String repr (void) const;

    /// make this class a shared object
    void mksho (void);

    /// add a new data member by quark
    /// @param quark the quark to add
    void madd (const long quark);

    /// @return the array of data members
    const Qarray& getmdata (void) const {
      return d_mdata;
    }

    /// @return an object in the class localset but do not evaluate
    Object* find (const long quark) const {
      return p_cset->find (quark);
    }

    /// operate this class with another object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// set an object as a const static object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to define as const
    /// @param object the object to set
    Object* cdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// set an object as a static object by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the quark to define as trans
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, const long quark, 
		  Object* object);

    /// evaluate a static member by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param quark  the member quark to evaluate
    Object* eval (Runnable* robj, Nameset* nset, const long quark);

    /// create a new instance with some arguments
    /// @param robj   the current runnable
    /// @param nset   the current nameset 
    /// @param args   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, Cons* args);

    /// apply this class with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param args   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Cons* args);

  private:
    // make the copy constructor private
    Class (const Class&);
    // make the assignment operator private
    Class& operator = (const Class&);
  };
}

#endif
