//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qpixmap.h>
#include <qimage.h>
#include <qpainter.h>
#include <qcursor.h>
#include <qapplication.h>

//#include <qscrollbar.h>

//Projectwide includes
#include "subalbumsIconView.h"
#include "subalbumPreviewWidget.h"
#include "layoutWidget.h"
#include "subalbumWidget.h"
#include "photoPreviewWidget.h"

//==============================================
SubalbumsIconView::SubalbumsIconView( QWidget *parent ) : QIconView( parent )
{
//  setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum);
  setMouseTracking(true);

  currentPseudoSelection = NULL;

  //connect mouse over events to paint pseudo selection in ligher blue
  connect( this, SIGNAL(onItem(QIconViewItem*)),
                this, SLOT(repaintGroup(QIconViewItem*)) );

  //clear any pseudo selection when mouse moves off icons
  connect( this, SIGNAL(onViewport()),
                this, SLOT(clearPseudoSelection()) );

  //compute textWidth for collection names using a calibration string
  QString calibrationString( qApp->translate("SubalbumPreviewWidget", "Calibration String") );
  QFontMetrics fm( qApp->font() );
  textWidth = fm.width( calibrationString );
}
//==============================================
void SubalbumsIconView::contentsDropEvent( QDropEvent *e )
{
  QIconView::contentsDropEvent( e );

   //if drop originated from this viewport then emit item moved signal
  if(e->source() == viewport() )
    emit itemHasMoved();
}
//==============================================
 void SubalbumsIconView::drawContents ( QPainter * p, int clipx, int clipy, int clipw, int cliph )
{
    if( bufferPixmap.size() != size())
    {  bufferPixmap.resize( size() ); }
    QPainter bufferPainter( &bufferPixmap, viewport() );
    int xOffset = clipx - contentsX();
    int yOffset = clipy - contentsY();

    bufferPainter.translate( -contentsX(), -contentsY() );
    QIconView::drawContents( &bufferPainter, clipx, clipy, clipw, cliph );
    bitBlt(p->device(), xOffset, yOffset, &bufferPixmap, xOffset, yOffset, clipw, cliph );
}
//==============================================
void SubalbumsIconView::contentsMousePressEvent ( QMouseEvent * e )
{
  //ignore all clicks other than left-clicks
  if( e->button() != Qt::LeftButton ) return;

  dragStartPos = e->pos();
  QIconView::contentsMousePressEvent( e );
}
//==============================================
QDragObject* SubalbumsIconView::dragObject()
{
  //no item selected?
  if( !currentItem() )
    return 0;

  //create drag object
  QIconDrag *drag = new QIconDrag( viewport() );

  //create buffer and paint item to buffer
  QRect r = currentItem()->rect();
  QPixmap buffer( r.width(), r.height() );
  QPainter painter( &buffer, this );
  painter.translate( -r.x(), -r.y() );
  ((SubalbumPreviewWidget*)currentItem())->paint( &painter );

  //clip off background color around edges which was used for anti-aliasing edges.
  //result image will have semi-selection oval around it.
  QBitmap bit = buffer.createHeuristicMask();
  buffer.setMask( bit );

  //set pixmap to use buffer
  drag->setPixmap( buffer, QPoint( dragStartPos.x() - r.x(), dragStartPos.y() - r.y() ) );

  //we don't want to show any rectangles, but if we don't append two null rectangles the last drag rectangles this iconview displayed
  //possibly form objects dropped onto it from outside the viewport, aka photos, will be drawn! :(
  drag->append( QIconDragItem(), QRect(), QRect() );

  return drag;
}
//==============================================
void SubalbumsIconView::contentsDragMoveEvent( QDragMoveEvent* e )
{
  QIconView::contentsDragMoveEvent( e );
   e->accept(true);

   //if source of drag is not from application then return
   if(e->source() == NULL)
     return;

  //if source of drag is from within this view then return, in the future we'll put
  //drag code in here to drag indicators for rearranging the items of the iconview
  if(e->source() == viewport() )
  {

  }
  //else if source is from photos iconview
  else if(e->source()->parentWidget() == ((LayoutWidget*)parentWidget()->parentWidget())->getSubalbum()->getPhotos() )
  {
    SubalbumPreviewWidget* item = (SubalbumPreviewWidget*)findItem( e->pos() );

    //if item pointer same as current pseudo selection ignore
    if(item == currentPseudoSelection)
    {
      return;
    }

    //unpaint old selection
    if(currentPseudoSelection != NULL)
    {
      currentPseudoSelection->setMousedOver(false);
      repaintItem(currentPseudoSelection);
    }

    //set new selection
    currentPseudoSelection = item;

    //repaint new selection
    if(currentPseudoSelection != NULL)
    {
      currentPseudoSelection->setMousedOver(true);
      repaintItem(currentPseudoSelection);
    }
  }
}
//==============================================
void SubalbumsIconView::repaintGroup( QIconViewItem* pseudoSelection)
{
  //if old pseudo selection unselect it
  clearPseudoSelection();

  //paint new selection
  currentPseudoSelection = (SubalbumPreviewWidget*)pseudoSelection;
  currentPseudoSelection->setMousedOver(true);
  repaintItem(currentPseudoSelection);
}
//==============================================
void SubalbumsIconView::clearPseudoSelection()
{
  //if old pseudo selection unselect it
  if(currentPseudoSelection != NULL)
  {
    currentPseudoSelection->setMousedOver(false);
    repaintItem(currentPseudoSelection);
    currentPseudoSelection = NULL;
  }
}
//==============================================
int SubalbumsIconView::getTextWidth()
{ return textWidth; }
//==============================================
QSize SubalbumsIconView::minimumSizeHint() const { return sizeHint(); }
//==============================================
QSize SubalbumsIconView::sizeHint() const
{
  QSize s = QIconView::sizeHint();
  
  //find max item width
  s.setWidth(0);
  QIconViewItem *item;
  for( item = firstItem(); item != NULL; item = item->nextItem() )
  {
    if(item->width() > s.width() )
      s.setWidth( item->width() );
  } 
  s.setWidth( s.width() + 2*spacing() + verticalScrollBar()->sizeHint().width() );
  return s;
}
//==============================================


