//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qlayout.h>
#include <qlabel.h>
#include <qfont.h>
#include <qcheckbox.h>
#include <qframe.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qvgroupbox.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qapplication.h>
#include <qtooltip.h>

//Projectwide includes
#include "layoutSettingsWidget.h"
#include "configuration.h"
#include "../config.h"
#include "../gui/window.h"
#include "../gui/titleWidget.h"

//==============================================
LayoutSettingsWidget::LayoutSettingsWidget( Configuration* config,
                                            QWidget* parent,
                                            const char* name ) : QWidget( parent, name)
{
  this->config = config;

  categoryLabel = new QLabel( tr("Appearance:"), this);
  QFont labelFont = categoryLabel->font();
  labelFont.setWeight(QFont::Bold);
  categoryLabel->setFont( labelFont );

  horizontalLine = new QFrame(this);
  horizontalLine->setLineWidth(2);
  horizontalLine->setMidLineWidth(1);
  horizontalLine->setFrameStyle( QFrame::HLine | QFrame::Raised );
  //----------------
  //General Settings:
  //-display image animations
  //-display tooltips
  generalSettings = new QVGroupBox( tr("General"), this);
  useAnimation = new QCheckBox( tr("Use animation"), generalSettings);
  showTooltips = new QCheckBox( tr("Show tooltips"), generalSettings);
  //----------------
  //Window placement and size Settings:
  //-restore old placement and size
  //-preset default size (% of screen size) and placement
  windowPlacementSize = new QVGroupBox( tr("Window Placement and Size"), this);
  restoreWindowPlacementSize = new QCheckBox( tr("Restore window placement and size"), windowPlacementSize);

  sizeFrame = new QFrame(windowPlacementSize);
  defaultWindowSizeLabel = new QLabel( tr("% of Screen:"), sizeFrame);
  defaultWindowSizeSlider = new QSlider(Qt::Horizontal, sizeFrame);
  defaultWindowSizeSlider->setMinValue(1);
  defaultWindowSizeSlider->setMaxValue(100);
  defaultWindowSizeValue = new QSpinBox(1,100,1,sizeFrame);
  defaultWindowSizeValue->setSuffix("%");

  defaultWindowPlacementLabel = new QLabel( tr("Placement:"), sizeFrame);
  defaultWindowPlacement = new QComboBox( sizeFrame );
  defaultWindowPlacement->insertItem( tr("Center") );
  defaultWindowPlacement->insertItem( tr("Top Left") );
  defaultWindowPlacement->insertItem( tr("Top Right") );
  defaultWindowPlacement->insertItem( tr("Bottom Left") );
  defaultWindowPlacement->insertItem( tr("Bottom Right") );

  //update spinbox value when slider moves
  connect( defaultWindowSizeSlider, SIGNAL(valueChanged(int)),
                   this, SLOT(defaultSizeSliderMoved(int)) );;

  //update slider when spinbox changes
  connect( defaultWindowSizeValue, SIGNAL(valueChanged(int)),
                   this, SLOT(defaultSizeSpinboxChanged(int)) );;

  //disable manual window size/placement settings when auto save position/location is checked
  connect( restoreWindowPlacementSize, SIGNAL(toggled(bool)),
                   this, SLOT(toggleDefaultSizeEnabled(bool)) );;
  //----------------
  //place window placement/size control in box grid
  manualPlacementGrid = new QGridLayout( sizeFrame, 2, 3, 0);  
  manualPlacementGrid->setSpacing( WIDGET_SPACING );

  manualPlacementGrid->addWidget(defaultWindowSizeLabel,  0, 0);
  manualPlacementGrid->addWidget(defaultWindowSizeSlider, 0, 1);
  manualPlacementGrid->setColStretch(1, 1);
  manualPlacementGrid->addWidget(defaultWindowSizeValue,  0, 2);

  manualPlacementGrid->addWidget(defaultWindowPlacementLabel,     1, 0);
  manualPlacementGrid->addMultiCellWidget(defaultWindowPlacement, 1, 1, 1, 2, Qt::AlignLeft);
  //----------------
  //Setup larger boxes in overall grid
  mainGrid = new QGridLayout( this, 5, 1, 0);
  mainGrid->setSpacing( WIDGET_SPACING );
  
  mainGrid->addWidget( categoryLabel,       0, 0, Qt::AlignLeft );
  mainGrid->addWidget( horizontalLine,      1, 0 );
  mainGrid->addWidget( generalSettings,     2, 0 );
  mainGrid->addWidget( windowPlacementSize, 3, 0 );
  mainGrid->setRowStretch( 4, 1 );
}
//==============================================
void LayoutSettingsWidget::defaultSizeSliderMoved(int v)
{
  //update spinbox
  defaultWindowSizeValue->setValue( v );
}
//==============================================
void LayoutSettingsWidget::defaultSizeSpinboxChanged(int v)
{
  //update slider
  defaultWindowSizeSlider->setValue( v );
}
//==============================================
void LayoutSettingsWidget::toggleDefaultSizeEnabled(bool b)
{
  sizeFrame->setDisabled(b);
}
//==============================================
void LayoutSettingsWidget::setDefaults(Configuration* config)
{
  config->setBool( "layout", "animation", true );
  config->setBool( "layout", "showTooltips", true );
  config->setBool( "layout", "restoreWindowPlacementSize", true);
  //----
  QDesktopWidget *desktop = QApplication::desktop();
  int width = (8*desktop->width()) / 10;
  int height = (8*desktop->height()) / 10;
  config->setInt( "layout", "windowWidth", width );
  config->setInt( "layout", "windowHeight", height );
  config->setInt( "layout", "windowPosX", (desktop->width() - width) / 2 );
  config->setInt( "layout", "windowPosY", (desktop->height() - height) / 2 );
  //----
  config->setInt( "layout", "defaultWindowSize", 80 );
  config->setString( "layout", "defaultWindowPlacement", 0 );
}
//==============================================
void LayoutSettingsWidget::loadSettings()
{
  useAnimation->setChecked( config->getBool( "layout", "animation" ));
  showTooltips->setChecked( config->getBool( "layout", "showTooltips" ));
  restoreWindowPlacementSize->setChecked( config->getBool( "layout", "restoreWindowPlacementSize" ));
  defaultWindowSizeValue->setValue( config->getInt( "layout", "defaultWindowSize" ));
  defaultWindowPlacement->setCurrentItem( config->getInt( "layout", "defaultWindowPlacement" ) );
}
//==============================================
void LayoutSettingsWidget::saveSettings()
{
  //set setting values in config object so they are properly saved to disk
  config->setBool( "layout", "animation", useAnimation->isChecked() );
  config->setBool( "layout", "showTooltips", showTooltips->isChecked() );
  config->setBool( "layout", "restoreWindowPlacementSize", restoreWindowPlacementSize->isChecked());
  config->setInt( "layout", "defaultWindowSize", defaultWindowSizeValue->value() );
  config->setInt( "layout", "defaultWindowPlacement", defaultWindowPlacement->currentItem() );

  //apply setting changes to application behavior
  QToolTip::setGloballyEnabled( config->getBool( "layout", "showTooltips" ) );
  ((Window*)qApp->mainWidget())->getTitle()->useAnimation( config->getBool( "layout", "animation" ) );
}
//==============================================
