/**********************************************************************
 **
 ** Talent - contains methods and attributes for a talent object for use 
 **          on the builder port and during new player login
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/


#ifndef TALENT_C
#define TALENT_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "talent.h"
#include "lexer.h"
#include "objtype.h"
#include "builder.h"
#include "player.h"
#include "objtype.h"
#include "newfuncts.h"
#include "utils.h"
#include "global.h"
#include "specials.h"
#include "code.h"

/***********************************************************************
 ** Talent (constructor) - loads a talent name and attributes
 **
 ** Parameters: talent_name - the name to give to the talent
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Talent::Talent(char *talent_name)
{
   Strings tmp_name;

   obj_type = OBJ_TYPE_TALENT;
   tmp_name.assign_word(talent_name, 1);

   set_name(tmp_name.str_show());

   set_description("Describe the talent here.\n");
   set_special(NULL);

   cost = 0;
   allowed = 1;
}


/***********************************************************************
 ** ~Talent (destructor) - cleans up the Talent
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Talent::~Talent(void)
{
}


/***********************************************************************
 ** set_description - sets the description string for this talent
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Talent::set_description(char *the_string)
{
   if (the_string == NULL)
      return;

   description = the_string;
}

/***********************************************************************
 ** get_description - gets the description string for this talent
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Talent::get_description(void)
{
   return description.str_show();
}



/***********************************************************************
 ** set_special - sets the special string indicating which special should 
 **               be run to set special attributes on players
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Talent::set_special(char *the_string)
{
   if (the_string == NULL)
      return;

   the_special = the_string;
}

/***********************************************************************
 ** get_special - sets the special on the talent
 **
 ** Parameters: None
 **
 ** Returns: pointer to the special string
 **
 ***********************************************************************/

char *Talent::get_special(void)
{
   return the_special.str_show();
}


/***********************************************************************
 ** set_cost - sets how much talent points this will use up or give back
 **            if it is a deficiency
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Talent::set_cost(int the_num)
{
   cost = the_num;
}

/***********************************************************************
 ** get_cost - sets how much talent points this will use up or give back
 **            if it is a deficiency
 **
 ** Returns: the offset value
 **
 ***********************************************************************/

int Talent::get_cost(void)
{
   return cost;
}



/***********************************************************************
 ** set_allowed - sets the number the player can select this talent
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Talent::set_allowed(int the_num)
{
   if (the_num <= 0)
      allowed = 1;
   else
      allowed = the_num;
}

/***********************************************************************
 ** get_allowed - gets the number the player can select this talent
 **
 ** Returns: the value
 **
 ***********************************************************************/

int Talent::get_allowed(void)
{
   return allowed;
}


/***********************************************************************
 ** load_talent - loads a talent from a file into memory
 **
 ** Parameters: the_file - where we are getting the talent from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Talent::load_talent(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Strings      holder;
   int          neg = 0;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in talent file", "load_talent");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the description */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_description(tmp_charholder);
   delete tmp_charholder;

   /* Set special value */
   the_token = get_token(the_file, '\0');     
   set_special(the_token->the_string);

   /* Set cost value */
   the_token = get_token(the_file, '\0');   
   if (the_token->token_type == T_MINUS)
   {
      neg = 1;
      the_token = get_token(the_file, '\0');
   }
  
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute cost for talent %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_talent");
      return -1;
   }
   if (neg)
      set_cost(-(atoi(the_token->the_string)));
   else
      set_cost(atoi(the_token->the_string));
   neg = 0;

   /* Set allowed value */
   the_token = get_token(the_file, '\0');   
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute allowed for talent %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_talent");
      return -1;
   }
   set_allowed(atoi(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** describe - describes the talent to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Talent::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GTalent: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GSpecial: \t&+g%s&*\n", get_special());
   the_builder->send_bldr("&+GCost: \t\t&+w%d&*\n", get_cost());
   the_builder->send_bldr("&+GAllowed: \t&+w%d&*\n", get_allowed());
   the_builder->send_bldr("&+GDesc: \n&+w%s&*\n", get_description());
   the_builder->send_bldr("\n\n");
}


/***********************************************************************
 ** describe - describes the talent to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Talent::describe(Player *the_player)
{
   the_player->send_plr("\n&+GTalent: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GSpecial: \t&+g%s&*\n", get_special());
   the_player->send_plr("&+GCost: \t\t&+w%d&*\n", get_cost());
   the_player->send_plr("&+GAllowed: \t&+w%d&*\n", get_allowed());
   the_player->send_plr("&+GDesc: \n&+w%s&*\n", get_description());
   the_player->send_plr("\n\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Talent::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a talent.\n"
               "   desc, special, allowed, and cost\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "desc",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&description) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "special",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
             send_bldr("You need to specify an Special to set to.\n");
         return -1;
      }
 
      holder.assign_word(the_parsed->get_speech(), 1);

      set_special(holder.str_show());

      the_builder->send_bldr("Special on %s set to: %s\n", get_name(), 
                                                     get_special());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "cost",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (((!isdigit(*(the_parsed->get_speech()))) && 
          (*(the_parsed->get_speech()) != '-')) ||
            ((*(the_parsed->get_speech()) == '-') &&
             (!isdigit(*(the_parsed->get_speech() + 1)))))
      {
         the_builder->send_bldr("You need to specify a number as Cost.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      set_cost(value);
      the_builder->send_bldr("Cost set to %d on talent object %s.\n",
                                                   get_cost(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "allowed",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as Allowed.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_allowed(value);
      the_builder->send_bldr("Allowed set to %d on talent object %s.\n",
                                          get_allowed(), get_name());
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a talent attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the talent to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ***********************************************************************/
   
void Talent::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\ntalent %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_description() == NULL) ? "" : 
                                                        get_description());
   fprintf(the_file, "%s\n", (get_special() == NULL) ? "none" : get_special());
   fprintf(the_file, "%d\n", get_cost());
   fprintf(the_file, "%d\n", get_allowed());
}


/***********************************************************************
 ** is_modified - has this talent been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Talent::is_modified(void)
{
   return modified;
}


/***********************************************************************
 ** set_modified - shall we set this? Yes
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Talent::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** copy_object - copies the object to a talent of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Talent::copy_object(Entity *copy_obj)
{
   Talent *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_TALENT)
      return 0;

   copy_from = (Talent *) copy_obj;

   /******* set the action attributes *****/
   set_description(copy_from->get_description());
   set_special(copy_from->get_special());
   set_cost(copy_from->get_cost());
   set_allowed(copy_from->get_allowed());

   return 1;
}


/***********************************************************************
 ** assign_talent - runs the special that will assign this talent to
 **                 the player
 **
 ** Parameters: the_player - the player to set
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Talent::assign_talent(Player *the_player)
{
   in_params   fill_param;
   special_env the_environment;
   Specials    *special_obj;
   Strings     holder;

   /* this is where the special, if it exists, is executed */
   if ((the_special.str_show() != NULL) &&
       (STRCASECMP(the_special.str_show(), "none")))
   {
      if ((special_obj = mainstruct->get_special(the_special.str_show()))  == NULL)
      {
         holder.sprintf("Special %s on talent %s doesn't exist.", 
                          the_special.str_show(), get_name());
         mainstruct->log_error(holder.str_show(), "assign_talent");
         return -1;
      }

      fill_param.primary_obj = NULL;
      fill_param.secondary_obj = NULL;
      fill_param.this_obj = the_player;

      the_environment.trig_used = "ontalent";
      the_environment.target_str = get_name();
      the_environment.exec_vars = NULL;
      special_obj->run_special(the_player, &fill_param, &the_environment);
   }

   the_player->set_talent_funds(the_player->get_talent_funds() - get_cost());
   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Talent::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Talent::get_mem_size_dynamic()
{
   int  size = 0;
 
   size += description.get_mem_size_dynamic();   
   size += the_special.get_mem_size_dynamic();   
   return size;
}


#endif
