/**********************************************************************
 ** LinkedList class: object that maintains a linked list of mudobjects, 
 **                   usually used for inventories of mudobjects
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudobject.h"
#include "linkedlist.h"
#include "timespan.h"
#include "merger.h"
#include "player.h"
#include "newfuncts.h"
#include "global.h"


/* these all were embedded functions in the header, but since these type
   have been misbehaving, I took these out.  Documentation later. */
LinkedNode::LinkedNode(MudObject *the_data)
{
   data = the_data; 
   next = prev = NULL; 
}

void LinkedNode::set_data(MudObject *the_data) 
{ 
   data = the_data; 
}

MudObject *LinkedNode::get_data() 
{ 
   return data; 
}

void LinkedNode::set_next(LinkedNode *nextnode) 
{ 
   next = nextnode; 
}

void LinkedNode::set_prev(LinkedNode *prevnode) 
{ 
   prev = prevnode; 
}

LinkedNode *LinkedNode::get_next() 
{ 
   return next; 
}

LinkedNode *LinkedNode::get_prev() 
{ 
   return prev; 
}


/*********************************************************************
 * CONSTRUCTOR - initializes the linked list
 *********************************************************************/
LinkedList::LinkedList()
{
   root = NULL;
}

/*********************************************************************
 * DESTRUCTOR - Cleans up the linked list and clears all allocated
 *              memory.
 *********************************************************************/
LinkedList::~LinkedList()
{  LinkedNode *theNode;
   LinkedNode *tmpNode;

   theNode = tmpNode = root;

   while (theNode != NULL)
   {  tmpNode = theNode->get_next();
      delete_LinkedNode(theNode);
      theNode = tmpNode;
   }
}


/*********************************************************************
 * add_entry - Adds an entry to the list. It doesnt check if the 
 *             mudobject to be added is already in the list.
 *
 * Parameters:
 * INPUT - pointer to a mudobject that needs to be added to the list
 * OUTPUT- 1 if succesful, 0 if not.
 *********************************************************************/
int LinkedList::add_entry(MudObject *new_entry)
{   LinkedNode *newNode;

    newNode = new_LinkedNode(new_entry);
    if (newNode == NULL)
    {
       printf("bla!!\n");
       return 0;
    }

    if (root != NULL)
    {  newNode->set_next(root);
       newNode->set_prev(NULL);
       newNode->set_data(new_entry);
	 root->set_prev(newNode);
	 root = newNode;
       return 1;
    }
    else
    {  newNode->set_next(NULL);
       newNode->set_prev(NULL);
	 root = newNode;
       return 1;
    }
    printf("bla2!\n");
    return 0;
}

/*********************************************************************
 * del_entry - Deletes an entry from the linked list
 *
 * Parameters:
 * INPUT - The MudObject that needs to be deleted from the linked list
 *         the mudobject will remain in memory, the node it is in will
 *         be deleted.
 * OUTPUT- 1 if succesful, 0 if not found.
 *********************************************************************/
int LinkedList::del_entry(MudObject *entry)
{   LinkedNode *tmpNode = root;
    LinkedNode *nxtNode;
    LinkedNode *prvNode;


    while (tmpNode != NULL)
    {   if (tmpNode->get_data() == entry)
        {   if (tmpNode == root)
            {   nxtNode = root->get_next();
                if (nxtNode != NULL)
                {  nxtNode->set_prev(NULL);
                }
                root = nxtNode;
            }
	    else
            {  nxtNode = tmpNode->get_next();
               prvNode = tmpNode->get_prev();

               if (nxtNode != NULL)
			nxtNode->set_prev(prvNode);
		   if (prvNode != NULL)
			prvNode->set_next(nxtNode);
            }
            if (current == tmpNode)
	       current = nxtNode;
	    delete_LinkedNode(tmpNode);
	    return 1;
       }
       tmpNode = tmpNode->get_next();
    }
    return 0;
}

/*********************************************************************
 * del_cur_entry - Deletes the entry that was requested by get_next,
 *                 get_first or get_prev.
 * Parameters:
 * OUTPUT- 0 if there is no current entry or the entry could not be
 *         found, 1 if there is.
 *********************************************************************/
int  LinkedList::del_cur_entry()
{  LinkedNode *prvNode, *nxtNode;

   if (current != NULL)
   {   prvNode = current->get_prev();
       nxtNode = current->get_next();
       
       if (prvNode != NULL)
	    prvNode->set_next(nxtNode);
	 if (nxtNode != NULL)
	    nxtNode->set_prev(prvNode);
       delete_LinkedNode(current);
       current = nxtNode;
       return 1;
   }	
   return 0;
}

/*********************************************************************
 * get_next - get the next entry in the linked list
 * 
 * Parameters:
 * OUTPUT- returns the next MudObject in the linked list, NULL if the
 *         end is reached.
 *********************************************************************/
MudObject *LinkedList::get_next()
{

   MudObject *the_data;

   if (current == NULL)
   {
      return NULL;
   }

   the_data = current->get_data();
   current = current->get_next();
   return the_data;
}

/*********************************************************************
 * get_prev - returns the previous entry in the linked list.
 * 
 * Parameters:
 * OUTPUT- Pointer to the previous mudobject or NULL if the beginning
 *         of the list was reached.
 *********************************************************************/
MudObject *LinkedList::get_prev()
{

    if (current == NULL)
    {
       return NULL;
    }
    else
    {   current = current->get_prev();
        if (current != NULL)
	{
           return current->get_data();
	}
    }
    return NULL;
}

/*********************************************************************
 * get_first - gets the first entry in the linked list
 * 
 * Parameters:
 * OUTPUT: returns NULL if the list is empty or a pointer to the
 *         first mudobject.
 *********************************************************************/
MudObject *LinkedList::get_first()
{
   MudObject *the_data = NULL;

   current = root;
   if (current != NULL)
   {
      the_data = current->get_data();
      current = current->get_next();
      return the_data;
   }
   return NULL;
}

/*********************************************************************
 * find - finds an entry in the linked list by searching for its name
 *
 * Parameters:
 * INPUT : a name
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
MudObject *LinkedList::find(char *thename, int *the_num)
{  MudObject *tmpObject;
   Strings   the_keywords;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
      the_keywords = tmpObject->get_keywords();
      if (the_keywords.find_in_str(thename))
      {
         if (*the_num <= 1)
            return tmpObject;
         else
            *the_num = *the_num - 1;
      }
      tmpObject = get_next(); 
   }

   return NULL;
}


/*********************************************************************
 * find_by_objname - like find but uses the object's name instead of
 *                   the object's title
 *
 * Parameters:
 * INPUT : a name, the_area
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/

MudObject *LinkedList::find_by_objname(char *the_name, char *the_area)
{  MudObject *tmpObject;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
        if (STRCASECMP(tmpObject->get_name(),the_name) == 0)
        { 
            if (!STRCASECMP(tmpObject->get_area(),the_area))
               return tmpObject;
        }
        tmpObject = get_next(); 
   }

   return NULL;
}


/*********************************************************************
 * find_altname
 *
 * Parameters:
 * INPUT : a name
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
MudObject *LinkedList::find_altname(char *the_name)
{  MudObject *tmpObject;
   Strings   the_altnames;

   tmpObject = get_first();

   while (tmpObject != NULL)
   {
      the_altnames = tmpObject->get_altname();
      if (the_altnames.find_in_str(the_name) > 0)
      {
          return tmpObject;
      }
      tmpObject = get_next(); 
   }

   return NULL;
}



/*********************************************************************
 * get_same_merger - gets a merger with the same parent specified
 *
 * Parameters:
 * INPUT : the_parent - the parent string we are looking for
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
Merger *LinkedList::get_same_merger(char *the_parent)
{  MudObject *tmpObject;
   Merger    *the_merger;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
      if (tmpObject->is_merger())
      { 
         the_merger = (Merger *) tmpObject;
         if ((the_merger->get_parent() != NULL) && 
             (STRCASECMP(the_merger->get_parent(), the_parent) == 0))
            return the_merger;
      }
      tmpObject = get_next(); 
   }
   return NULL;
}

/***********************************************************************
 ** reset_current - resets the current linked list to the beginning
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void LinkedList::reset_current()
{
   current = root;
}


/***********************************************************************
 ** display_list - displays a linked list in inventory format
 **
 ** Parameters: the_player - the player we send the data to
 **
 ** Returns: num objects displayed for success, -1 for failure
 **
 ***********************************************************************/

int LinkedList::display_list(Player *the_player)
{
   MudObject  *the_obj;
   int        count = 0;
   int        line_len = 0;
   Strings    send_str;
   Strings    holder;
   reset_current();
   the_obj = get_next();

   while (the_obj != NULL)
   {  
      if (the_obj->is_an_item())
      {
         Flags *tmp_itemflags;

         tmp_itemflags = ((Item *) the_obj)->get_itemflags();

         if (the_obj->is_merger())
	    holder.sprintf("%s%s(%ld)%s", ((line_len == 0) ? "\n" : ", "), 
                            the_obj->get_title(), 
                            ((Merger *) the_obj)->get_number_of(), 
                 tmp_itemflags->get_flag(ITEMFLAG_LIT) ? " &+Y(lit)&*" : "");
         else 
	 {
            if (tmp_itemflags->get_flag(ITEMFLAG_WORN))
	    {
               the_obj = get_next();
               continue;
	    }

            holder.sprintf("%s%s%s", ((line_len == 0) ? "\n" : ", "), 
                                                    the_obj->get_title(),
               tmp_itemflags->get_flag(ITEMFLAG_LIT) ? " &+Y(lit)&*" : "");
	 }
      }
      else
         holder.sprintf("%s%s", ((line_len == 0) ? "\n" : ", "), 
                                                    the_obj->get_title());
      send_str.str_cat(holder.str_show());

      count++;
      the_obj = get_next();
      if ((line_len = send_str.str_len()) >= the_config.max_line_len)
      {
         the_player->send_plr(send_str.str_show());
         send_str.truncate(0);
         line_len = 0;
      }
   }
   if (send_str.str_show() != NULL)
      the_player->send_plr(send_str.str_show());
   the_player->send_plr("\n");

   return count;
}


/***********************************************************************
 ** store_current - passes out the current pointer so it can be restored
 **                 at a later time
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the current node
 ** 
 ***********************************************************************/

LinkedNode *LinkedList::store_current()
{
   return current;
}


/***********************************************************************
 ** restore_current - restores the current pointer to what we give it
 **
 ** Parameters: new_current - the new current value to restore to
 ** 
 ***********************************************************************/

void LinkedList::restore_current(LinkedNode *new_current)
{
   current = new_current;
}

/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int LinkedList::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int LinkedList::get_mem_size_dynamic()
{
   int  size = 0;
   LinkedNode *tmp_node;

   tmp_node = root;
   while (tmp_node != NULL)
   {
      size += sizeof(*tmp_node);
      tmp_node = tmp_node->get_next();
   }
   return size;
}









